package ClearCase::Wrapper::Site::CGC;

$VERSION = '0.01';

use AutoLoader 'AUTOLOAD';

use vars '@Admins';

use strict;

# A list of users who are exempt from restrictions implemented herein.
@Admins = qw(vobadm);

# Override the user's preference while interacting with clearcase.
umask 002 if defined(umask) && umask > 2 &&
			    $ENV{LOGNAME} && !grep(/^$ENV{LOGNAME}$/, @Admins);

#############################################################################
# Usage Message Extensions
#############################################################################
{
   local $^W = 0;
   no strict 'vars';

   # Usage message additions for actual cleartool commands that we extend.
   $mkview	= "\n* [-me] [-clone]";

   # Usage messages for pseudo cleartool commands that we implement here.
   local $0 = $ARGV[0] || '';
}

#############################################################################
# Command Aliases
#############################################################################

1;

__END__

=head1 NAME

ClearCase::Wrapper::Site::CGC - ClearCase::Wrapper overlay for CGC site

=head1 SYNOPSIS

This is an C<overlay module> for B<ClearCase::Wrapper> containing
site-specific extensions for Clear Guidance Consulting. See C<perldoc
ClearCase::Wrapper> for more details.

=head1 CLEARTOOL ENHANCEMENTS

=over 4

=item * MKVIEW

Extended in the following ways:

=over 4

=item 1. New B<-me> flag

Supports the B<-me> flag to prepend $LOGNAME to the view name,
separated by an underscore. This enables the convention that all user
views be named B<E<lt>usernameE<gt>_E<lt>whateverE<gt>>.

=item 2. New B<-clone> flag

This allows you to specify another view from which to copy the config
spec and other properties. Note that it does I<not> copy view-private
files from the old view, just the view properties.

=item 3. Enforced naming convention

Non-privileged users may only create views whose names are prefixed
with E<lt>usernameE<gt>_.

=item 4. Default storage location

If the user does not provide an explicit storage location (using
C<-stgloc> or C<-host/-hpath/-gpath>, etc.), standard storage location
I<view> is used.  Thus a user can simply type B<"mkview -tag foo -me">
and the view will be created as E<lt>usernameE<gt>_foo with the view
storage placed in a default location as determined by the CC admin.

=back

=cut

sub mkview {
    # Our private options (e.g. -clone) need to come out of the argv
    # right here, whereas for real CC flags we want to sample them but
    # leave them in the argv.
    my %opt;
    GetOptions(\%opt, qw(clone=s));
    {
	local(@ARGV) = @ARGV;	# operate on temp argv
	GetOptions(\%opt, qw(tag=s stgloc=s host=s snapshot
						shareable_dos nshareable_dos));
    }

    # This is an error; let CC handle dying with the appropriate msg.
    return 0 unless $opt{tag};

    # If the user specified a storage location or triple or an
    # explicit view-storage-pname, assume they know what they're
    # doing and get out of the way.
    push(@ARGV, qw(-stgloc view))
	    unless $opt{stgloc} || $opt{host} || $ARGV[-1] =~ m%^[/\\]%;

    # Policy: users' views should be prefixed by username.
    die Msg('E', "personal view names must match $ENV{LOGNAME}_*")
		    if @Admins && !grep(/^$ENV{LOGNAME}$/, @Admins) &&
						$opt{tag} !~ /^$ENV{LOGNAME}_/;

    # If an option was used requiring a special config spec, make the
    # view here, change the cspec, then exit. Must be done this way
    # because mkview provides no way to specify the initial config spec.
    # Also clone other properties such as cache size and text mode.
    if ($opt{clone}) {
	my $ct = ClearCase::Argv->new;
	chomp(my @data = $ct->lsview([qw(-prop -full)], $opt{clone})->qx);
	my %lsview = map {(split /:\s*/)[0,1]} @data;
	splice(@ARGV, 1, 0, '-tmode', $lsview{'Text mode'})
		   if $lsview{'Text mode'};
	my %properties = map {$_ => 1} split(/\s+/, $lsview{Properties});
	for (keys %properties) { splice(@ARGV, 1, 0, "-$_") if /shareable_do/ }
	my($cachebytes) = map {(split /\s+/)[0]} map {(split /:\s*/)[1]}
		    reverse $ct->getcache([qw(-view -s)], $opt{clone})->qx;
	splice(@ARGV, 1, 0, '-cachesize', $cachebytes);
	$ct->autofail(1);
	$ct->argv(@ARGV)->system;
	my $cstmp = ".mkview.$$.cs.$opt{tag}";
	my $ctx = ClearCase::Argv->cleartool;
	Argv->new("$ctx catcs -tag $opt{clone} > $cstmp")->system;
	$ct->setcs('-tag', $opt{tag}, $cstmp)->system;
	unlink($cstmp);
	exit 0;
    }

    # Generally speaking user's views should be -nshareable while
    # standard nightly-build views should be -shareable. Since by
    # convention the former are prefixed with username, we make
    # the shareability default contingent on that while always
    # allowing a literal flag to win.
    splice(@ARGV, 1, 0, ($opt{tag} =~ /^$ENV{LOGNAME}_/) ? '-nsh' : '-sha')
			unless $opt{shareable_dos} || $opt{nshareable_dos} ||
			       $opt{snapshot};

    # Done. Allow the amended mkview cmdline to be executed.
    return 0;
}

=back

=head1 SEE ALSO

perl(1), ClearCase::Wrapper, ClearCase::Wrapper::DSB

=cut
