package App::Manoc::Controller::APIv1::DHCPLease;
#ABSTRACT: Controller for DHCP APIs

use Moose;

our $VERSION = '2.99.2'; ##TRIAL VERSION

use namespace::autoclean;

BEGIN { extends 'App::Manoc::Controller::APIv1' }


sub lease_base : Chained('deserialize') PathPart('dhcp/lease') CaptureArgs(0) {
    my ( $self, $c ) = @_;
    $c->stash( resultset => $c->model('ManocDB::DHCPLease') );
}


sub lease_post : Chained('lease_base') PathPart('') POST {
    my ( $self, $c ) = @_;

    $c->stash(
        api_validate => {
            type  => 'hash',
            items => {
                server => {
                    type     => 'scalar',
                    required => 1,
                },
                leases => {
                    type     => 'array',
                    required => 1,
                },
            },
        }
    );
    $c->forward('validate') or return;

    my $req_data = $c->stash->{api_request_data};

    my $server_name = $req_data->{server};
    my $server      = $c->model('ManocDB::DHCPServer')->find('name');
    if ( !$server ) {
        push @{ $c->stash->{api_field_errors} }, 'Unknown server';
        return;
    }

    my $records = $req_data->{leases};

    my $n_created = 0;
    my $rs        = $c->stash->{resultset};
    foreach my $r (@$records) {
        my $macaddr = $r->{macaddr}                                    or next;
        my $ipaddr  = App::Manoc::IPAddress::IPv4->new( $r->{ipaddr} ) or next;
        my $start   = $r->{start}                                      or next;
        my $end     = $r->{end}                                        or next;

        my $hostname = $r->{hostname};
        my $status   = $r->{status};

        $rs->update_or_create(
            {
                server   => $server,
                macaddr  => $macaddr,
                ipaddr   => $ipaddr,
                hostname => $hostname,
                start    => $start,
                end      => $end,
                status   => $status,
            }
        );
        $n_created++;
    }
    my $data = { message => "created $n_created entries", };

    $c->stash( api_response_data => $data );
}

__PACKAGE__->meta->make_immutable;

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::APIv1::DHCPLease - Controller for DHCP APIs

=head1 VERSION

version 2.99.2

=head1 ACTIONS

=head2 lease_base

Base action lease actions

=head2 lease_post

POST api/v1/dhcp/lease

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
