NAME
====

Geo::Coordinates::UTM - Perl extension for Latitude Longitude conversions.

SYNOPSIS
========

use Geo::Coordinates::UTM;

my ($zone,$easting,$northing)=|latlon-to-utm($ellipsoid,$latitude,$longitude);

my ($latitude,$longitude)=|utm-to-latlon($ellipsoid,$zone,$easting,$northing);

my ($zone,$easting,$northing)=|mgrs-to-utm($mgrs);

my ($latitude,$longitude)=|mgrs-to-latlon($ellipsoid,$mgrs);

my ($mgrs)=|utm-to-mgrs($zone,$easting,$northing);

my ($mgrs)=|latlon-to-mgrs($ellipsoid,$latitude,$longitude);

my @ellipsoids=ellipsoid-names;

my($name, $r, $sqecc) = |ellipsoid-info 'WGS-84';

DESCRIPTION
===========

This module will translate latitude longitude coordinates to Universal Transverse Mercator(UTM) coordinates and vice versa.

Mercator Projection
-------------------

The Mercator projection was first invented to help mariners. They needed to be able to draw a straight line on a map and follow that bearing to arrive at a destination. In order to do this, Mercator invented a projection which preserved angle, by projecting the earth's surface onto a cylinder, sharing the same axis as the earth itself. This caused all Latitude and Longitude lines to be straight and to intersect at a 90 degree angle, but the downside was that the scale of the map increased as you moved away from the equator so that the lines of longitude were parallel.

Because the scale varies, areas near the poles appear much larger on the map than a similar sized object near the equator. The Mercator Projection is useless near the poles since the scale becomes infinite.

Transverse Mercator Projection
------------------------------

A Transverse Mercator projection takes the cylinder and turns it on its side. Now the cylinder's axis passes through the equator, and it can be rotated to line up with the area of interest. Many countries use Transverse Mercator for their grid systems. The disadvantage is that now neither the lines of latitude or longitude (apart from the central meridian) are straight.

Universal Transverse Mercator
-----------------------------

The Universal Transverse Mercator(UTM) system sets up a universal world wide system for mapping. The Transverse Mercator projection is used, with the cylinder in 60 positions. This creates 60 zones around the world. Positions are measured using Eastings and Northings, measured in meters, instead of Latitude and Longitude. Eastings start at 500,000 on the centre line of each zone. In the Northern Hemisphere, Northings are zero at the equator and increase northward. In the Southern Hemisphere, Northings start at 10 million at the equator, and decrease southward. You must know which hemisphere and zone you are in to interpret your location globally. Distortion of scale, distance and area increase away from the central meridian.

UTM projection is used to define horizontal positions world-wide by dividing the surface of the Earth into 6 degree zones, each mapped by the Transverse Mercator projection with a central meridian in the center of the zone. UTM zone numbers designate 6 degree longitudinal strips extending from 80 degrees South latitude to 84 degrees North latitude. UTM zone characters designate 8 degree zones extending north and south from the equator. Eastings are measured from the central meridian (with a 500 km false easting to insure positive coordinates). Northings are measured from the equator (with a 10,000 km false northing for positions south of the equator).

UTM is applied separately to the Northern and Southern Hemisphere, thus within a single UTM zone, a single X / Y pair of values will occur in both the Northern and Southern Hemisphere. To eliminate this confusion, and to speed location of points, a UTM zone is sometimes subdivided into 20 zones of Latitude. These grids can be further subdivided into 100,000 meter grid squares with double-letter designations. This subdivision by Latitude and further division into grid squares is generally referred to as the Military Grid Reference System (MGRS). The unit of measurement of UTM is always meters and the zones are numbered from 1 to 60 eastward, beginning at the 180th meridian. The scale distortion in a north-south direction parallel to the central meridian (CM) is constant However, the scale distortion increases either direction away from the CM. To equalize the distortion of the map across the UTM zone, a scale factor of 0.9996 is applied to all distance measurements within the zone. The distortion at the zone boundary, 3 degrees away from the CM is approximately 1%.

Datums and Ellipsoids
---------------------

Unlike local surveys, which treat the Earth as a plane, the precise determination of the latitude and longitude of points over a broad area must take into account the actual shape of the Earth. To achieve the precision necessary for accurate location, the Earth cannot be assumed to be a sphere. Rather, the Earth's shape more closely approximates an ellipsoid (oblate spheroid): flattened at the poles and bulging at the Equator. Thus the Earth's shape, when cut through its polar axis, approximates an ellipse. A "Datum" is a standard representation of shape and offset for coordinates, which includes an ellipsoid and an origin. You must consider the Datum when working with geospatial data, since data with two different Datum will not line up. The difference can be as much as a kilometer!

EXAMPLES
========

A description of the available ellipsoids and sample usage of the conversion routines follows

Ellipsoids
----------

The Ellipsoids available are as follows:

over
====

6

  * 1 Airy

  * 2 Australian National

  * 3 Bessel 1841

  * 4 Bessel 1841 (Nambia)

  * 5 Clarke 1866

  * 6 Clarke 1880

  * 7 Everest 1830 (India)

  * 8 Fischer 1960 (Mercury)

  * 9 Fischer 1968

  * 10 GRS 1967

  * 11 GRS 1980

  * 12 Helmert 1906

  * 13 Hough

  * 14 International

  * 15 Krassovsky

  * 16 Modified Airy

  * 17 Modified Everest

  * 18 Modified Fischer 1960

  * 19 South American 1969

  * 20 WGS 60

  * 21 WGS 66

  * 22 WGS-72

  * 23 WGS-84

  * 24 Everest 1830 (Malaysia)

  * 25 Everest 1956 (India)

  * 26 Everest 1964 (Malaysia and Singapore)

  * 27 Everest 1969 (Malaysia)

  * 28 Everest (Pakistan)

  * 29 Indonesian 1974

  * 30 Arc 1950

  * 31 NAD 27

  * 32 NAD 83

back
====



ellipsoid-names
---------------

The ellipsoids can be accessed using ellipsoid-names. To store thes into an array you could use 

    my @names = ellipsoid-names;

ellipsoid-info
--------------

Ellipsoids may be called either by name, or number. To return the ellipsoid information, ( "official" name, equator radius and square eccentricity) you can use ellipsoid-info and specify a name. The specified name can be numeric (for compatibility reasons) or a more-or-less exact name. Any text between parentheses will be ignored.

    my($name, $r, $sqecc) = |ellipsoid-info 'wgs84';
    my($name, $r, $sqecc) = |ellipsoid-info 'WGS 84';
    my($name, $r, $sqecc) = |ellipsoid-info 'WGS-84';
    my($name, $r, $sqecc) = |ellipsoid-info 'WGS-84 (new specs)';
    my($name, $r, $sqecc) = |ellipsoid-info 23;

latlon-to-utm
-------------

Latitude values in the southern hemisphere should be supplied as negative values (e.g. 30 deg South will be -30). Similarly Longitude values West of the meridian should also be supplied as negative values. Both latitude and longitude should not be entered as deg,min,sec but as their decimal equivalent, e.g. 30 deg 12 min 22.432 sec should be entered as 30.2062311

The ellipsoid value should correspond to one of the numbers above, e.g. to use WGS-84, the ellipsoid value should be 23

For latitude 57deg 49min 59.000sec North longitude 02deg 47min 20.226sec West

using Clarke 1866 (Ellipsoid 5)

    ($zone,$east,$north)=|latlon-to-utm('clarke 1866',57.803055556,-2.788951667)

returns 

    $zone  = 30V
    $east  = 512543.777159849
    $north = 6406592.20049111

On occasions, it is necessary to map a pair of (latitude, longitude) coordinates to a predefined zone. This is done by p[roviding a value for the optional named parameter zone as follows:

    ($zone, $east, $north)=|latlon-to-utm('international', :zone($zone_number),
                                         $latitude, $longitude)

For instance, Spain territory goes over zones 29, 30 and 31 but sometimes it is convenient to use the projection corresponding to zone 30 for all the country.

Santiago de Compostela is at 42deg 52min 57.06sec North, 8deg 32min 28.70sec West

    ($zone, $east, $norh)=|latlon-to-utm('international',  42.882517, -8.541306)

returns

    $zone = 29T
    $east = 537460.331
    $north = 4747955.991

but forcing the conversion to zone 30:

    ($zone, $east, $norh)=|latlon-to-utm('international', :zone(30),
                                         42.882517, -8.541306)

returns

    $zone = 30T
    $east = 47404.442
    $north = 4762771.704

utm-to-latlon
-------------

Reversing the above example,

    ($latitude,$longitude)=|utm-to-latlon(5,'30V',512543.777159849,6406592.20049111)

returns

    $latitude  = 57.8030555601332
    $longitude = -2.7889516669741

    which equates to

    latitude  57deg 49min 59.000sec North
    longitude 02deg 47min 20.226sec West

latlon-to-mgrs
--------------

Latitude values in the southern hemisphere should be supplied as negative values (e.g. 30 deg South will be -30). Similarly Longitude values West of the meridian should also be supplied as negative values. Both latitude and longitude should not be entered as deg,min,sec but as their decimal equivalent, e.g. 30 deg 12 min 22.432 sec should be entered as 30.2062311

The ellipsoid value should correspond to one of the numbers above, e.g. to use WGS-84, the ellipsoid value should be 23

For latitude 57deg 49min 59.000sec North longitude 02deg 47min 20.226sec West

using WGS84 (Ellipsoid 23)

    ($mgrs)=|latlon-to-mgrs(23,57.8030590197684,-2.788956799)

returns 

    $mgrs  = 30VWK1254306804

mgrs-to-latlon
--------------

Reversing the above example,

    ($latitude,$longitude)=|mgrs-to-latlon(23,'30VWK1254306804')

returns

    $latitude  = 57.8030590197684
    $longitude = -2.788956799645

mgrs-to-utm
-----------

    Similarly it is possible to convert MGRS directly to UTM

        ($zone,$easting,$northing)=|mgrs-to-utm('30VWK1254306804')

    returns

        $zone = 30V
        $easting = 512543
        $northing = 6406804

utm-to-mgrs
-----------

    and the inverse converting from UTM yo MGRS is done as follows

       ($mgrs)=|utm-to-mgrs('30V',512543,6406804);

    returns
        $mgrs = 30VWK1254306804

AUTHOR
======

Graham Crookham, grahamc@cpan.org

THANKS
======

Thanks go to the following:

Felipe Mendonca Pimenta for helping out with the Southern hemisphere testing.

Michael Slater for discovering the Escape \Q bug.

Mark Overmeer for the ellipsoid_info routines and code review.

Lok Yan for the >72deg. N bug.

Salvador Fandino for the forced zone UTM and additional tests

Matthias Lendholt for modifications to MGRS calculations

Peder Stray for the short MGRS patch

COPYRIGHT
=========

Copyright (c) 2000,2002,2004,2007,2010,2013 by Graham Crookham. All rights reserved.

copyright (c) 2017 by Kevin Pye.

This package is free software; you can redistribute it and/or modify it under the same terms as Perl itself. 

