# NAME

Proch::N50 - a small module to calculate N50 (total size, and total number of sequences) for a FASTA or FASTQ file. It's easy to install, with minimal dependencies.

# VERSION

version 1.5.0

# SYNOPSIS

    use Proch::N50 qw(getStats getN50);
    my $filepath = '/path/to/assembly.fasta';

    # Get N50 only: getN50(file) will return an integer
    print "N50 only:\t", getN50($filepath), "\n";

    # Full stats
    my $seq_stats = getStats($filepath);
    print Data::Dumper->Dump( [ $seq_stats ], [ qw(*FASTA_stats) ] );
    # Will print:
    # %FASTA_stats = (
    #               'N50' => 65,
    #               'N75' => 50,
    #               'N90' => 4,
    #               'min' => 4,
    #               'max' => 65,
    #               'dirname' => 'data',
    #               'auN' => 45.02112,
    #               'size' => 130,
    #               'seqs' => 6,
    #               'filename' => 'test.fa',
    #               'status' => 1
    #             );

    # Get also a JSON object
    my $seq_stats_with_JSON = getStats($filepath, 'JSON');
    print $seq_stats_with_JSON->{json}, "\n";
    # Will print:
    # {
    #    "status" : 1,
    #    "seqs" : 6,
    #    <...>
    #    "filename" : "small_test.fa",
    #    "N50" : 65,
    # }
    # Directly ask for the JSON object only:
    my $json = jsonStats($filepath);
    print $json;

# NAME

Proch::N50 - a small module to calculate N50 (total size, and total number of sequences) for a FASTA or FASTQ file. It's easy to install, with minimal dependencies.

# VERSION

version 1.4.2

# METHODS

## getN50(filepath)

This function returns the N50 for a FASTA/FASTQ file given, or 0 in case of error(s).

## getStats(filepath, alsoJSON)

Calculates N50 and basic stats for &lt;filepath>. Returns also JSON if invoked
with a second parameter.
This function return a hash reporting:

- _size_ (int)

    total number of bp in the files

- _N50_, _N75_, _N90_ (int)

    the actual N50, N75, and N90 metrices

- _auN_ (float)

    the area under the Nx curve, as described in [https://lh3.github.io/2020/04/08/a-new-metric-on-assembly-contiguity](https://lh3.github.io/2020/04/08/a-new-metric-on-assembly-contiguity).
    Returs with 5 decimal digits.

- _min_ (int)

    Minimum length observed in FASTA/Q file

- _max_ (int)

    Maximum length observed in FASTA/Q file

- _seqs_ (int)

    total number of sequences in the files

- _filename_ (string)

    file basename of the input file

- _dirname_ (string)

    name of the directory containing the input file (as received)

- _path_ (string)

    name of the directory containing the input file (resolved to its absolute path)

- _json_ (string: JSON pretty printed)

    (pretty printed) JSON string of the object (only if JSON is installed)

## jsonStats(filepath)

Returns the JSON string with basic stats (same as $result->{json} from _getStats_(File, JSON)).
Requires JSON::PP installed.

## \_n50fromHash(hash, totalsize)

This is an internal helper subroutine that perform the actual N50 calculation, hence its addition
to the documentation.
Expects the reference to an hash of sizes `$size{SIZE} = COUNT` and the total sum of sizes obtained
parsing the sequences file.
Returns N50, min and max lengths.

# Dependencies

## Module (N50.pm)

- [FASTX::Reader](https://metacpan.org/pod/FASTX%3A%3AReader) (required)
- [JSON::PP](https://metacpan.org/pod/JSON%3A%3APP), <File::Basename> (core modules)

## Stantalone program (n50.pl)

- [Term::ANSIColor](https://metacpan.org/pod/Term%3A%3AANSIColor)

- [JSON](https://metacpan.org/pod/JSON)

    (optional) when using `--format JSON`

- [Text::ASCIITable](https://metacpan.org/pod/Text%3A%3AASCIITable)

    (optional) when using `--format screen`. This might be substituted by a different module in the future.

# SUPPORT

This module and the n50 program have limited support.
SeqFu ([https://telatin.github.io/seqfu2](https://telatin.github.io/seqfu2)) is a compiled suite of 
utilities that includes a **seqfu stats** module, a faster replacement
for the `n50` program.

If you are interested in contributing to the development of this module, or
in reporting bugs, please refer to repository
[https://github.com/telatin/proch-n50/issues](https://github.com/telatin/proch-n50/issues).

# AUTHOR

Andrea Telatin <andrea@telatin.com>

# COPYRIGHT AND LICENSE

This software is Copyright (c) 2018-2022 by Andrea Telatin.

This is free software, licensed under:

    The MIT (X11) License
