# this code is written in Unicode/UTF-8 char-set
# including Japanese characters.

use strict;
use warnings;
use utf8;

use Test::More tests => 13;

BEGIN {
	use_ok(
		'Lingua::JA::Fold',
		qw(fold length_full length_half tab2space kana_half2full)
	)
};

########################################################################
# 2. replace a [TAB] with 4 of [SPACE]s.

my $text = 'かんじ	漢字';
my $got = tab2space('text' => $text, 'space' => 4);
my $expected = 'かんじ    漢字';

is ($got, $expected,
	'tab2space()');

########################################################################
# 3. convert a half-width 'Kana' letter to a full-width one.

$text = 'ｱｲｳｴｵ';
$got = kana_half2full($text);
$expected = 'アイウエオ';

is ($got, $expected,
	'kana_half2full()');

########################################################################
# 4. count length for full-width

$text = 'abcdeかんじ漢字ｱｲｳｴｵ｡';
$got = length_full($text);

is ($got, 10.5,
	'length_full(): for ascii + kanji + half_width_kana');

########################################################################
# 5. count length for half-width

$got = length_half($text);

is ($got, 21,
	'length_half(): for ascii + kanji + half_width_kana');

########################################################################
# 6. fold() (line-split: single line; end with newline)

$text = <<'EOF';
あいうえおかきくけこ
EOF

$expected = <<'EOF';
あいうえお
かきくけこ
EOF

$got = fold(
	'text' => $text,
	'length' => 5,
	'mode' => 'traditional',
);

is ($got, $expected,
	'fold(): line-split: single line; end with newline');

########################################################################
# 7. fold() (line-split: single line; end without newline)

chomp($text = <<'EOF');
あいうえおかきくけこ
EOF

chomp($expected = <<'EOF');
あいうえお
かきくけこ
EOF

$got = fold(
	'text' => $text,
	'length' => 5,
	'mode' => 'traditional',
);

is ($got, $expected,
	'fold(): line-split: single line; end without newline');

########################################################################
# 8. fold() (line-split: multi line; end with newline)

$text = <<'EOF';
あいうえおかきくけこ
さしすせそたちつてと
なにぬねのはひふへほ
EOF

$expected = <<'EOF';
あいうえお
かきくけこ
さしすせそ
たちつてと
なにぬねの
はひふへほ
EOF

$got = fold(
	'text' => $text,
	'length' => 5,
	'mode' => 'traditional',
);

is ($got, $expected,
	'fold(): line-split: multi line; end with newline');


########################################################################
# 9. fold() (line-split: multi line; end without newline)

chomp($text = <<'EOF');
あいうえおかきくけこ
さしすせそたちつてと
なにぬねのはひふへほ
EOF

chomp($expected = <<'EOF');
あいうえお
かきくけこ
さしすせそ
たちつてと
なにぬねの
はひふへほ
EOF

$got = fold(
	'text' => $text,
	'length' => 5,
	'mode' => 'traditional',
);

is ($got, $expected,
	'fold(): line-split: multi line; end without newline');

########################################################################
# 10. fold a text under 2 full width letters par a line. (full-width)

$text = 'abcdeかんじ漢字ｱｲｳｴｵ｡';

$got = fold(
	'text' => $text,
	'length' => 2,
	'mode' => 'full-width',
);

$expected = 'abcd
eか
んじ
漢字
ｱｲｳｴ
ｵ｡';

is ($got, $expected,
	'fold(): full-width; for a short text');

########################################################################
# 11. long text trial (full-width)

$text = <<'EOF';

apougaobuaEmailアドレスｱｲｳｴｵ｡も必須です（こちらから返事をする際に必要となりますので、アドレスの記入ミスをなさらぬようご注意下さい）。改行は、原則として段落を変えたい時のみ使用するgaoubaようにしてください。
aaaa手a動で行を折り返して長さを揃える必要はありません。 作成中に誤って消してしまった場合はショックが大きいものです。特ｶｷｸｹｺに長文の場合などは、一旦、テキストエディタやワープロ等で原稿を作成してから、それをメールの書き込み欄にコピー＆ペーストして送信するやり方にすれば安aaa全です。

※短い行※


 Emailアドレスも必須です（こちらから返事をoubabaする際に必要となりますので、アドレスの記入ミスをなさらぬようご注意下さい）。改行は、原則として段落を変えたい時のみ使用するようにしてください。aaaa手a動で行を折り返して長さを揃える必要はありません。作成中に誤って消してしまった場合はショックが大きいもagaのです。
特にagabb長文の場合などは、一旦、テキストエディタやワープロ等で原稿を作成してから、それをメールの書き込み欄にコピー＆ペーストして送信するやり方にすれば安全です。Emailアドレスも必須です（こちらかｱｲｳｴｵ｡ら返事をする際に必要となりますので、アドレスの記入ミスをなさらぬようご注意下さい）。 改行は、原則として段落を変えたい時のみ使用するようにしてください。aaaa手a動で行を折り返して長さを揃える必要はありません。 
作成中に誤って消してしまった場合はショックが大きいものです。特に長文の場合などは、一旦、テキストエディタやワープロ等で原稿を作成してから、それをメールの書き込み欄にコピー＆ペーストして送信するやり方にすれば安全です。


EOF

$expected = <<'EOF';

apougaobuaEmailアドレスｱｲｳｴｵ｡も必須です
（こちらから返事をする際に必要となります
ので、アドレスの記入ミスをなさらぬようご
注意下さい）。改行は、原則として段落を変
えたい時のみ使用するgaoubaようにしてくだ
さい。
aaaa手a動で行を折り返して長さを揃える必
要はありません。 作成中に誤って消してし
まった場合はショックが大きいものです。特
ｶｷｸｹｺに長文の場合などは、一旦、テキスト
エディタやワープロ等で原稿を作成してから
、それをメールの書き込み欄にコピー＆ペー
ストして送信するやり方にすれば安aaa全で
す。

※短い行※


 Emailアドレスも必須です（こちらから返事
をoubabaする際に必要となりますので、アド
レスの記入ミスをなさらぬようご注意下さい
）。改行は、原則として段落を変えたい時の
み使用するようにしてください。aaaa手a動
で行を折り返して長さを揃える必要はありま
せん。作成中に誤って消してしまった場合は
ショックが大きいもagaのです。
特にagabb長文の場合などは、一旦、テキス
トエディタやワープロ等で原稿を作成してか
ら、それをメールの書き込み欄にコピー＆ペ
ーストして送信するやり方にすれば安全です
。Emailアドレスも必須です（こちらかｱｲｳｴｵ
｡ら返事をする際に必要となりますので、ア
ドレスの記入ミスをなさらぬようご注意下さ
い）。 改行は、原則として段落を変えたい
時のみ使用するようにしてください。aaaa手
a動で行を折り返して長さを揃える必要はあ
りません。 
作成中に誤って消してしまった場合はショッ
クが大きいものです。特に長文の場合などは
、一旦、テキストエディタやワープロ等で原
稿を作成してから、それをメールの書き込み
欄にコピー＆ペーストして送信するやり方に
すれば安全です。


EOF

$got = fold(
	'text' => $text,
	'length' => 20,
	'mode' => 'full-width',
);

is ($got, $expected,
	'fold(): full-width; for a long  text');

########################################################################
# 12. long text trial (ignore difference whether full or half)

$expected = <<'EOF';

apougaobuaEmailアドレスｱ
ｲｳｴｵ｡も必須です（こちらから返事をす
る際に必要となりますので、アドレスの記入
ミスをなさらぬようご注意下さい）。改行は
、原則として段落を変えたい時のみ使用する
gaoubaようにしてください。
aaaa手a動で行を折り返して長さを揃え
る必要はありません。 作成中に誤って消し
てしまった場合はショックが大きいものです
。特ｶｷｸｹｺに長文の場合などは、一旦、
テキストエディタやワープロ等で原稿を作成
してから、それをメールの書き込み欄にコピ
ー＆ペーストして送信するやり方にすれば安
aaa全です。

※短い行※


 Emailアドレスも必須です（こちらか
ら返事をoubabaする際に必要となりま
すので、アドレスの記入ミスをなさらぬよう
ご注意下さい）。改行は、原則として段落を
変えたい時のみ使用するようにしてください
。aaaa手a動で行を折り返して長さを揃
える必要はありません。作成中に誤って消し
てしまった場合はショックが大きいもaga
のです。
特にagabb長文の場合などは、一旦、テ
キストエディタやワープロ等で原稿を作成し
てから、それをメールの書き込み欄にコピー
＆ペーストして送信するやり方にすれば安全
です。Emailアドレスも必須です（こち
らかｱｲｳｴｵ｡ら返事をする際に必要とな
りますので、アドレスの記入ミスをなさらぬ
ようご注意下さい）。 改行は、原則として
段落を変えたい時のみ使用するようにしてく
ださい。aaaa手a動で行を折り返して長
さを揃える必要はありません。 
作成中に誤って消してしまった場合はショッ
クが大きいものです。特に長文の場合などは
、一旦、テキストエディタやワープロ等で原
稿を作成してから、それをメールの書き込み
欄にコピー＆ペーストして送信するやり方に
すれば安全です。


EOF

$got = fold(
	'text' => $text,
	'length' => 20,
);

is ($got, $expected,
	'fold(): ignore difference whether full or half');

########################################################################
# 13. fold() (full-width; traditional manner)

$text = <<'EOF';
(４)符号　記号ともいう。句読点をはじめ、表記上の符号には、すべて一マスをあてる。感嘆符！疑問符？カギ「」カッコ（）ヤマ型カッコ〈〉など。注意すべきことは、文の中止、あるいは終止が行末にきたときは、次の行頭に打たず、行末の文字と一緒に書きこむということである。閉じる符号――）」』など――も同様に扱う。つまり、テン「、」まる「。」や閉じる符号は、行頭には据えないということである。その他、行頭に据えないものとして、くりかえし記号（々ゝ）などがある。「！？」は、行頭に書いて差しつかえない。ただ、「！？」の下は、すぐ下にカギの受けがくるとき以外は、一マスあけることになっている。句読点の下を一マスあけた原稿をよくみかけるが、これはあけない。

１２３４５６７
１２３４５６７

１２３４５６７、
１２３４５６７，
１２３４５６７。
１２３４５６７．
１２３４５６７」
１２３４５６７’
１２３４５６７』
１２３４５６７”
１２３４５６７〟
１２３４５６７）
１２３４５６７】
１２３４５６７〉
１２３４５６７》
１２３４５６７］
１２３４５６７〕
１２３４５６７｝
１２３４５６７々
１２３４５６７ゝ
１２３４５６７ゞ
１２３４５６７ヽ
１２３４５６７ヾ
１２３４５６７〃

１２３４５６７、，。．」’』”〟）】〉》］〕｝々ゝゞヽヾ〃


１２aiueo３４、
１２ｱｲｳｴｵ３４，



EOF

$expected = <<'EOF';
(４)符号　記号
ともいう。句読
点をはじめ、表
記上の符号には、
すべて一マスを
あてる。感嘆符
！疑問符？カギ
「」カッコ（）
ヤマ型カッコ〈〉
など。注意すべ
きことは、文の
中止、あるいは
終止が行末にき
たときは、次の
行頭に打たず、
行末の文字と一
緒に書きこむと
いうことである。
閉じる符号――）」』
など――も同様
に扱う。つまり、
テン「、」まる
「。」や閉じる
符号は、行頭に
は据えないとい
うことである。
その他、行頭に
据えないものと
して、くりかえ
し記号（々ゝ）
などがある。「
！？」は、行頭
に書いて差しつ
かえない。ただ、
「！？」の下は、
すぐ下にカギの
受けがくるとき
以外は、一マス
あけることにな
っている。句読
点の下を一マス
あけた原稿をよ
くみかけるが、
これはあけない。

１２３４５６７
１２３４５６７

１２３４５６７、
１２３４５６７，
１２３４５６７。
１２３４５６７．
１２３４５６７」
１２３４５６７’
１２３４５６７』
１２３４５６７”
１２３４５６７〟
１２３４５６７）
１２３４５６７】
１２３４５６７〉
１２３４５６７》
１２３４５６７］
１２３４５６７〕
１２３４５６７｝
１２３４５６７々
１２３４５６７ゝ
１２３４５６７ゞ
１２３４５６７ヽ
１２３４５６７ヾ
１２３４５６７〃

１２３４５６７、，。．」’』”〟）】〉》］〕｝々ゝゞヽヾ〃


１２aiueo３４、
１２ｱｲｳｴｵ３４，



EOF

$got = fold(
	'text' => $text,
	'length' => 7,
	'mode' => 'traditional',
);

is ($got, $expected,
	'fold(): full-width; with traditional manner');
