package OpenInteract::Handler::LookupEdit;

# $Id: LookupEdit.pm,v 1.10 2002/01/02 02:43:56 lachoy Exp $

# See 'doc/lookup.pod' for description of the fields in the action
# table we use.

use strict;
use OpenInteract::Handler::GenericDispatcher;
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::LookupEdit::ISA     = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::LookupEdit::VERSION = sprintf("%d.%02d", q$Revision: 1.10 $ =~ /(\d+)\.(\d+)/);

use constant REQUIRED_SECURITY => SEC_LEVEL_WRITE;
use constant DEFAULT_METHOD    => 'list_lookups';
use constant DEFAULT_DISPLAY   => 'column';

my $BLANK_COUNT = 5;
my $NEW_KEY     = '_new_';
my $REMOVE_KEY  = '_remove_';


# Our handler acts as the normal redirector, but instead of passing in
# the security to the individual task we just assume that any access
# == all access

sub handler {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $task = lc shift @{ $R->{path}{current} } || DEFAULT_METHOD;

    # Just use 'security_object' here because we know it will hit the
    # right database

    my $level = $R->security_object->check_security({ class     => $class,
                                                      object_id => '0' });
    if ( $level < REQUIRED_SECURITY ) {
        $R->throw({ code  => 305,
                    type  => 'security',
                    extra => { user_level     => $level,
                               required_level => REQUIRED_SECURITY,
                               class          => $class,
                               task           => $task } });
    }
    return $class->$task();
}


# Just find all the lookup actions

sub list_lookups {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->{page}{title} = 'List Available Lookups';
    return $R->template->handler({},
                                 { lookup_list => $class->_find_all_lookups,
                                   status_msg  => $p->{status_msg},
                                   error_msg   => $p->{error_msg} },
                                 { name => 'lookup::lookup_classes' });
}


# If data partitioning is specified, the view when accessing the
# lookup table is of a dropdown of the available values by which to
# partition the data

sub partition_listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $lookup_info = $p->{lookup_info};
    unless ( $lookup_info ) {
        my ( $error_msg );
        ( $lookup_info, $error_msg ) = $class->_find_lookup_info;
        unless ( $lookup_info ) {
            return $class->list_lookups({ error_msg => $error_msg });
        }
    }
    unless ( $lookup_info->{partition_field} ) {
        my $error_part_msg = "Cannot use ($lookup_info->{lookup_type}) as partitioned -- " .
                             "no value for 'partition_field' specified in action config.";
        return $class->list_lookups({ error_msg => $error_part_msg });
    }
    my $partition_values = eval { $class->_find_distinct_values(
                                               $lookup_info->{object_key},
                                               $lookup_info->{partition_field} ) };
    if ( $@ ) {
        $p->{error_msg} = "Could not retrieve values for " .
                          "($lookup_info->{partition_field}): " .
                          $SPOPS::Error::system_msg;
    }
    $R->{page}{title} = 'Lookup Partitioning Values';
    return $R->template->handler({}, { value_list  => $partition_values,
                                       lookup_type => $lookup_info->{lookup_type},
                                       error_msg   => $p->{error_msg} },
                                 { name => 'lookup::lookup_partitions' });
}


# List relevant entries in a particular lookup table

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my ( $lookup_info, $error_msg ) = $class->_find_lookup_info;
    unless ( $lookup_info ) {
        return $class->list_lookups({ error_msg => $error_msg });
    }

    my @lookup_keys = qw( field_list label_list size_list title
                          lookup_type partition_field );
    my %params = map { $_ => $lookup_info->{ $_ } } @lookup_keys;

    $params{blank_count} = $BLANK_COUNT;
    $params{remove_key}  = $REMOVE_KEY;
    $params{new_key}     = $NEW_KEY;

    if ( $params{partition_field} ) {
        $params{partition_value} = $R->apache->param( 'partition_value' );
        unless ( $params{partition_value} ) {
            return $class->partition_listing({ lookup_info => $lookup_info });
        }
        $params{label_list} ||= [];
        my %lbl = map { $params{field_list}->[ $_ ] => $params{label_list}->[ $_ ] }
                      ( 0 .. ( scalar @{ $params{field_list} } - 1 ) );
        $params{partition_label} = $lbl{ $params{partition_field} } ||
                                   $params{partition_field};
    }

    $params{lookup_list} = eval {  $class->_lookup_entries( $lookup_info,
                                                            $params{partition_value} ) };
    if ( $@ ) {
        $params{error_msg} = "Error retrieved when trying to lookup entries: $@";
    }
    else {

        # Check to see if the lookup action has defined a set of related
        # objects -- that is, the user when editing the lookup values
        # should choose one from many values.

        $lookup_info->{relate} ||= {};
        foreach my $field_name ( keys %{ $lookup_info->{relate} } ) {
            my $relate_info = $lookup_info->{relate}{ $field_name };
            next if ( $params{related}{ $field_name } );
            $params{related}{ $field_name } = $relate_info;
            $params{related}{ $field_name }{list} =
                                        eval { $class->_lookup_related_objects(
                                                                  $relate_info->{object},
                                                                  $relate_info ) };
            if ( $@ ) {
                $params{error_msg} .= "<br>Cannot lookup related $field_name: $@";
            }
        }
    }

    my $display_type = $R->apache->param( 'display_type' ) || DEFAULT_DISPLAY;
    my $tmpl_name    = ( $display_type eq 'column' )
                         ? 'lookup_listing_columns' : 'lookup_listing';
    $R->{page}{title} = 'Edit Lookup Entries';
    return $R->template->handler({}, \%params,
                                 { name => "lookup::$tmpl_name" });
}


sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my $lookup_type  = $apr->param( 'lookup_type' );
    my ( $lookup_info, $error_msg ) = $class->_find_lookup_info( $lookup_type );
    unless ( $lookup_info ) {
        return $class->list_lookups({ error_msg => $error_msg });
    }
    my $object_key   = $lookup_info->{object_key};
    my $lookup_class = $R->$object_key();
    my @id_list      = $class->_retrieve_id_list( $apr, $lookup_info->{field_list}->[0] );
    my @new_id_list  = map { "$NEW_KEY$_" } ( 1 .. $BLANK_COUNT );
    my @save_params  = ( $apr, $lookup_class, $lookup_info->{field_list} );
    my @error = ();
    foreach my $id ( @id_list, @new_id_list ) {
        $R->DEBUG && $R->scrib( 1, "Trying to find values for ID ($id)" );
        eval { $class->_persist( @save_params, $id ) };
        if ( $@ ) {
            $R->scrib( 0, "Cannot save ID ($id): $SPOPS::Error::system_msg" );
            push @error, "Error saving ($id): $SPOPS::Error::system_msg";
        }
    }
    return $class->list_lookups({
              error_msg  => join( "<br>\n", @error ),
              status_msg => "Entries for $lookup_type entered successfully" });
}


# $field is just a sample field used to get IDs

sub _retrieve_id_list {
    my ( $class, $apr, $field ) = @_;
    my @fields = grep ! /^$field\-$NEW_KEY/, grep /^$field/, $apr->param;
    my ( @id_list );
    foreach my $this_field ( @fields ) {
        $this_field =~ /^$field\-(.*)$/;
        push @id_list, $1;
    }
    return @id_list;
}


sub _find_all_lookups {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my $CONFIG = $R->CONFIG;
    my ( @lookup_list );
    foreach my $key ( keys %{ $CONFIG->{action} } ) {
        next unless ( $key );
        my ( $lookup_info, $error_msg ) = $class->_find_lookup_info( $key );
        if ( $lookup_info ) {
            $R->DEBUG && $R->scrib( 1, "Found lookup item ($key)" );
            push @lookup_list, $lookup_info;
        }
    }
    return \@lookup_list;
}


sub _find_distinct_values {
    my ( $class, $object_type, $field ) = @_;
    my $R = OpenInteract::Request->instance;
    my $object_class = $R->$object_type();
    return $object_class->db_select({ select_modifier => 'DISTINCT',
                                      select          => [ $field ],
                                      from            => [ $object_class->table_name ],
                                      order           => $field,
                                      return          => 'single-list' });
}


sub _find_lookup_info {
    my ( $class, $lookup_type ) = @_;
    my $R = OpenInteract::Request->instance;
    $lookup_type ||= $R->apache->param( 'lookup_type' );
    unless ( $lookup_type ) {
        return ( undef,  'Cannot list lookup entries without a lookup type.' );
    }
    my $lookup_info = $R->CONFIG->{action}{ $lookup_type };
    unless ( ref $lookup_info and $lookup_info->{is_lookup} ) {
        return ( undef, "Cannot list lookup entries: ($lookup_type) is not a valid lookup." );
    }
    $lookup_info->{lookup_type} = $lookup_type;
    return ( $lookup_info, undef );

}


sub _lookup_entries {
    my ( $class, $lookup_info, $partition_value ) = @_;
    my $R = OpenInteract::Request->instance;
    my $lookup_object_key = $lookup_info->{object_key};
    my $lookup_class = $R->$lookup_object_key();
    $R->DEBUG && $R->scrib( 1, "Trying to find all entries in ($lookup_class);",
                               "in the order ($lookup_info->{order})" );
    my %args = ( order => $lookup_info->{order} );
    if ( $partition_value ) {
        $R->DEBUG && $R->scrib( 1, "Filtering entries by ",
                                   "($lookup_info->{partition_field}) = ($partition_value)" );
        $args{where} = "$lookup_info->{partition_field} = ?";
        $args{value} = [ $partition_value ];
    }
    my $listing = eval { $lookup_class->fetch_group( \%args ) };
    if ( $@ ) {
        die "$SPOPS::Error::system_msg\n";
    }
    return $listing;
}


sub _lookup_related_objects {
    my ( $class, $object_type, $params ) = @_;
    my $R = OpenInteract::Request->instance;
    my $listing = eval { $R->$object_type()->fetch_group({ order => $params->{order} }) };
    if ( $@ ) { die $SPOPS::Error::system_msg }
    return $listing;
}


sub _persist {
    my ( $class, $apr, $lookup_class, $field_list, $id ) = @_;
    my $R = OpenInteract::Request->instance;
    my $is_new = ( $id =~ /^$NEW_KEY/ );
    my $object =  ( $is_new )
                    ? $lookup_class->new : $lookup_class->fetch( $id );
    my $not_blank = 0;
    my $do_remove = $apr->param( "$REMOVE_KEY-$id" );
    if ( $do_remove ) {
        return if ( $is_new );
        $R->DEBUG && $R->scrib( 1, "Trying to remove entry for ID ($id)" );
        return $class->_remove( $object );
    }
    foreach my $field ( @{ $field_list } ) {
        my $value = $apr->param( "$field-$id" );
        $R->DEBUG && $R->scrib( 1, "Found value: ($id) ($field): ($value)" );
        $object->{ $field } = $value;
        $not_blank++ if ( $value );
    }
    $object->save if ( $not_blank );
    return $object;
}


# We might want to add more stuff here...

sub _remove {
    my ( $class, $object ) = @_;
    return unless ( ref $object and $object->isa( 'SPOPS' ) );
    return $object->remove;
}


1;

__END__

=pod

=head1 NAME

OpenInteract::LookupEdit - Edit many simple objects at once

=head1 SYNOPSIS

 # See lookup/doc/lookup.pod for setup and usage

=head1 DESCRIPTION

This module implements generic lookup table editing. What this is and
does is fully described in the package documentation, so check that out.

Here we will only discuss the implementation.

=head1 METHODS

B<handler( \%params )>

This method is called with every request of this action. We parse the
URL and dispatch to the proper task, first checking to ensure that the
security is correct. (Default security of C<SEC_LEVEL_WRITE> is
required.)

B<list_lookups( \%params )>

Cycle through all the actions available and display the ones that are
lookup actions. (Lookup actions have the property 'lookup' set the
'yes'.

B<partition_listing( \%params )>

If the user has specified a lookup to be partitioned, then display a
dropdown with the DISTINCT values of the C<partition_field> specified
in the lookup action information.

This is normally called from the C<listing> task (see below) when we
encounter a request to list a partitioned lookup without a partition
field value.

B<listing( \%params )>

List lookup objects in a form for editing. We also create blank
entries so you can enter new lookup objects.

B<edit( \%params )>

Parse the form generated by the C<listing> task and for each create a
new object, edit an exiting object, or remove an existing object
depending on the user choice.

=head2 Internal Methods

B<_retrieve_id_list( $apache, $field_name )>

Parses the submitted form to find all IDs submitted by a form. The
exact C<$field_name> given does not matter, just as long as it is one
of the fields being edited.

B<_find_all_lookups()>

Iterate through the available actions and find the lookup
items.

Returns a list of hashrefs, each hashref describing a lookup actions.

B<find_distinct_values( $object_type, $distinct_field )>

Find all unique instances of C<$distinct_field> in the table specified
by C<$object_type>.

Returns an arrayref of distinct values for C<$distinct_field>, sorted.

B<_find_lookup_info( $action )>

Retrieves information from the action table for C<$action>. If
C<$action> is not a lookup, returns undef for the action information.

Returns: two element list. The first item a hashref of action
information, the second is a message.

B<_lookup_entries( \%action_info [, $partition_value ] )>

Run the query to retrieve the relevant information corresponding to
C<\%action_info>. If C<$partition_value> is specified we filter the
results by finding only records where the C<partition_field> key in
C<\%action_info> is C<$partition_value>.

B<_lookup_related_objects( $object_type, \%params )>

Just run a query to retrieve all objects of type C<$object_type>. If
the C<order> key in C<\%params> is specified we use it for sorting.

B<_persist( $apache, $lookup_class, \@field_name, $id )>

If the user has requested to remove the object, remove it.

Returns: the results of an SPOPS C<remove()> call on the object.

Otherwise, create the object with the new values and save it.

Returns: The resulting C<$object>.

B<_remove( $object )>

Remove the object C<$object>. (Not much right now, but a hook for
later.)

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
