/*                                                                
**  Copyright (C) 1996,2007,2010,2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* paramset.c
**/

#ifdef NONCIAO

#include "config.h"

#else

/* CIAO has readline */

#define HAVE_LIBREADLINE 1
#define HAVE_READLINE_READLINE_H
#define HAVE_READLINE_HISTORY 1
#define HAVE_READLINE_HISTORY_H 1

#endif


#include <stdlib.h>
#include <string.h>

#include "pfile.h"
#include "ptemplat.h"
#include "parameter.h"
#include "range.h"
#include "find.h"
#include "listhook.h"

/* The following code is from the AX_LIB_READLINE AC Macro docs */
#ifdef HAVE_LIBREADLINE
#  if defined(HAVE_READLINE_READLINE_H)
#    include <readline/readline.h>
#  elif defined(HAVE_READLINE_H)
#    include <readline.h>
#  elif defined(HAVE_EDIT_READLINE_READLINE_H)
#    include <edit/readline/readline.h>
#  elif defined(HAVE_EDITLINE_READLINE_H)
#    include <editline/readline.h>
#  else /* !defined(HAVE_READLINE_H) */
extern char *readline ();
#  endif /* !defined(HAVE_READLINE_H) */
char *cmdline = NULL;
#else /* !defined(HAVE_READLINE_READLINE_H) */

/* no readline; make one up */

static FILE* rl_instream;
static FILE* rl_outstream;

static char *readline( const char* prompt ) {

#define BUFFER_SIZE 1024

    char *buffer = malloc( BUFFER_SIZE );
    size_t len;
    fputs( prompt, rl_outstream );
    fgets( buffer, BUFFER_SIZE, rl_instream );
    len = strlen(buffer);
    if ( buffer[len-1] == '\n' )
	buffer[len-1] = '\0';

    return buffer;
}
#endif /* HAVE_LIBREADLINE */

#ifdef HAVE_READLINE_HISTORY
#  if defined(HAVE_READLINE_HISTORY_H)
#    include <readline/history.h>
#  elif defined(HAVE_HISTORY_H)
#    include <history.h>
#  elif defined(HAVE_EDIT_READLINE_HISTORY_H)
#    include <edit/readline/history.h>
#  else /* !defined(HAVE_HISTORY_H) */
extern void add_history ();
extern int write_history ();
extern int read_history ();
#  endif /* defined(HAVE_READLINE_HISTORY_H) */
  /* no history */
#endif /* HAVE_READLINE_HISTORY */

static char *UnquoteValue (char *in, char *out);
static void PQueryUser (ParamFile pfile, Parameter * p, Value * value);


#if ! defined(NONCIAO) && defined(HAVE_LIBREADLINE)
static int
print_escapes (void)
{
  if (NULL != getenv ("CIAO_RL_PREFIX"))
    {
      fprintf (rl_outstream, getenv ("CIAO_RL_PREFIX"));
    }

  return (0);
}
#endif


int 
ParamSetX (ParamFile pfile, const char *name, VType type, void *value)
{
  Parameter *p;
  int field;
  char xname[SZ_PFLINE];
  int mode = pfile->mode;
  char *freeval = NULL;
  int ok = 0;

  if (value && (TypeIndex (type) == StringType))
    {
      value = QuoteValue (value, NULL);
      freeval = value;
    }

  field = PParseName (name, xname);

  if (NULL != (p = ParamLookup (pfile->psets, xname)))
    {
      switch (field)
	{
	case PMODE:
	  if ((value = Convert (value, type, NULL, ModeType)))
	    {
	      p->pmode = *(int *) value;
	      ok = 1;
	    }
	  break;
	case PMIN:
	  {
	    Value pvalue;
	    Value *result;

	    pvalue.value = NULL;
	    pvalue.indir = NULL;
	    VNewValue (&pvalue, value, type);

	    while (((result = VIsIndirect (&pvalue)) == NULL)
		   && ((result = VRange (pfile, p, &pvalue)) == NULL)
		   && ((result = VAccess (pfile, p, p->ptype, &pvalue)) ==
		       NULL) && !(mode & HHMODE))
	      PQueryUser (pfile, p, &pvalue);

	    VNewValue (&p->pmin, pvalue.value, pvalue.type);

	    if (pvalue.value)
	      free (pvalue.value);
	    if (result)
	      ok = 1;
	    break;
	  }
	case PMAX:
	  {
	    Value pvalue;
	    Value *result;

	    pvalue.value = NULL;
	    pvalue.indir = NULL;
	    VNewValue (&pvalue, value, type);

	    while (((result = VIsIndirect (&pvalue)) == NULL)
		   && ((result = VAccess (pfile, p, p->ptype, &pvalue)) ==
		       NULL) && !(mode & HHMODE))
	      PQueryUser (pfile, p, &pvalue);

	    VNewValue (&p->pmax, pvalue.value, pvalue.type);

	    if (pvalue.value)
	      free (pvalue.value);
	    if (result)
	      ok = 1;
	    break;
	  }
	case PVALUE:
	case PWORTH:
	  {
	    Value pvalue;
	    Value *result;
	    int interact = 0;

	    pvalue.value = NULL;
	    pvalue.indir = NULL;
	    VNewValue (&pvalue, value, type);

	    if (p->phook != NULL)
	      InitList (p);

	    while (((result = VIsIndirect (&pvalue)) == NULL)
		   && ((result = cxcValidate (pfile, p, &pvalue, interact)) ==
		       NULL) && !(mode & HHMODE))
	      {
		interact = 1;
		PQueryUser (pfile, p, &pvalue);
	      }
	    VNewValue (&p->pvalue, pvalue.value, pvalue.type);

	    if (pvalue.value)
	      free (pvalue.value);
	    if (result)
	      ok = 1;
	    break;
	  }
	case PPROMPT:
	  if ((value = Convert (value, type, NULL, StringType)))
	    {
	      int length = strlen (value);

	      if (p->pprompt)
		free (p->pprompt);
	      p->pprompt = (char *) malloc (length + 1);
	      memmove (p->pprompt, value, length);
	      p->pprompt[length] = '\0';
	      ok = 1;
	    }
	  break;
	default:
	  break;
	}
    }
  else
    {
      parerr = PARNOTFOUND;
      value = NULL;
    }


  if (freeval != NULL)
    free (freeval);
  return ok;
}


void *
ParamGetX (ParamFile pfile, const char *name, VType type, int *xmode)
{
  Parameter *p;
  Parameter *a;
  int field;
  void *value = NULL;
  char xname[SZ_PFLINE];

  int zero = 0;

  if (*name == '$')
    {
      if (!strcmp (name, "$nargs"))
	{
	  if (pfile->alist != NULL)
	    {
	      value =
		Convert (&pfile->alist->nparam, IntegerType, NULL, type);
	    }
	  else
	    {
	      value = Convert (&zero, IntegerType, NULL, type);
	    }
	}
      else
	{
	  return NULL;
	}
    }
  else
    {

      field = PParseName (name, xname);

      if (NULL != (p = ParamLookup (pfile->psets, xname)))
	{
	  int mode;

	  if (p->pmode & AMODE)
	    {
	      mode = *xmode;
	      if (mode & HMODE)
		mode |= LMODE;
	    }
	  else
	    {
	      mode = p->pmode;
	      if (mode & HMODE)
		mode &= ~LMODE;
	    }

	  if (*xmode & (HMODE | HHMODE) || mode & (HMODE | HHMODE))
	    mode &= ~QMODE;
	  if (*xmode & HHMODE)
	    mode |= HHMODE;

	  a = ParamLookup (pfile->alist, xname);
	  parerr = 0;

	  switch (field)
	    {
/*
	         case PTYPE:
	                value = (char *)malloc(SZ_PFLINE);
	                Type2String(p->ptype, value);
	                break;
*/
	    case PMODE:
	      value = Convert (&p->pmode, ModeType, NULL, type);
	      break;

	    case PMIN:
	      {
		Value *result;

		while (((result = VRange (pfile, p, &p->pmin)) == NULL)
		       && ((result = VAccess (pfile, p, p->ptype, &p->pmin))
			   == NULL) && !(mode & HHMODE))
		  PQueryUser (pfile, p, &p->pmin);

		value = VConvert (&p->pmin, NULL, type);
		break;
	      }
	    case PMAX:
	      {
		Value *result;

		while (((result =
			 VAccess (pfile, p, p->ptype, &p->pmax)) == NULL)
		       && !(mode & HHMODE))
		  PQueryUser (pfile, p, &p->pmax);

		value = VConvert (&p->pmax, NULL, type);
		break;
	      }
	    case PWORTH:
	      {
		Value *fld = a ? &a->pvalue : &p->pvalue;
		Value *result;
		int inter = 0;
		if (IsInList (p))
		  {
		    result = NextListValue (pfile, p);
		  }
		else
		  {
		    if ((a == NULL) && (mode & QMODE))
		      {
			if (p->pquery != PARAM_ALREADY_READ)
			  {
			    inter = 1;
			    PQueryUser (pfile, p, fld);
			  }
		      }



		    while (((result =
			     cxcValidate (pfile, p, fld, inter)) == NULL)
			   && !(mode & HHMODE))
		      {
			PQueryUser (pfile, p, fld);
			p->redirect = 0;
			inter = 1;
		      }
		    if (result)
		      {
			if (a && mode & LMODE)
			  {
			    VNewValue (&p->pvalue, result->value,
				       result->type);
			    if (a && a->pvalue.indir
				&& (a->pvalue.indir == result->indir))
			      {
				/* since a and p are both freed later, copy ** 
				 ** indir value to new buffer to avoid F.U.M */
				char *inval = (char *)
				  malloc (strlen (result->indir) + 1);
				strcpy (inval, result->indir);
				p->pvalue.indir = inval;
			      }
			    else
			      p->pvalue.indir = result->indir;
			  }

			if (IsInList (p))
			  result = NextListValue (pfile, p);

			p->pquery = PARAM_ALREADY_READ;
		      }
		    else
		      {
			value = NULL;
		      }
		  }

		value = VConvert (result, NULL, type);
		break;
	      }
	    case PVALUE:
	      value = VConvert (&p->pvalue, NULL, type);
	      break;
	    case PPROMPT:
	      {
		value = Convert (p->pprompt, StringType, NULL, type);
		break;
	      }
	    default:
	      value = NULL;
	    }
	}
      else
	{
	  parerr = PARNOTFOUND;
	  value = NULL;
	}
    }

  if (value && TypeIndex (type) == StringType)
    UnquoteValue (value, value);
  return value;
}


static void
PQueryUser (ParamFile pfile, Parameter * p, Value * value)
{
  Value pvalue;
  char dv[SZ_PFLINE], dn[SZ_PFLINE], dx[SZ_PFLINE];
  char line[SZ_PFLINE], prompt[SZ_PFLINE], buf_tmp[SZ_PFLINE];
  char *dvp, *dnp, *dxp, *rdline, *buf_p;
  int count;

  FILE *dev_tty = NULL;

  memset (dv, 0, SZ_PFLINE);
  memset (dn, 0, SZ_PFLINE);
  memset (dx, 0, SZ_PFLINE);
  memset (line, 0, SZ_PFLINE);
  memset (prompt, 0, SZ_PFLINE);
  memset (buf_tmp, 0, SZ_PFLINE);

  if (p == NULL)
    return;

  dv[0] = '\0';

  dvp = VConvert (&p->pvalue, dv, StringType);

#ifdef HAVE_READLINE_HISTORY
  if (dvp && strlen (dvp))
    add_history (dvp);
#endif

  /* If the PVALUE of the value is an indirection lookup the
   ** PWORTH of it. 
   **/
  pvalue.value = NULL;
  pvalue.indir = NULL;
  VNewValue (&pvalue, p->pvalue.value, p->pvalue.type);
  if (dvp && VIsIndirect (&pvalue))
    {
      strcat (dv, " -> ");
      if (VAccess (pfile, p, StringType, &pvalue))
	{
	  if (p->pvalue.indir)	/* update redirect in prompt */
	    {
	      strcpy (dv, p->pvalue.indir);
	      strcat (dv, " -> ");
	    }
	  /* else strcat(dv, "... -> "); ** change cyclic redirect prompt? */
	  strcat (dv, pvalue.value);
#ifdef HAVE_READLINE_HISTORY
	  add_history (pvalue.value);
#endif
	}
      else if ((p->pvalue.value) && (p->pvalue.indir))
	{
	  buf_p = VConvert (&p->pvalue, buf_tmp, StringType);
	  strcpy (dv, p->pvalue.indir);
	  strcat (dv, " -> ");
	  strcat (dv, buf_p);
	  free (p->pvalue.value);
	  p->pvalue.value = p->pvalue.indir;
	  p->pvalue.indir = NULL;
	  p->pvalue.type = StringType;
	}
      else
	{
	  buf_p = VConvert (&p->pvalue, buf_tmp, StringType);
	  if (buf_p && (strcmp (p->pdefault, buf_p) != 0))
	    strcpy (dv, buf_p);
	  else
	    strcat (dv, "INDEF");
	}
    }
  if (pvalue.value)
    free (pvalue.value);
  if (pvalue.indir)
    {
      free (pvalue.indir);
      pvalue.indir = NULL;
    }
  if (value->indir)
    {
      free (value->indir);
      value->indir = NULL;
    }

#ifdef HAVE_READLINE_HISTORY
  /*      if ( value->value && TypeIndex(value->type) == StringType )
     add_history(value->value); */
#endif

  dnp = VConvert (&p->pmin, dn, StringType);
  dxp = VConvert (&p->pmax, dx, StringType);

  if (parerr)
    paramerr (0, "pquery", p->pname);

  strcpy (prompt, p->pprompt ? p->pprompt : p->pname);
  strcat (prompt, dnp || dxp ? " (" : "");
  strcat (prompt, dnp ? dn : "");
  if (strchr (dn, '|') == NULL)	/* Don't put : when enumerated */
    strcat (prompt, dnp || dxp ? ":" : "");
  strcat (prompt, dxp ? dx : "");
  strcat (prompt, dnp || dxp ? ")" : "");
  strcat (prompt, dvp ? " (" : "");
  strcat (prompt, dvp ? dv : "");
  strcat (prompt, dvp ? ")" : "");
  strcat (prompt, ": ");


  if (strstr (dn, "|"))
    {
      char enumer[MAXBUFSIZE];
      char *enumer_p;
      strcpy (enumer, dn);

      enumer_p = strtok (enumer, "|");
      while (enumer_p)
	{
#ifdef HAVE_READLINE_HISTORY
	  if (strlen (enumer_p))
	    add_history (enumer_p);
#endif
	  enumer_p = strtok (NULL, "|");
	}

    }

  if (dev_tty == NULL)
    {
      dev_tty = fopen ("/dev/tty", "r");
      rl_instream = dev_tty;
      rl_outstream = stderr;
    }

#if ! defined(NONCIAO) && defined(HAVE_LIBREADLINE)
  if (NULL != getenv ("CIAO_RL_PREFIX"))
    {
      rl_pre_input_hook = &print_escapes;
    }
#endif

  rdline = readline (prompt);

#if ! defined(NONCIAO) && defined(HAVE_LIBREADLINE)
  if (NULL != getenv ("CIAO_RL_PREFIX"))
    {
      fprintf (rl_outstream, "[m");
    }
#endif

  if (rdline)
    {
      strcpy (line, rdline);
      /* strip off trail white space */
      while ((count = strlen (line)) > 1)
	{
	  if (line[count - 1] == ' ')
	    {
	      line[count - 1] = '\0';
	    }
	  else
	    {
	      break;
	    }
	}
    }
  else
    {
      line[0] = 0;
    }

#ifdef HAVE_READLINE_HISTORY
  if (rdline && strlen (line))
    add_history (line);
#endif

  if ((rdline != NULL) && (count > 0) && *((char *) line) != '\n')
    {
      line[count] = '\0';

      QuoteValue (line, line);

      VNewValue (value, line, StringType);
    }
  else
    VNewValue (value, p->pvalue.value, p->pvalue.type);

  if (rdline)
    free (rdline);

}


int
PParseName (const char *fname, char *xname)
{
  char f[SZ_PFLINE];
  char *subfield;
  int field;

  strcpy (f, fname);
  subfield = strchr (f, '.');

  if (subfield)
    {
      *subfield++ = '\0';

      /* switch */
      if (!strcmp (subfield, "p_mode"))
	field = PMODE;
/*	else if ( !strcmp(subfield, "p_type")  )	field = PTYPE; */
      else if (!strcmp (subfield, "p_value"))
	field = PVALUE;
      else if (!strcmp (subfield, "p_min"))
	field = PMIN;
      else if (!strcmp (subfield, "p_max"))
	field = PMAX;
      else if (!strcmp (subfield, "p_prompt"))
	field = PPROMPT;
      else
	{
	  parerr = PARSUBFIELD;
	  field = 0;
	}
    }
  else
    field = PWORTH;

  strcpy (xname, f);
  return field;
}


char *
QuoteValue (char *in, char *out)
{
  char *tbuf, *t;
  int i;
  int len = strlen (in);

  t = tbuf = (char *) malloc ((len * 2) + 1);

  if (!strcmp (in, "\"\""))
    {
      tbuf[0] = '\0';
    }
  else
    {
      for (i = 0; i < len; i++)
	{
	  if (in[i] == '\n')
	    {
	      *t++ = '\\';
	      *t++ = 'n';
	      continue;
	    };
	  if (in[i] == '\t')
	    {
	      *t++ = '\\';
	      *t++ = 't';
	      continue;
	    };

	  *t++ = in[i];
	}
      *t = '\0';
    }

  if (out == NULL)
    out = tbuf;
  else
    {
      strcpy (out, tbuf);
      free (tbuf);
    }

  return out;
}


static char *
UnquoteValue (char *in, char *out)
{
  char *tbuf, *t;
  int i;
  int len = strlen (in);

  t = tbuf = (char *) malloc (len + 1);

  for (i = 0; i < len; i++)
    {
      if (in[i] == '\\')
	{
	  if (i + 1 < len)
	    {
	      if (in[i + 1] == 'n')
		{
		  *t++ = '\n';
		  i++;
		  continue;
		}
	      if (in[i + 1] == 't')
		{
		  *t++ = '\t';
		  i++;
		  continue;
		}
	    }
	  *t++ = '\\';
	}
      else
	*t++ = in[i];
    }
  *t = '\0';

  if (out == NULL)
    out = tbuf;
  else
    {
      strcpy (out, tbuf);
      free (tbuf);
    }

  return out;
}
