package Unicode::Collate::CJK::Stroke;

use 5.006;
use strict;
use warnings;

our $VERSION = '1.23';

my %u2p;
my $wt = 0x8000;

while (<DATA>) {
    last if /^__END__/;
    my @c = split;
    for my $c (@c) {
	next if !$c;
	$u2p{hex($c)} = $wt if $c !~ /-/;
	$wt++;
    }
}

sub weightStroke {
    my $u = shift;
    return exists $u2p{$u} ? $u2p{$u} : undef;
}

1;
__DATA__
FDD0-2801
4E00 4E28 4E36 4E3F 4E40 4E41 2E84 4E59 4E5A 4E5B
200CA 200CB 200CC 200CD 200D1 4E85 2010C 3006 3007 3021
3025 303B
FDD0-2802
4E01 4E02 4E03 4E04 4E05 4E06 4E29 4E37 4E42 4E43
4E44 20086 20087 2008A 4E5C 4E5D 4E86 2010E 4E8C 4EA0
4EBA 4EBB 513F 5165 516B 2E86 5182 5196 51AB 2E87
51E0 51F5 2E88 5200 5201 5202 529B 52F9 5315 531A
5338 5341 2E8A 535C 5369 5382 53B6 2E80 53C8 5DDC
8BA0 2ECF 2ED6 28E0F 3022 3024 3026
FDD0-2803
4E07 4E08 4E09 4E0A 4E0B 4E0C 4E90 5344 3404 4E2A
4E2B 4E38 4E49 4E45 4E46 4E47 4E48 4E4A 4E5E 4E5F
4E60 4E87 4E8D 4E8E 4E8F 4EA1 4EBC 4EBD 4EBE 4EBF
5140 5166 51E1 51E2 51E3 5203 5204 529C 52FA 5342
5343 353E 536A 536B 53C9 53E3 56D7 571F 58EB 5902
590A 5915 5927 5928 5973 5B50 5B51 5B52 5B53 5B80
5BF8 2E8C 2E8D 5C0F 5C22 21BC1 21BC2 5C38 5C6E 5C71
5DDB 5DDD 21FE8 5DE5 5DF1 5DF2 5DF3 5DFE 5E72 4E61
5E7A 5E7F 5EF4 5EFE 5F0B 5F13 2E95 5F50 5F51 5F61
5F73 5FC4 624C 624D 6C35 72AD 7E9F 2EBE 8279 2ECC
95E8 961D 98DE 9963 9A6C 3005 3023 3027
FDD0-2804
4E0D 4E0E 4E0F 4E10 4E11 4E12 4E13 4E2D 4E2E 4E2F
4E30 4E39 4E3A 4E4B 4E4C 5C39 4E63 4E64 4E65 4E66
4E88 4E91 4E92 4E93 4E94 4E95 4E96 4EA2 4EA3 4EC0
4EC1 4EC2 4EC3 4EC4 4EC5 4EC6 4EC7 4EC8 4EC9 4ECA
4ECB 4ECC 4ECD 4ECE 4ECF 4ED0 4ED1 4ED2 4ED3 5141
5142 5143 5167 516C 516D 516E 516F 5183 5184 5185
5186 5187 5188 34C1 5197 5198 51E4 20630 51F6 20676
5205 5206 5207 5208 529D 529E 52FB 52FC 52FD 52FE
52FF 5300 5301 5302 352B 5316 5339 533A 3539 5345
5346 5347 5348 535D 535E 536C 5383 5384 5385 5386
53B7 53B8 53B9 53CA 53CB 53CC 53CD 53CE 5720 5721
2123C 58EC 5903 5929 592A 592B 592C 592D 5B54 5C10
5C11 5C23 5C24 5C3A 5C6F 4E62 5C72 5DF4
5DFF 5E00 5E01 5E7B 5EFF 5F00 5F0C 5F14 5F15 5F16
5FC3 5FC6 6208 6236 6237 6238 624B 624E 22A66 652F
6534 6535 6587 6597 65A4 65B9 65E0 65E1 2E9C 65E5
66F0 2E9D 6708 6728 6729 233B4 6B20 6B62 6B79 6BB3
6BCB 6BCC 6BD4 6BDB 6C0F 6C14 6C34 706B 706C 2EA5
722A 722B 7236 723B 4E2C 723F 7247 3E26 7259 2EA7
725B 725C 72AC 738B 248E9 793B 25605 7F53 8002 8080
2EBC 89C1 8BA1 8BA2 8BA3 8BA4 8BA5 8D1D 8F66 2ECD
8FB6 95E9 97E6 98CE 3028 3029
FDD0-2805
4E17 3400 4E14 4E15 4E16 4E18 4E19 4E1A 4E1B 4E1C
4E1D 4E31 4E3B 4E3C 4E4D 4E4E 4E4F 4E50 20094 4E67
4E97 3430 3431 3432 3433 3434 3435 3436 3437 4ED4
4ED5 4ED6 4ED7 4ED8 4ED9 4EDA 4EDB 4EDC 4EDD 4EDE
4EDF 4EE0 4EE1 4EE2 4EE3 4EE4 4EE5 4EE6 4EE7 4EE8
4EE9 4EEA 4EEB 4EEC 4EED 201A9 201AB 5144 5145 34B0
5170 5189 518A 518B 518C 20547 5199 519A 51AC 51AD
51AE 51AF 51E5 51E6 51E7 51F7 51F8 51F9 51FA 51FB
5209 520A 520B 520C 520D 529F 52A0 52A1 52A2 5303
5304 5305 5306 5307 5317 531B 531C 531D 531E 5349
534A 534C 535F 5360 5361 5362 536D 536E 536F 20A11
5387 5388 5389 53BA 53BB 53BC 53CF 53D0 53D1 53E4
53E5 53E6 53E7 53E8 53E9 53EA 53EB 53EC
53ED 53EE 53EF 53F0 53F1 53F2 53F3 53F4 53F5 53F6
53F7 53F8 53F9 53FA 53FB 53FC 53FD 53FE 20BA8 20BA9
56D8 56D9 56DA 56DB 56DC 3626 5722 5723 5724 5725
5726 5727 58ED 5904 5916 5917 5918 592E 592F 5930
5931 5932 5933 5934 5974 5975 5976 5B55 5B81 5B82
5B83 5B84 5BF9 5C12 5C13 5C14 5C15 5C3B 5C3C 5C73
5C74 5C75 5C76 5C77 5DE6 5DE7 5DE8 3832 3833 5E02
5E03 5E04 5E05 5E73 5E7C 5E80 5E81 5E82 5E83 5F01
5F0D 5F17 5F18 5F52 38FA 5FC5 5FC7 5FC9 5FCA 225AF
6209 620A 620B 6239 6250 6251 6252 6253 6254 6255
6256 624F 65A5 65E6 65E7 2313D 66F1 672A 672B 672C
672D 672E 672F 6730 6B63 6B7A 6BCD 6C10 6C11 6C15
6C3A 6C36 6C37 6C38 6C39 6C3B 6C3E 6C3F
6C40 6C41 6C43 6C44 6C45 6C47 6C48 6C49 706D 72AE
72AF 72B0 7384 7389 738A 738C 738D 74DC 74E6 7518
751F 7528 7529 7530 7531 7532 7533 7534 7535 2EAA
758B 24D13 7592 7676 767D 76AE 76BF 76EE 77DB 77E2
77F3 2EAC 793A 79B8 79BE 7A74 7ACB 7EA0 7F52 2626A
2626B 808A 827A 8864 27607 8BA6 8BA7 8BA8 8BA9 8BAA
8BAB 8BAC 8BAD 8BAE 8BAF 8BB0 8BB1 8F67 8FB7 9092
9093 9485 957F 95EA 961E 961F 9964 9965 9A6D 9E1F
9F99
FDD0-2806
3401 4E1E 4E1F 4E20 4E21 4E22 4E51 4E52 4E53 4E54
4E68 4E69 4E6A 4E6B 4E6C 4E6D 4E6E 4E6F 4E70 4E89
4E98 4E99 4E9A 342B 4EA4 4EA5 4EA6 4EA7 3438 3439
343B 343F 3440 4EEE 4EEF 4EF0 4EF1 4EF2 4EF3 4EF4
4EF5 4EF6 4EF7 4EF8 4EF9 4EFA 4EFB 4EFC 4EFD 4EFE
4EFF 4F00 4F01 4F02 4F03 4F04 4F05 4F06 4F07 4F08
4F09 4F0A 4F0B 4F0C 4F0D 4F0E 4F0F 4F10 4F11 4F12
4F13 4F14 4F15 4F16 4F17 4F18 4F19 4F1A 4F1B 4F1C
4F1D 4F1E 4F1F 4F20 4F21 4F22 4F23 4F24 4F25 4F26
4F27 4F28 4F29 4F2A 4F2B 4F2C 4F64 201C1 201D4 5146
5147 5148 5149 514A 5168 6C3D 5171 5172 5173 5174
518D 518E 519B 519C 51B0 51B1 51B2 51B3 51B4 205B3
51E8 51E9 51EA 51EB 51FC 520E 520F 5210
5211 5212 5213 5214 5215 5216 5217 5218 5219 521A
521B 52A3 52A4 52A5 52A6 52A7 52A8 5308 531F 5320
5321 5322 353B 534B 534D 534E 534F 5350 5370 5371
3542 538A 538B 538C 538D 53BD 53BE 53D2 3566 53FF
5400 5401 5402 5403 5404 5405 5406 5407 5408 5409
540A 540B 540C 540D 540E 540F 5410 5411 5412 5413
5414 5415 5416 5417 20BBF 20BC6 56DD 56DE 56DF 56E0
56E1 56E2 56E3 5728 5729 572A 572B 572C 572D 572E
572F 5730 5731 5732 5733 5734 5735 5736 5737 5738
5739 573A 2124F 58EE 5905 5919 591A 591B 5935 5936
5937 5938 5939 593A 593B 593C 36A5 5977 5978 5979
597A 597B 597C 597D 597E 597F 5980 5981 5982 5983
5984 5985 5986 5987 5988 216B8 216BA 5B56
5B57 5B58 5B59 21944 5B85 5B86 5B87 5B88 5B89 5BFA
5BFB 5BFC 5C16 5C17 5C18 5C25 5C26 5C27 5C3D 21C2A
5C70 5C78 5C79 5C7A 5C7B 5C7C 5C7D 5C7E 5C7F 5C80
5C81 5C82 5C83 21D46 5DDE 5DDF 5DE9 5DEA 3834 3835
3836 5E06 5E07 5E08 5E74 5E75 5E76 5E84 5E85 5E86
5EF5 5F02 5F0E 5F0F 5F10 5F19 5F1A 5F1B 5F1C 5F53
5F74 5F75 5FC8 5FCB 5FCF 5FD3 5FD4 5FD5 5FD6 5FD9
5FDA 5FDB 225BE 620C 620D 620E 620F 6210 6257 6258
6259 625A 625B 625C 625D 625E 6260 6261 6262 6263
6264 6265 6266 6267 6268 6269 626A 626B 626C 625F
6530 6536 6537 65E8 65E9 65EA 65EB 65EC 65ED 65EE
65EF 3B30 66F2 66F3 6709 3B41 6731 6732 6733 6734
6735 6736 6737 6738 6739 673A 673B 673C
673D 673E 673F 6740 6741 6742 6743 6B21 6B22 6B64
6B7B 6BCE 6BD5 6C12 6C16 6C17 6C18 6C3C 6C46 6C4A
6C4B 6C4C 6C4D 6C4E 6C4F 6C50 6C51 6C52 6C53 6C54
6C55 6C57 6C58 6C59 6C5A 6C5B 6C5C 6C5D 6C5F 6C60
6C61 6C62 6C63 6C64 6C77 706E 706F 7070 7071 7072
7073 7237 725D 725E 725F 72B1 72B2 72B3 72B4 72B5
72B7 72B8 24725 3EA8 3EA9 3EAA 738E 738F 7390 7391
752A 7536 767E 767F 2EAE 793C 7A75 7AF9 7C73 7CF8
7CF9 7EA1 7EA2 7EA3 7EA4 7EA5 7EA6 7EA7 7EA8 7EA9
7EAA 7EAB 7F36 7F51 2EB6 7F8A 2634B 2634C 7FBD 8001
8003 800C 8012 8033 807F 26612 8089 808B 808C 808D
808E 81E3 81EA 81F3 81FC 26951 820C 821B 821F 826E
8272 8278 827B 827C 827D 827E 827F 8280
8281 8282 864D 866B 8840 884C 8863 897E 897F 8980
89C2 8BB2 8BB3 8BB4 8BB5 8BB6 8BB7 8BB8 8BB9 8BBA
8BBB 8BBC 8BBD 8BBE 8BBF 8BC0 8D1E 8D1F 8D20 8D71
8F68 8FB8 8FB9 8FBA 8FBB 8FBC 8FBD 9094 9096 9097
9098 9099 909A 909B 909C 909D 9486 9487 95EB 95EC
95ED 95EE 95EF 9620 9621 9622 9623 9624 9875 9966
9967 9A6E 9A6F 9A70 9F50
FDD0-2807
4E23 4E24 4E25 4E3D 20021 4E32 20068 4E55 4E71 4E72
4E8A 20118 4E9C 4EA8 4EA9 4EAA 3446 4F2D 4F2E 4F2F
4F30 4F31 4F32 4F33 4F34 4F35 4F36 4F37 4F38 4F39
4F3A 4F3B 4F3C 4F3D 4F3E 4F3F 4F40 4F41 4F42 4F43
4F44 4F45 4F46 4F47 4F48 4F49 4F4A 4F4B 4F4D 4F4E
4F4F 4F50 4F51 4F52 4F53 4F54 4F55 4F56 4F57 4F58
4F59 4F5A 4F5B 4F5C 4F5D 4F5E 4F5F 4F60 4F61 4F62
4F63 4F65 4F66 4F67 4F68 201F2 514B 514C 514D 514E
514F 5150 5151 34B3 5175 518F 519D 34C8 51B5 51B6
51B7 51B8 51B9 51BA 51BB 51EC 34DF 521C 521D 521E
521F 5220 5221 5222 5223 5224 5225 5226 5227 5228
5229 522A 522B 522C 522D 52A9 52AA 52AB 52AC 52AD
52AE 52AF 52B0 52B1 52B2 52B3 52B4 5309
208D5 3530 5323 5324 5325 3537 533B 5363 5364 5372
5373 5374 5375 538E 538F 5390 5391 53BF 53D3 356D
3570 3572 5418 5419 541A 541B 541C 541D 541E 541F
5420 5421 5422 5423 5424 5425 5426 5427 5428 5429
542A 542B 542C 542D 542E 542F 5430 5431 5432 5433
5434 5435 5436 5437 5438 5439 543A 543B 543C 543D
543E 543F 5440 5441 5442 5443 5444 5445 5446 5447
5448 5449 544A 544B 544C 544D 544E 544F 5450 5451
5452 5453 5454 5455 5456 5457 5458 5459 545A 545B
545C 20BCB 20BE2 20BEB 20BFB 20BFF 56E4 56E5 56E6 56E7
56E8 56E9 56EA 56EB 56EC 56ED 56EE 56EF 56F0 56F1
56F2 56F3 56F4 56F5 3629 362B 362D 362E 3630 573B
573C 573D 573E 573F 5740 5741 5742 5743
5744 5745 5746 5747 5748 5749 574A 574B 574C 574D
574E 574F 5750 5751 5752 5753 5754 5755 5756 5757
5758 5759 575A 575B 575C 575D 575E 575F 5760 2127C
58EF 58F0 58F1 58F2 58F3 5906 590B 593D 593E 593F
5940 5941 5942 36AA 36AC 5989 598A 598B 598C 598E
598F 5990 5991 5992 5993 5994 5995 5996 5997 5998
5999 599A 599B 599C 599D 599E 599F 59A0 59A1 59A2
59A3 59A4 59A5 59A6 59A7 59A8 59A9 59AA 59AB 216C0
216C1 216C2 216D3 216D5 5B5A 5B5B 5B5C 5B5D 5B5E 5B8A
5B8B 5B8C 5B8D 5B8E 5B8F 5B90 5B91 5B92 5BFD 5BFE
5BFF 5C28 5C29 5C2A 5C2B 5C2C 5C3E 5C3F 5C40 5C41
5C42 5C43 5C84 5C85 5C86 5C87 5C88 5C89 5C8A 5C8B
5C8C 5C8E 5C8F 5C90 5C91 5C92 5C93 5C94
5C95 5C96 5C97 5C98 5C99 5C9A 5C9B 5C9C 5C8D 5DE0
5DEB 5DF5 3837 3838 3839 383B 5E09 5E0A 5E0B 5E0C
5E0D 5E0E 5E0F 5E10 5E87 5E88 5E89 5E8A 5E8B 5E8C
5E8D 5E8E 5E8F 5E90 5E91 5E92 5E93 5E94 5EF7 5F03
5F04 5F05 5F1D 5F1E 5F1F 5F20 223BD 5F62 5F63 5F64
5F76 5F77 5F78 5F79 5F7A 5F7B 3900 5FCC 5FCD 5FCE
5FD0 5FD1 5FD2 5FD7 5FD8 5FDC 3908 5FDF 5FE1 5FE3
5FE4 5FE6 5FE7 5FE8 5FEA 5FEB 5FEC 5FED 5FEE 5FEF
5FF0 5FF1 5FF2 5FF3 5FF4 5FF6 5FF7 5FF8 5FF9 5FFA
5FFB 5FFC 5FFE 6000 6003 6004 6005 6006 6211 6212
6213 22993 623A 623B 623C 39D1 626D 626E 626F 6270
6271 6272 6273 6274 6275 6276 6277 6279 627A 627B
627C 627D 627E 6280 6281 6282 6283 6284
6285 6286 6287 6288 6289 628A 628B 628C 628D 628E
628F 6290 6291 6292 6293 6294 6295 6296 6297 6298
6299 629A 629B 629C 629D 629E 629F 62A0 62A1 62A2
62A3 62A4 62A5 6278 6538 6539 653A 653B 653C 22EEF
6588 6598 65F0 65F1 65F2 65F3 65F4 65F5 65F6 65F7
65F8 66F4 66F5 3B42 3B43 3B44 3B45 3B46 6744 6745
6746 6747 6748 6749 674A 674B 674C 674D 674E 674F
6750 6751 6752 6753 6754 6755 6756 6757 6758 6759
675A 675B 675C 675D 675E 675F 6760 6761 6762 6763
6764 6765 6766 6767 6768 6769 6781 233CC 6B24 3C50
6B65 6B7C 239C2 6BCF 6BD0 23AEE 6BDC 6BDD 6C19 6C1A
6C42 6C56 6C5E 6C65 6C66 6C67 6C68 6C69 6C6A 6C6B
6C6D 6C6E 6C6F 6C70 6C71 6C72 6C73 6C74
6C75 6C76 6C78 6C79 6C7A 6C7B 6C7C 6C7D 6C7E 6C7F
6C81 6C82 6C83 6C84 6C85 6C86 6C87 6C88 6C89 6C8B
6C8C 6C8D 6C8E 6C8F 6C90 6C91 6C92 6C94 6C95 6C96
6C98 6C99 6C9A 6C9B 6C9C 6C9E 6C9F 6CA0 6CA1 6CA2
6CA3 6CA4 6CA5 6CA6 6CA7 6CA8 6CA9 6CAA 23C99 23C9A
23C9B 3DA5 7074 7075 7076 7077 7078 7079 707A 707B
707C 707D 707E 707F 7080 7260 7261 7262 7263 7264
24618 72B6 72B9 72BA 72BB 72BC 72BD 72BE 72BF 72C1
72C2 72C3 72C4 72C5 72C6 72C7 72C8 2472F 3EAD 7392
7393 7394 7395 7396 7397 7398 7399 739A 739B 248F0
248F1 248F2 248F3 74E7 752B 752C 3F55 7537 7538 7539
753A 753C 7593 7594 7595 7596 7597 7680 7681 7682
7683 76C0 76C1 76EF 77E3 77F4 77F5 77F6
25419 793D 4126 79BF 79C0 79C1 79C2 79C3 7A76 7A77
7ACC 7ACD 7CFA 7CFB 7EB6 7EAC 7EAD 7EAE 7EAF 7EB0
7EB1 7EB2 7EB3 7EB4 7EB5 7EB7 7EB8 7EB9 7EBA 7EBB
7EBC 7EBD 7EBE 7F55 8034 8090 8091 8092 8093 8094
8095 8096 8097 8098 8099 809A 809B 809C 809D 809E
809F 80A0 81EB 826F 8283 8284 8285 8286 8287 8288
8289 828A 828B 828C 828D 828E 828F 8290 8291 8292
8293 8295 8296 8297 26B05 26B0A 866C 2760C 898B 89C3
2EC6 89D2 278B2 8A00 8A01 8BC1 8BC2 8BC3 8BC4 8BC5
8BC6 8BC7 8BC8 8BC9 8BCA 8BCB 8BCC 8BCD 8BCE 8BCF
8BD0 8BD1 8BD2 8C37 8C46 8C55 8C78 8C9D 8D21 8D22
8D64 8D70 2ECA 8DB3 8EAB 8ECA 8F69 8F6A 8F6B 8F9B
8FB0 8FB5 5DE1 8FBE 8FBF 8FC0 8FC1 8FC2
8FC3 8FC4 8FC5 8FC6 8FC7 8FC8 8FC9 28468 2846C 28473
9091 909E 909F 90A0 90A1 90A2 90A3 90A4 90A5 90A6
90A7 90A8 90A9 90AA 90AC 286AA 286AB 9149 91C6 91CC
9488 9489 948A 948B 948C 95F0 95F1 95F2 95F3 95F4
95F5 95F6 95F7 9625 9626 9627 9628 9629 962A 962B
962C 962D 962E 962F 9630 9631 9632 9633 9634 9635
9636 28E36 28E39 97E7 98CF 9968 9969 996A 996B 996C
996D 996E 9A71 9A72 9A73 9A74 9E20 9E21 9EA6 9F9F
FDD0-2808
4E26 4E27 4E33 4E56 4E73 4E74 4E75 4E76 4E77 4E78
200EE 4E8B 4E9B 4E9D 4E9E 4E9F 342D 4EAB 4EAC 4F4C
344C 3450 4F69 4F6A 4F6B 4F6C 4F6D 4F6E 4F6F 4F70
4F71 4F72 4F73 4F74 4F75 4F76 4F77 4F78 4F79 4F7A
4F7B 4F7C 4F7D 4F7E 4F7F 4F80 4F81 4F82 4F83 4F84
4F85 4F86 4F87 4F88 4F89 4F8A 4F8B 4F8C 4F8D 4F8E
4F8F 4F90 4F91 4F92 4F93 4F94 4F95 4F96 4F97 4F98
4F99 4F9A 4F9B 4F9C 4F9D 4F9E 4F9F 4FA0 4FA1 4FA2
4FA3 4FA4 4FA5 4FA6 4FA7 4FA8 4FA9 4FAA 4FAB 4FAC
4FAD 20204 2020C 20214 5152 5153 5154 5155 5156 20487
5169 5176 5177 5178 5190 519E 51BC 51BD 51BE 51BF
51C0 205C3 51ED 51EE 51EF 51FD 34E4 522E 522F 5230
5231 5232 5233 5234 5235 5236 5237 5238
5239 523A 523B 523C 523D 523E 523F 5240 5241 5242
2070E 5246 351A 52B5 52B6 52B7 52B8 52B9 52BA 52BB
52BC 52BD 52BE 52BF 52C6 530A 530B 530C 20916 5326
533C 5351 5352 5353 5354 5355 5356 5365 5366 5367
209E7 5376 5377 5378 5379 537A 5392 5393 5394 5395
20A50 53C0 53C1 53C2 53D4 53D5 53D6 53D7 53D8 3577
3578 545D 545E 545F 5460 5461 5462 5463 5464 5465
5466 5467 5468 5469 546A 546B 546C 546D 546E 546F
5471 5473 5474 5475 5476 5477 5478 5479 547A 547B
547C 547D 547E 547F 5480 5481 5482 5483 5484 5485
5486 5487 5488 5489 548A 548B 548C 548D 548E 548F
5490 5491 5492 5493 5494 5495 5496 5497 5498 5499
549A 549B 549C 549D 20C0B 20C0D 20C20 20C34
20C3A 20C3B 20C41 20C42 20C43 3620 56F6 56F7 56F8 56F9
56FA 56FB 56FC 56FD 56FE 3631 3632 3633 3634 3635
5761 5762 5763 5764 5765 5766 5767 5768 5769 576A
576B 576C 576D 576E 576F 5770 5771 5772 5773 5774
5775 5776 5777 5778 5779 577A 577B 577C 577D 577E
577F 5780 5781 5782 5783 5784 5785 5786 5787 5788
5789 578A 212A8 212A9 212B0 5907 590C 591C 591D 5943
5944 5945 5946 5947 5948 5949 594B 594C 594D 2160A
5954 36B0 36B1 36B5 36B9 36BC 36C1 59AC 59AD 59AE
59AF 59B0 59B1 59B2 59B3 59B4 59B5 59B6 59B7 59B8
59B9 59BA 59BB 59BC 59BD 59BE 59BF 59C0 59C1 59C2
59C3 59C4 59C5 59C6 59C7 59C8 59C9 59CA 59CB 59CC
59CD 59CE 59CF 59D0 59D1 59D2 59D3 59D4
59D6 59D7 216DF 216E6 216E7 216E8 216FA 216FB 216FC 216FE
3740 5B5F 5B60 5B61 5B62 5B63 5B64 5B65 5B66 5B67
21958 5B93 5B94 5B95 5B96 5B97 5B98 5B99 5B9A 5B9B
5B9C 5B9D 5B9E 5B9F 5BA0 5BA1 219DB 5C00 5C19 5C1A
5C2D 5C44 5C45 5C46 5C47 5C48 5C49 5C4A 37B9 37BE
5C9D 5C9E 5C9F 5CA0 5CA1 5CA2 5CA3 5CA4 5CA5 5CA6
5CA7 5CA8 5CA9 5CAA 5CAB 5CAC 5CAD 5CAE 5CAF 5CB0
5CB1 5CB2 5CB3 5CB4 5CB5 5CB6 5CB7 5CB8 5CB9 5CBA
5CBB 5CBC 5CBD 5CBE 5CBF 5CC0 5CC1 5CC2 5CC3 5CC4
5CC5 21D90 3830 5DF6 5E11 5E12 5E13 5E14 5E15 5E16
5E17 5E18 5E19 5E1A 5E1B 5E1C 2207E 5E77 5E78 5E98
5E95 5E96 5E97 5E99 5E9A 5E9C 5E9D 5E9E 5E9F 5EF6
38A0 5EF8 5EF9 5F06 5F21 5F22 5F23 5F24
5F25 5F26 5F27 5F28 5F29 5F2A 223D0 5F54 5F55 38CC
2248B 5F7C 5F7D 5F7E 5F7F 5F80 5F81 5F82 5F83 5F84
5FDD 5FDE 5FE0 5FE2 5FE5 5FE9 5FF5 5FFD 5FFF 6001
6002 3914 6007 6008 6009 600A 600B 600C 600D 600F
6010 6011 6013 6014 6015 6016 6017 6019 601A 601B
601C 601E 601F 6021 6022 6026 6027 6029 602A 602B
602C 602D 602E 602F 6030 6032 6033 6034 6035 6036
603A 603D 603E 603F 2261B 2261C 2262B 6214 6215 6216
6217 623D 623E 623F 6240 627F 39D4 39D5 39D9 39DA
39DC 39DD 39DE 39DF 62A6 62A7 62A8 62A9 62AA 62AB
62AC 62AD 62AE 62AF 62B0 62B1 62B2 62B3 62B4 62B5
62B6 62B7 62B8 62B9 62BA 62BB 62BC 62BD 62BE 62BF
62C0 62C1 62C2 62C3 62C4 62C5 62C6 62C7
62C8 62C9 62CA 62CB 62CC 62CD 62CE 62D0 62D1 62D2
62D3 62D4 62D5 62D6 62D7 62D8 62D9 62DA 62DB 62DD
62DE 62DF 62E0 62E1 62E2 62E3 62E4 62E5 62E6 62E7
62E8 62E9 22ACF 22AD5 3A81 653D 653E 6589 23044 65A6
65A7 65A8 65A9 65BA 65BB 65BC 230DA 65F9 65FA 65FB
65FC 65FD 65FE 65FF 6600 6601 6602 6603 6604 6605
6606 6607 6608 6609 660A 660B 660C 660D 660E 660F
6610 6611 6612 6613 6614 6615 6616 6617 6618 6619
66F6 3B33 670A 670B 670C 670D 3B47 3B48 3B49 3B4A
3B4B 3B4C 3B4D 3B4E 3B4F 3B50 676A 676B 676C 676D
676E 676F 6770 6771 6772 6773 6774 6775 6776 6777
6778 6779 677A 677B 677C 677D 677E 677F 6780 6782
6783 6784 6785 6786 6787 6788 6789 678A
678B 678C 678D 678E 678F 6790 6791 6792 6793 6794
6795 6796 6797 6798 6799 679A 679B 679C 679D 679E
679F 67A0 67A1 67A2 67A3 67A4 67A5 67A6 67A7 67A8
67A9 67AA 67AB 67AC 67AD 233DE 233E6 233F4 233F5 233F9
233FA 233FE 23400 67F9 3C20 6B23 6B25 6B26 6B27 6B66
6B67 6B68 6B69 6B7D 6B7E 6B7F 6B80 6B81 6BB4 6BD1
6BDE 6BDF 6C13 6C1B 6C1C 6C1D 6C6C 6C80 6C8A 6C93
6C9D 3CCB 3CCD 3CD1 6CAB 6CAC 6CAD 6CAE 6CB0 6CB1
6CB2 6CB3 6CB4 6CB5 6CB6 6CB7 6CB8 6CB9 6CBA 6CBB
6CBC 6CBD 6CBE 6CBF 6CC0 6CC1 6CC2 6CC3 6CC4 6CC5
6CC6 6CC7 6CC8 6CCA 6CCB 6CCC 6CCD 6CCE 6CCF 6CD0
6CD1 6CD2 6CD3 6CD4 6CD5 6CD6 6CD7 6CD8 6CD9 6CDB
6CDC 6CDD 6CDE 6CDF 6CE0 6CE1 6CE2 6CE3
6CE4 6CE5 6CE6 6CE7 6CE8 6CE9 6CEA 6CEB 6CEC 6CED
6CEE 6CEF 6CF1 6CF2 6CF3 6CF7 6CF8 6CF9 6CFA 6CFB
6CFC 6CFD 6CFE 23CB5 23CB7 23CC7 23CC8 23CC9 6D30 7087
7081 7082 7083 7084 7085 7086 7088 7089 708A 708B
708C 708D 708E 708F 7090 7091 7092 7093 7094 7095
7096 7097 7098 7099 709A 709B 709C 709D 709E 241A3
241A4 241A5 241AC 241B5 722C 722D 7238 7240 7248 3E2F
7265 7266 7267 7268 7269 726A 726B 726C 2462A 72C0
3E62 3E69 72C9 72CB 72CC 72CD 72CE 72CF 72D0 72D1
72D2 72D3 72D4 72D5 72D6 72D7 72D8 72D9 72DA 72DB
72DC 72DD 72DE 739C 739D 739E 739F 73A0 73A1 73A2
73A3 73A4 73A5 73A6 73A7 73A8 73A9 73AA 73AB 73AC
73AD 73AE 73AF 73B0 73B1 248FB 248FF 24900
24901 2490C 74DD 74E8 74E9 7519 753B 753D 753E 753F
7540 7541 7542 7543 7544 7545 758C 7598 7599 759A
759B 759C 759D 759E 759F 75A0 75A1 7677 7684 76AF
76C2 76F0 76F1 76F2 76F3 76F4 76F5 77E4 77E5 77F7
77F8 77F9 77FA 77FB 77FC 77FD 77FE 77FF 7800 7801
25425 793E 793F 7940 7941 7942 7943 79C4 79C5 79C6
79C7 79C8 79C9 79CA 4192 7A78 7A79 7A7A 7A7B 41C4
7ACE 7ACF 7AFA 7AFB 7C74 7C75 7C76 42B5 7CFC 7CFD
7CFE 7CFF 7EBF 7EC0 7EC1 7EC2 7EC3 7EC4 7EC5 7EC6
7EC7 7EC8 7EC9 7ECA 7ECB 7ECC 7ECD 7ECE 7ECF 7ED0
7F37 7F54 7F56 7F57 7F59 7F8B 7F8C 8005 8013 8035
8083 808F 43D9 80A1 80A2 80A3 80A4 80A5 80A6 80A7
80A8 80A9 80AA 80AB 80AC 80AD 80AE 80AF
80B0 80B1 80B2 80B3 80B4 80B5 80B6 80B7 80B8 80B9
80BA 80BB 80BC 80BD 80BE 80BF 80C0 80C1 81E4 81E5
81FD 81FE 820D 820E 820F 8220 8270 8298 8299 829A
829B 829C 829D 829E 829F 82A0 82A1 82A2 82A3 82A4
82A5 82A6 82A7 82A8 82A9 82AA 82AB 82AC 82AD 82AE
82AF 82B0 82B1 82B2 82B3 82B4 82B5 82B6 82B7 82B8
82B9 82BA 82BC 82BD 82BE 82C0 82C1 82C2 82C3 82C4
82C5 82C6 82C7 82C8 82C9 82CA 82CB 82CC 82CD 82CE
82CF 833E 26B13 26B15 26B28 82BF 864E 864F 866D 866E
866F 8670 8671 8672 275E0 8865 8868 89C4 89C5 8BD3
8BD4 8BD5 8BD6 8BD7 8BD8 8BD9 8BDA 8BDB 8BDC 8BDD
8BDE 8BDF 8BE0 8BE1 8BE2 8BE3 8BE4 8BE5 8BE6 8BE7
8BE8 8BE9 8C56 8D23 8D24 8D25 8D26 8D27
8D28 8D29 8D2A 8D2B 8D2C 8D2D 8D2E 8D2F 8ECB 8F6C
8F6D 8F6E 8F6F 8F70 8FCA 8FCB 8FCC 8FCD 8FCE 8FCF
8FD0 8FD1 8FD2 8FD3 8FD4 8FD5 8FD6 8FD7 8FD8 8FD9
8FDA 8FDB 8FDC 8FDD 8FDE 8FDF 8FEC FA24 28482 90AD
90AE 90AF 90B0 90B1 90B2 90B3 90B4 90B5 90B6 90B7
90B8 90B9 90BA 90BB 286BC 91C7 91D1 91D2 948D 948E
948F 9490 9491 9492 9493 9494 9495 9496 9497 9577
9578 9580 95F8 95F9 961C 9637 9638 9639 963A 963B
963C 963D 963E 963F 9640 9641 9642 9643 9644 9645
9646 9647 9648 9649 96B6 96B9 2ED7 96E8 9751 9752
975E 9763 9876 9877 996F 9970 9971 9972 9973 9974
9A75 9A76 9A77 9A78 9A79 9A7A 9A7B 9A7C 9A7D 9A7E
9A7F 9A80 9C7C 9E22 9E23 9E24 9EFE 9F21
9F7F
FDD0-2809
4E34 4E3E 4E57 3420 4E79 4E7A 4E7B 4E7C 4EAD 4EAE
4EAF 4EB0 4EB1 4EB2 4FAE 4FAF 4FB0 4FB1 4FB2 4FB3
4FB4 4FB5 4FB6 4FB7 4FB8 4FB9 4FBA 4FBB 4FBC 4FBD
4FBE 4FBF 4FC0 4FC1 4FC2 4FC3 4FC4 4FC5 4FC6 4FC7
4FC8 4FC9 4FCA 4FCB 4FCC 4FCD 4FCE 4FCF 4FD0 4FD1
4FD2 4FD3 4FD4 4FD5 4FD6 4FD7 4FD8 4FD9 4FDA 4FDB
4FDC 4FDD 4FDE 4FDF 4FE0 4FE1 4FE2 4FE3 4FE4 4FE5
4FE6 4FE7 4FE8 4FE9 4FEA 4FEB 4FEC 4FED 2025B 5157
5158 5159 2048E 516A 5179 517B 5191 5192 519F 51A0
51C1 51C2 51C3 205CA 205D0 205D5 51FE 5243 5244 5245
5247 5248 5249 524A 524B 524C 524D 524E 524F 5250
5251 52C0 52C1 52C2 52C3 52C4 52C5 52C7 52C8 52C9
52CA 52CB 530D 5327 5328 5329 533D 5357
5358 537B 537C 537D 5396 5397 5398 5399 539A 539B
53D9 53DA 53DB 53DC 53DD 5470 5472 3584 549E 549F
54A0 54A1 54A2 54A3 54A4 54A5 54A6 54A7 54A8 54A9
54AA 54AB 54AC 54AD 54AE 54AF 54B0 54B1 54B2 54B3
54B4 54B5 54B6 54B7 54B8 54B9 54BA 54BB 54BC 54BD
54BE 54BF 54C0 54C1 54C2 54C3 54C4 54C5 54C6 54C7
54C8 54C9 54CA 54CB 54CC 54CD 54CE 54CF 54D0 54D1
54D2 54D3 54D4 54D5 54D6 54D7 54D8 54D9 54DA 54DB
54DC 54DD 54DE 54DF 20C53 20C65 20C77 20C78 20C7C 20C8D
20C96 20C9C 3622 56FF 5700 211D9 3636 3637 3638 3639
363B 363E 578B 578C 578D 578E 578F 5790 5791 5792
5793 5794 5795 5796 5797 5798 5799 579A 579B 579C
579D 579E 579F 57A0 57A1 57A2 57A3 57A4
57A5 57A6 57A7 57A8 57A9 57AA 57AB 57AC 57AD 57AE
57AF 57B0 57B1 57B2 57B3 57B4 57B5 57CE 212E3 58F4
58F5 5908 5909 590D 369A 594E 594F 5950 5951 5952
5953 5955 5956 21613 21619 59D5 36C3 36C4 36C5 36C7
36C8 598D 59D8 59D9 59DA 59DB 59DC 59DD 59DE 59DF
59E0 59E1 59E2 59E3 59E4 59E5 59E6 59E7 59E8 59E9
59EA 59EB 59ED 59EE 59EF 59F0 59F1 59F2 59F3 59F4
59F5 59F6 59F7 59F8 59F9 59FA 59FB 59FC 59FD 59FE
59FF 5A00 5A01 5A02 5A03 5A05 5A06 5A07 5A08 2170D
21710 21726 5A0D 5B68 5B69 5B6A 5BA2 5BA3 5BA4 5BA5
5BA6 5BA8 5BA9 5BAA 5BAB 219F3 5C01 5C02 5C06 5C1B
5C1C 5C1D 5C2E 5C2F 5C4B 5C4C 5C4D 5C4E 5C4F 5CC6
5CC7 5CC8 5CC9 5CCA 5CCB 5CCC 5CCD 5CCE
5CCF 5CD0 5CD1 5CD2 5CD3 5CD4 5CD5 5CD6 5CD7 5CD8
5CD9 5CDA 5CDB 5CDC 5CDD 5CDE 5CDF 5CE0 5CE1 5CE2
5CE3 5CE4 5CE5 5CE6 5CE7 21DB6 21DBA 21DCA 21DD1 5CF8
5DEC 5DED 5DF7 5DF8 5DF9 5DFA 5DFB 5E1D 5E1E 5E1F
5E20 5E21 5E22 5E23 5E24 5E25 5E26 5E27 2209A 5E7D
5E9B 5EA0 5EA1 5EA2 5EA3 5EA4 5EA5 5EA6 22208 5EB0
5EFA 5EFB 5EFC 22321 22325 5F07 5F08 5F2B 5F2C 5F2D
5F2E 5F2F 223D7 5F56 5F65 5F66 5F85 5F86 5F87 5F88
5F89 5F8A 5F8B 5F8C 5F8D 5F94 600E 6012 6018 601D
6020 6023 6024 6025 6028 6031 6037 6038 6039 603B
603C 3922 3926 3927 392D 6040 6042 6043 6044 6045
6046 6047 6048 6049 604A 604C 604D 604E 6051 6052
6053 6054 6057 6058 605B 605C 605E 605F
6060 6061 6062 6064 6066 6068 606A 606B 606C 606E
6070 6071 6072 6078 6079 607A 607B 607C 607D 22668
6218 6241 6242 6243 39D8 62CF 62DC 39E1 39E2 39E5
39E6 62EA 62EB 62EC 62ED 62EE 62EF 62F0 62F1 62F4
62F5 62F6 62F7 62F8 62F9 62FA 62FB 62FC 62FD 62FE
6300 6301 6302 6303 6304 6305 6306 6307 6309 630A
630B 630C 630D 630E 630F 6311 6312 6313 6314 6315
6316 6317 6318 631C 631D 631E 631F 6320 6321 6322
6323 6324 6325 6326 6327 22AE6 22AE8 22B0E 22B22 6531
653F 3A83 3A84 3A85 6540 6541 6542 6543 6544 6545
3ABC 3ABD 65AA 65AB 3AC6 65BD 65BE 65BF 65C0 65E2
3ADE 3AE0 661A 661B 661C 661D 661E 661F 6620 6621
6622 6623 6624 6625 6626 6627 6628 6629
662A 662B 662C 662D 662E 662F 6630 6631 6632 6633
6634 6635 6636 6637 6638 6639 663A 663B 663C 663D
663E 663F 2317D 23182 66F7 670E 670F 6710 6711 3B51
3B52 3B53 3B54 3B55 3B56 3B57 67AE 67AF 67B0 67B1
67B2 67B3 67B4 67B5 67B6 67B7 67B8 67B9 67BA 67BB
67BC 67BE 67BF 67C0 67C1 67C2 67C3 67C4 67C5 67C6
67C7 67C8 67C9 67CA 67CB 67CC 67CD 67CE 67CF 67D0
67D1 67D2 67D3 67D4 67D5 67D6 67D7 67D8 67D9 67DA
67DB 67DC 67DD 67DE 67DF 67E0 67E2 67E3 67E4 67E5
67E6 67E7 67E8 67E9 67EA 67EB 67EC 67ED 67EE 67EF
67F0 67F1 67F2 67F3 67F5 67F6 67F7 67F8 67FA 67FB
67FC 67FD 67FE 67FF 6800 6801 6802 6803 6804 6805
6806 6807 6808 6809 680A 680B 680C 680D
680E 680F 6810 6811 6852 3C26 6B28 6B29 6B2A 3C54
6B6A 6B6B 6B82 6B83 6B84 6B85 6B86 6B87 6BB5 6BB6
6BD2 3C8B 6BD6 6BD7 6BD8 6BE0 6BE1 23B5A 6C1E 6C1F
6C20 6C21 6C22 6C97 6CAF 6CC9 6CF4 6CF6 3CD6 3CDC
6CDA 6CFF 6D00 6D01 6D02 6D03 6D04 6D05 6D06 6D07
6D08 6D09 6D0A 6D0B 6D0C 6D0E 6D0F 6D10 6D11 6D12
6D13 6D14 6D15 6D17 6D18 6D19 6D1A 6D1B 6D1D 6D1E
6D1F 6D20 6D21 6D22 6D23 6D24 6D25 6D26 6D27 6D28
6D29 6D2A 6D2B 6D2C 6D2D 6D2E 6D31 6D32 6D33 6D34
6D35 6D36 6D37 6D38 6D39 6D3A 6D3B 6D3C 6D3D 6D3E
6D3F 6D40 6D41 6D42 6D43 6D44 6D45 6D47 6D48 6D49
6D4A 6D4B 6D4C 6D4D 6D4E 6D4F 6D50 6D51 6D52 6D53
6D54 6D55 23CFC 23CFD 23CFE 23CFF 3DAD 709F
70A0 70A1 70A2 70A3 70A4 70A5 70A6 70A7 70A8 70A9
70AA 70AB 70AC 70AD 70AE 70AF 70B0 70B1 70B2 70B3
70B4 70B5 70B6 70B7 70B8 70B9 70BA 70BB 70BC 70BD
70BE 70BF 70C0 70C1 70C2 70C3 241CD 241E2 722E 722F
7230 24505 723C 7241 7249 724A 726D 726E 726F 7270
7271 7272 7273 7274 7275 72CA 72DF 72E0 72E1 72E2
72E3 72E4 72E5 72E6 72E7 72E8 72E9 72EA 72EB 72EC
72ED 72EE 72EF 72F0 72F1 72F2 2478F 7385 3EB1 73B2
73B3 73B4 73B5 73B6 73B7 73B8 73B9 73BB 73BD 73BE
73BF 73C0 73C1 73C2 73C3 73C4 73C5 73C6 73C7 73C8
73C9 73CA 73CB 73CC 73CD 73CE 73CF 73D0 73D1 24916
24917 24919 2492F 24933 24934 74EA 74EB 74EC 74ED 74EE
74EF 74F0 74F1 74F2 751A 7520 752D 752E
3F58 3F59 7546 7547 7548 7549 754A 754B 754C 754D
754E 754F 7550 7551 7552 7553 3F7C 75A2 75A3 75A4
75A5 75A6 75A7 75A8 75A9 75AA 75AB 75AC 75AD 75AE
75AF 75BA 7678 7679 767A 7685 7686 7687 7688 76C3
76C4 76C5 76C6 76C7 76C8 401D 76F6 76F7 76F8 76F9
76FA 76FB 76FC 76FD 76FE 76FF 7700 7701 7702 7703
7704 7705 7706 7707 7708 7709 770A 770B 770C 770D
2512B 77DC 77E6 77E7 77E8 409A 6CF5 7802 7803 7804
7805 7806 7807 7808 7809 780A 780B 780C 780D 780E
780F 7810 7811 7812 7813 7814 7815 7816 7817 7818
7819 781A 781B 781C 2542F 25430 4100 7944 7945 7946
7947 7948 7949 794A 794B 794C 794D 794E 89C6 25635
79B9 79BA 4132 79CB 79CC 79CD 79CE 79CF
79D0 79D1 79D2 79D3 79D4 79D5 79D6 79D7 25772 4195
7A7C 7A7D 7A7E 7A7F 7A80 7A81 7A82 7A83 25946 7AD0
7AD1 7AD2 7AD3 7AD4 7AD5 7AD6 7AD7 7AFC 7AFD 7AFE
7AFF 7B00 7B01 7B02 7B03 25AE9 427A 5A04 7C77 7C78
7C79 7C7A 7C7B 7C7C 7C7D 7C7E 7C7F 7C80 7C81 7C82
42B6 42B7 42B9 7D00 7D01 7D02 7D03 7D04 7D05 7D06
7D07 7D08 7D09 7ED1 7ED2 7ED3 7ED4 7ED5 7ED6 7ED7
7ED8 7ED9 7EDA 7EDB 7EDC 7EDD 7EDE 7EDF 4342 7F38
7F58 7F5A 7F8D 7F8E 7F8F 7F91 26351 7FBE 7FBF 263F5
263F8 8007 800D 800E 800F 8010 8011 8014 8036 8037
43DF 80C2 80C3 80C4 80C5 80C6 80C7 80C8 80C9 80CA
80CB 80CC 80CD 80CE 80CF 80D0 80D1 80D2 80D3 80D5
80D6 80D7 80D8 80D9 80DA 80DB 80DC 80DD
80DE 80DF 80E0 80E1 80E2 80E3 80E4 80E5 80E6 80E7
80E8 80E9 80EA 80EB 8109 81F4 81FF 8221 8222 8223
8224 8294 82D0 82D1 82D2 82D3 82D4 82D5 82D6 82D7
82D8 82D9 82DA 82DB 82DC 82DD 82DE 82DF 82E0 82E1
82E2 82E3 82E4 82E5 82E6 82E7 82E8 82E9 82EA 82EB
82EC 82ED 82EE 82EF 82F0 82F1 82F2 82F3 82F4 82F5
82F6 82F7 82F8 82F9 82FA 82FB 82FC 82FD 82FE 82FF
8300 8301 8302 8303 8304 8305 8306 8307 8309 830A
830B 830C 830D 830E 830F 8310 8311 8313 8314 8315
26B50 26B51 26B52 26B53 26B5B 833A 8650 8673 8674 8675
8676 8677 8678 8679 867A 867B 867C 867D 867E 867F
8680 8681 8682 8683 460F 8841 8842 884D 884E 275E4
8866 8867 8869 886A 886B 886C 27632 27639
8981 898C 89C7 89C8 89C9 89D3 89D4 8A02 8A03 8A04
8A05 8A06 8A07 8A08 8BEA 8BEB 8BEC 8BED 8BEE 8BEF
8BF0 8BF1 8BF2 8BF3 8BF4 8BF5 8BF6 8C9E 8C9F 8CA0
8D30 8D31 8D32 8D33 8D34 8D35 8D36 8D37 8D38 8D39
8D3A 8D3B 8D72 8D73 8D74 8D75 8DB4 8ECC 8ECD 8F71
8F72 8F73 8F74 8F75 8F76 8F77 8F78 8F79 8F7A 8F7B
8FE0 8FE1 8FE2 8FE3 8FE4 8FE5 8FE6 8FE7 8FE8 8FE9
8FEA 8FEB 8FED 8FEE 8FEF 8FF0 8FF1 8FF2 8FF3 90BC
90BD 90BE 90BF 90C0 90C1 90C2 90C3 90C4 90C5 90C6
90C7 90C8 90C9 90CA 90CB 90CD 90CE 90CF 90D0 90D1
90D3 90D5 286D8 286E6 90F1 914A 914B 91CD 91D3 91D4
9498 9499 949A 949B 949C 949D 949E 949F 94A0 94A1
94A2 94A3 94A4 94A5 94A6 94A7 94A8 94A9
94AA 94AB 94AC 94AD 94AE 94AF 9581 9582 95FA 95FB
95FC 95FD 95FE 95FF 9600 9601 9602 964A 964B 964C
964D 964E 964F 9650 9651 9652 9653 9654 9655 28E65
28E66 9762 9769 97CB 97E8 97ED 97F3 9801 9878 9879
987A 987B 98A8 98D0 98D1 98D2 98DB 2EDE 98DF 98E0
9975 9976 9977 9978 9979 997A 997B 997C 9996 29810
9999 9A81 9A82 9A83 9A84 9A85 9A86 9A87 9A88 9A89
2EE3 9CEC 9E25 9E26 9E27 9E28 9E29
FDD0-280A
2003E 4E35 4E58 4E7D 4EB3 3465 3466 4FEE 4FEF 4FF0
4FF1 4FF2 4FF3 4FF4 4FF5 4FF6 4FF7 4FF8 4FF9 4FFA
4FFB 4FFC 4FFD 4FFE 4FFF 5000 5001 5002 5003 5004
5005 5006 5007 5008 5009 500A 500B 500C 500D 500E
500F 5010 5011 5012 5013 5014 5015 5016 5017 5018
5019 501A 501B 501C 501D 501E 501F 5020 5021 5022
5023 5024 5025 5026 5027 5028 5029 502A 502B 502C
502D 502E 502F 5030 5031 5032 5033 5034 5035 5036
5037 5038 5039 503A 503B 503C 503D 503E 503F 20274
20275 20299 2029E 202A0 5056 515A 515B 20491 20492 517A
517C 5193 5194 51A1 51A2 51A3 51A4 51A5 51A6 51A7
2058E 51C4 51C5 51C6 51C7 51C8 51C9 51CA 51CB 51CC
51CD 51CE 205DF 205E0 20656 5252 5253 5254
5255 5256 5257 5258 5259 525A 525B 525C 525D 525E
525F 5260 5261 5262 5263 5264 5265 5266 5267 20731
52CC 52CD 52CE 52CF 52D0 52D1 20873 530E 532A 532B
537F 539C 539D 539E 539F 8652 53DE 53DF 3597 3598
54E0 54E1 54E2 54E3 54E4 54E5 54E6 54E7 54E8 54E9
54EA 54EB 54EC 54ED 54EE 54EF 54F0 54F1 54F2 54F3
54F4 54F5 54F6 54F7 54F8 54F9 54FA 54FB 54FC 54FD
54FE 54FF 5500 5501 5502 5503 5504 5505 5506 5507
5508 5509 550A 550B 550D 550E 550F 5510 5511 5512
5513 5514 5515 5516 5517 5518 5519 551A 551B 551C
551D 551E 551F 5520 5521 5522 5523 5524 5525 5526
5527 20CB5 20CB8 20CCF 20CD3 20CD4 20CD5 20CD6 20CDD 20CED
20CFF 20D15 3623 5701 5702 5703 5704 5705
5706 57B6 57B7 57B8 57B9 57BA 57BB 57BC 57BD 57BE
57BF 57C0 57C1 57C2 57C3 57C4 57C5 57C6 57C7 57C8
57C9 57CA 57CB 57CC 57CD 57CF 57D0 57D1 57D2 57D3
57D4 57D5 57D6 57D7 57D8 57D9 57DA 57DB FA0F 212FE
21302 21303 21304 21305 5832 58F6 590E 590F 591E 21596
594A 5957 5958 5959 595A 36CE 36D1 36D3 36D4 36D6
36DD 36E1 36E2 59EC 5A09 5A0A 5A0B 5A0C 5A0E 5A0F
5A10 5A11 5A12 5A13 5A14 5A15 5A16 5A17 5A18 5A19
5A1A 5A1B 5A1C 5A1D 5A1E 5A1F 5A20 5A21 5A22 5A23
5A24 5A25 5A26 5A27 5A28 5A29 5A2A 5A2D 5A2E 5A2F
5A30 5A31 5A32 5A33 5A34 2173A 2173B 2173C 21757 3743
5B6B 5B6C 5B6D 2196A 5BA7 5BAC 5BAD 5BAE 5BAF 5BB0
5BB1 5BB2 5BB3 5BB4 5BB5 5BB6 5BB7 5BB8
5BB9 5BBA 5BBB 5BBC 5BBD 5BBE 5C03 5C04 5C05 3797
5C50 5C51 5C52 5C53 5C54 5C55 5C56 5C57 5C58 21C70
5CE8 5CE9 5CEA 5CEB 5CEC 5CED 5CEE 5CEF 5CF0 5CF1
5CF2 5CF3 5CF4 5CF5 5CF6 5CF7 5CF9 5CFA 5CFB 5CFC
5CFD 5CFE 5CFF 5D00 5D01 5D02 5D03 5D04 5D05 21DEB
21DF9 382B 5DEE 5DFC 5E28 5E29 5E2A 5E2B 5E2C 5E2D
5E2E 5E2F 5E30 5E31 220C7 221C3 5EA7 5EA8 5EA9 5EAA
5EAB 5EAC 5EAD 5EAE 5EAF 5EFD 5F09 5F30 5F31 5F32
5F33 5F67 5F68 22491 5F8E 5F8F 5F90 5F91 5F92 5F93
5F95 224ED 3920 392B 6041 604B 604F 6050 6055 6056
6059 605A 605D 6063 6065 6067 6069 606D 606F 6073
6074 6075 6076 6077 3931 3933 3934 3936 3937 3938
3939 607E 6080 6081 6082 6083 6084 6085
6087 6088 608B 608C 608D 608E 608F 6091 6092 6093
6094 6095 6096 6097 6099 609A 609B 609C 609D 609E
609F 60A2 60A6 60A7 60A9 60AD 60AE 60AF 2267A 22696
22698 6219 6244 6245 6246 6247 39EC 62F2 62F3 62FF
6308 6310 6319 631A 631B 39F8 6328 6329 632A 632B
632C 632D 632E 632F 6330 6331 6333 6334 6335 6336
6337 6338 6339 633A 633C 633D 633E 633F 6340 6341
6342 6343 6344 6345 6346 6347 6348 6349 634A 634B
634C 634D 634E 634F 6350 6351 6352 6353 6354 6355
6356 6357 6358 6359 635A 635B 635C 635D 635E 635F
6360 6361 6362 6363 6364 22B3F 22B43 22B6A 63E4 3A87
3A88 3A89 6546 6547 6548 6549 654A 654B 654C 2304B
658A 658B 6599 659A 3AC9 65C1 65C2 65C3
65C4 65C5 65C6 65CA 6640 6641 6642 6643 6644 6645
6646 6647 6648 6649 664A 664B 664C 664D 664E 664F
6650 6651 6652 6653 6654 6655 6656 231A4 231A5 665F
6660 66F8 66FA 66FB 3B34 3B35 6712 6713 6714 6715
6717 67BD 67E1 67F4 3B58 3B59 3B5A 3B5B 3B5C 3B5D
3B5E 3B5F 3B60 3B61 3B62 3B63 3B64 3B65 3B66 3B67
6812 6813 6814 6815 6816 6817 6818 6819 681A 681B
681C 681D 681E 681F 6820 6821 6822 6823 6824 6825
6826 6827 6828 6829 682A 682B 682C 682D 682E 682F
6830 6831 6832 6833 6834 6835 6836 6837 6838 6839
683A 683B 683C 683D 683E 683F 6840 6841 6842 6843
6844 6845 6846 6847 6848 6849 684A 684B 684C 684D
684E 684F 6850 6851 6853 6854 6855 6856
6857 6858 6859 685A 685B 685C 685D 685E 685F 6860
6861 6862 6863 6864 6865 6866 6867 6868 6869 686A
2343F 23450 2346F 23472 6B2B 6B2C 6B2D 6B2E 6B2F 6B30
6B31 6B34 6B6C 6B6D 3C61 6B88 6B89 6B8A 6B8B 6BB7
6BD9 6BE2 6BE3 6BE4 6BE5 6BE6 6BE7 6BE8 6BE9 6BEA
6C23 6C24 6C25 6C26 6C27 6C28 6C29 6CF0 6D1C 6D2F
6D46 3CEF 6D0D 6D16 6D56 6D57 6D58 6D59 6D5A 6D5B
6D5C 6D5D 6D5E 6D5F 6D60 6D61 6D62 6D63 6D64 6D65
6D66 6D67 6D68 6D69 6D6A 6D6B 6D6C 6D6D 6D6E 6D6F
6D70 6D71 6D72 6D73 6D74 6D75 6D76 6D77 6D78 6D79
6D7A 6D7B 6D7C 6D7D 6D7E 6D7F 6D80 6D81 6D82 6D83
6D84 6D85 6D86 6D87 6D88 6D89 6D8A 6D8B 6D8C 6D8D
6D8F 6D90 6D91 6D92 6D93 6D94 6D95 6D96
6D97 6D98 6D9A 6D9B 6D9C 6D9D 6D9E 6D9F 6DA0 6DA1
6DA2 6DA3 6DA4 6DA5 6DA6 6DA7 6DA8 6DA9 23D40 23D5B
3DB4 70C4 70C5 70C6 70C7 70C8 70C9 70CA 70CB 70CC
70CD 70CE 70CF 70D0 70D1 70D2 70D3 70D4 70D5 70D6
70D7 70D8 70D9 70DA 70DB 70DC 70DD 70DE 70DF 70E0
70E1 70E2 70E3 70E4 70E5 70E6 70E7 70E8 70E9 70EA
70EB 70EC 70ED 70EE 241FC 2421B 7231 7239 7242 24578
3E20 7276 7277 7278 7279 727A 3E74 72F3 72F4 72F5
72F6 72F7 72F8 72F9 72FA 72FB 72FC 72FD 72FE 7300
7301 7302 7303 7386 3EB8 73BA 73BC 3EBF 3EC2 3EC7
73D2 73D3 73D4 73D5 73D6 73D7 73D8 73D9 73DA 73DB
73DC 73DD 73DE 73DF 73E0 73E1 73E2 73E3 73E4 73E5
73E6 73E7 73E8 73E9 73EA 73EB 73EC 73ED
73EE 73EF 73F0 73F1 73F2 7409 2493E 2493F 24940 24941
24942 24943 73F9 74DE 74DF 74F3 74F4 74F5 7521 7554
7555 7556 7557 7558 7559 755A 755B 755C 755D 755E
755F 758D 75B0 75B1 75B2 75B3 75B4 75B6 75B7 75B8
75B9 75BB 75BC 75BD 75BE 75BF 75C0 75C1 75C2 75C3
75C4 75C5 75C6 75C7 75C8 75C9 7560 768A 768B 768C
768D 24F5C 3FED 76B0 76B1 4000 76C9 76CA 76CB 76CC
76CD 76CE 76CF 76D0 76D1 25052 4026 770E 770F 7710
7711 7712 7713 7714 7715 7716 7717 7718 7719 771A
771B 771C 771D 771E 771F 7720 7721 7722 7723 7724
7727 7728 7729 772A 772B 772C 773F 25148 77DD 77E9
40A8 781D 781E 781F 7820 7821 7822 7823 7824 7825
7827 7828 7829 782A 782B 782C 782D 782E
782F 7830 7831 7832 7833 7834 7835 7836 7837 7838
7839 783A 783B 783C 783D 783E 783F 7840 7841 25446
2546C 2546E 4103 794F 7950 7951 7952 7953 7954 7955
7956 7957 7958 7959 795A 795B 795C 795D 795E 795F
7960 7962 25651 2573D 79D8 79D9 79DA 79DB 79DC 79DD
79DE 79DF 79E0 79E1 79E2 79E3 79E4 79E5 79E6 79E7
79E8 79E9 79EA 79EB 79EC 79ED 79EE 79EF 79F0 7A84
7A85 7A86 7A87 7A88 7A89 7A8A 7A8B 7A8C 7A8D 7A8E
25956 41CA 7AD8 7AD9 7ADA 7ADB 7ADC 7ADD 7ADE 41D7
41DB 7B05 7B06 7B07 7B08 7B09 7B0A 7B0B 7B0C 7B0D
7B0E 7B0F 7B10 7B11 7B12 7B13 7B14 7B15 7B04 7C83
7C84 7C85 7C86 7C87 7C88 7C89 7C8A 7C8B 7C8C 7C8D
7C8E 7C8F 7C90 7C91 42BC 7D0A 7D0B 7D0C
7D0D 7D0E 7D0F 7D10 7D11 7D12 7D13 7D14 7D15 7D16
7D17 7D18 7D19 7D1A 7D1B 7D1C 7D1D 7D1E 7D1F 7D20
7D21 7D22 7D23 7D24 7D25 7D26 7D27 7EE0 7EE1 7EE2
7EE3 7EE4 7EE5 7EE6 7EE7 7EE8 4343 7F39 7F3A 7F3C
7F5B 7F5C 7F5D 7F5E 7F5F 7F60 7F61 7F62 7F90 7F92
7F93 7F94 7F96 7F97 7F98 7F99 7FC0 7FC1 7FC2 7FC3
7FC4 7FC5 7FC6 26402 26410 26411 26412 8004 8006 800A
8015 8016 8017 8018 8019 8038 8039 803A 803B 803C
803D 803E 803F 8040 8041 8042 8081 8082 43ED 80ED
80EE 80EF 80F0 80F1 80F2 80F3 80F4 80F5 80F6 80F7
80F8 80F9 80FA 80FB 80FC 80FD 80FF 8100 8101 8102
8103 8104 8105 8106 8107 8108 810A 810B 810C 810D
810E 810F 8110 8111 8112 8113 266AF 266B1
266B5 81EC 81ED 26939 8200 8201 8210 4465 8225 8226
8227 8228 8229 822A 822B 822C 822D 822E 822F 8230
8231 8273 449F 44A0 44A2 82BB 8308 8316 8317 8318
8319 831A 831B 831C 831E 831F 8320 8321 8322 8324
8325 8326 8327 8328 8329 832A 832B 832C 832D 832E
832F 8330 8331 8332 8333 8334 8335 8336 8337 8338
8339 833C 833D 833F 8340 8341 8342 8343 8344 8345
8347 8348 8349 834A 834B 834C 834D 834E 834F 8350
8351 8352 8354 8355 8356 8357 8358 835A 835B 835C
835D 835E 835F 8360 8361 8362 8363 8364 8365 8366
8367 8368 8369 836A 836C 836D 836E 836F 26B75 26B82
26B96 26B97 26B9D 26BB3 8353 8651 8653 8654 8684 8685
8686 8687 8689 868A 868B 868C 868D 868E
868F 8690 8691 8692 8693 8694 8695 8696 8697 8698
8699 869A 869B 869C 869D 869E 869F 86A0 86A1 86A2
86A3 86A4 86A5 86A6 86A7 86A8 86A9 86AA 86AC 8843
8844 4615 884F 886D 886E 886F 8870 8871 8872 8873
8874 8875 8876 8877 8878 8879 887A 887B 887C 887D
887E 887F 8880 8881 8882 8883 8884 8885 8886 8887
27655 27656 27657 898D 898E 89CA 8A09 8A0A 8A0B 8A0C
8A0D 8A0E 8A0F 8A10 8A11 8A12 8A13 8A14 8A15 8A16
8A17 8A18 8A19 8A1A 27967 8BF7 8BF8 8BF9 8BFA 8BFB
8BFC 8BFD 8BFE 8BFF 8C00 8C01 8C02 8C03 8C04 8C05
8C06 8C07 8C08 8C09 8C0A 8C38 8C47 8C48 8C57 8C79
8C7A 8C7B 8CA1 8CA2 8CA3 8CA4 27D2F 8D3C 8D3D 8D3E
8D3F 8D40 8D41 8D42 8D43 8D44 8D45 8D46
4798 8D76 8D77 8D78 47D5 8DB5 8DB6 8DB7 8DB8 8EAC
8ECE 8ECF 8ED0 8ED1 8ED2 8ED3 8ED4 8ED5 8F7C 8F7D
8F7E 8F7F 8F80 8F81 8F82 8F83 8FB1 8FF4 8FF5 8FF6
8FF7 8FF8 8FF9 8FFA 8FFB 8FFC 8FFD 8FFE 8FFF 9000
9001 9002 9003 9004 9005 9006 9007 9008 9009 900A
9095 90D6 90D7 90D8 90D9 90DA 90DB 90DC 90DD 90DE
90DF 90E0 90E1 90E2 90E3 90E4 90E5 90E6 90E7 914C
914D 914E 914F 9150 9151 9152 91D5 91D6 91D7 91D8
91D9 91DA 91DB 91DC 91DD 91DE 91DF 91E0 91E1 91E2
94B0 94B1 94B2 94B3 94B4 94B5 94B6 94B7 94B8 94B9
94BA 94BB 94BC 94BD 94BE 94BF 94C0 94C1 94C2 94C3
94C4 94C5 94C6 94C7 94C8 94C9 94CA 94CB 94CC 94CD
94CE 9583 9584 9585 28CCA 28CCD 9603 9604
9605 9606 9656 9657 9658 9659 965B 965C 965D 965E
965F 9660 9661 9662 9663 9664 9665 9666 9667 9668
9669 965A 28E97 96BA 96BB 96BC 96BD 96BE 987C 987D
987E 987F 9880 9881 9882 9883 9884 98E2 98E3 98E4
997D 997E 997F 9980 9981 9982 99AC 9A8A 9A8B 9A8C
9A8D 9A8E 9A8F 9AA8 9AD8 9ADF 9B25 9B2F 9B32 9B3C
9C7D 9E2A 9E2B 9E2C 9E2D 9E2E 9E2F 9E30 9E31 9E32
9E33 9E34 9E35 9E36 9F80
FDD0-280B
3422 4E7E 4E7F 4E80 3464 5040 5041 5042 5043 5044
5045 5046 5047 5048 5049 504A 504B 504C 504D 504E
504F 5050 5051 5052 5053 5054 5055 5057 5058 5059
505A 505B 505C 505D 505E 505F 5060 5061 5062 5063
5064 5065 5066 5067 5069 506A 506B 506C 506D 506E
506F 5070 5071 5072 5073 5074 5075 5076 5077 5078
5079 507A 507B 507C 507D 507E 507F 202B7 202BF 202C0
202E5 515C 515D 515E 517D 34BC 5195 51A8 51CF 51D0
51D1 205EB 51F0 5268 526A 526B 526C 526D 526E 526F
5270 5271 5276 20779 3520 52D2 52D3 52D4 52D5 52D6
52D8 52D9 52DA 530F 5310 5313 352D 5318 5319 3531
532C 532D 532E 533E 533F 5340 5359 5368 537E 53A0
53A1 53A2 53A3 53A9 53C3 53C4 550C 35A1
35A5 35AD 5528 5529 552A 552B 552C 552D 552E 552F
5530 5531 5532 5533 5534 5535 5536 5537 5538 5539
553A 553B 553C 553D 553E 553F 5540 5541 5542 5543
5544 5545 5546 5547 5548 5549 554A 554B 554C 554D
554E 554F 5550 5551 5552 5553 5554 5555 5556 5557
5558 555A 555B 555C 555D 555E 5560 5561 5562 5563
5564 5565 5566 5567 5568 5569 556A 556C 556D 556E
556F 5570 5571 5572 5573 5574 5575 5576 5577 5578
5579 20D28 20D31 20D32 20D46 20D47 20D48 20D49 20D4C 20D4D
20D4E 20D6F 20D71 20D74 20D7C 20D7E 20D7F 20D96 20D9C 20DA7
20DB2 556B 55B6 5707 5708 5709 570A 570B 570F 3647
3648 3649 57DC 57DD 57DE 57DF 57E0 57E1 57E2 57E3
57E4 57E5 57E6 57E7 57E8 57E9 57EB 57EC
57ED 57EE 57EF 57F0 57F1 57F2 57F3 57F4 57F5 57F6
57F7 57F8 57F9 57FA 57FB 57FC 57FD 57FE 57FF 5800
5801 5802 5803 5804 5805 5806 5807 5808 5809 580A
580B 580C 580D 580E 580F 5810 5811 5812 5813 5814
5815 21336 2133A 57EA 5835 58F7 58F8 591F 5920 595B
595C 595D 595E 2163E 595F 5962 5A2B 5A3D 36E5 36E6
5A2C 5A35 5A36 5A37 5A38 5A39 5A3A 5A3B 5A3C 5A3E
5A3F 5A40 5A41 5A42 5A43 5A44 5A45 5A46 5A47 5A48
5A49 5A4A 5A4B 5A4C 5A4D 5A4E 5A4F 5A50 5A51 5A52
5A53 5A54 5A55 5A56 5A57 5A58 5A59 5A5A 5A5B 5A5C
5A5D 5A5E 5A5F 5A60 5A61 5A62 5A63 5A64 5A65 5A66
5A67 5A68 5A69 5A6A 5A6B 5A6C 5A6D 5A6E 5A6F 5A70
5A71 5A72 5A73 5A74 5A75 5A76 2176C 2176D
2176E 2176F 21770 21771 21773 21774 5A8E 5B6E 5B6F 5B70
5B72 5BBF 5BC0 5BC1 5BC2 5BC3 5BC4 5BC5 5BC6 5BC7
5BC8 5BC9 21A2D 21A34 5C07 5C08 5C09 5C59 5C5A 5C5B
5C5C 5C5D 5C60 5D06 5D07 5D08 5D09 5D0A 5D0B 5D0C
5D0D 5D0E 5D0F 5D10 5D11 5D12 5D13 5D14 5D15 5D16
5D17 5D18 5D19 5D1A 5D1B 5D1C 5D1D 5D1E 5D1F 5D20
5D21 5D22 5D23 5D24 5D25 5D26 5D27 5D28 5D29 5D2A
5D2B 5D2C 5D2D 5D2E 5D2F 5D30 21E1C 21E37 21E3D 5DE2
5DE3 3831 5E32 5E33 5E34 5E35 5E36 5E37 5E38 5E39
5E3A 5E3B 5E3C 5E3E 5EB1 5EB2 5EB3 5EB4 5EB5 5EB6
5EB7 5EB8 5EB9 5EBA 5EBB 5EBC 5EBE 5F34 5F35 5F36
5F37 5F38 5F39 223FA 5F57 5F69 5F6B 5F6C 5F96 5F97
5F98 5F99 5F9B 5F9C 5F9D 5F9E 5F9F 5FA0
5FA1 5FA2 5FA3 5FA4 22513 2251B 3930 3932 3935 393B
607F 6086 6089 608A 6090 6098 60A0 60A1 60A3 60A4
60A5 60A8 60AA 60AB 60AC 393F 394D 60B0 60B1 60B4
60B5 60B7 60B8 60BA 60BB 60BC 60BD 60BE 60BF 60C0
60C2 60C3 60C5 60C6 60C7 60C8 60CA 60CB 60CD 60CF
60D0 60D3 60D4 60D5 60D7 60D8 60D9 60DA 60DB 60DC
60DD 60DE 60DF 60E4 60E6 60E7 60E8 60EC 60ED 60EE
60EF 226F4 226F5 226F6 22712 22714 2271B 2271F 621A 621B
621C 621D 6248 6332 633B 39FB 39FE 3A00 3A01 3A02
3A03 3A04 3A06 6365 6366 6367 6368 6369 636A 636B
636C 636D 636E 636F 6370 6371 6372 6373 6374 6375
6376 6377 6378 6379 637A 637B 637C 637D 637E 637F
6380 6381 6382 6383 6384 6385 6386 6387
6388 6389 638A 638B 638D 638E 638F 6390 6391 6392
6393 6395 6396 6397 6398 6399 639A 639B 639C 639D
639E 639F 63A0 63A1 63A2 63A4 63A5 63A6 63A7 63A8
63A9 63AA 63AB 63AC 63AD 63AE 63AF 63B3 63B4 63B5
63B6 63B7 63B8 63B9 63BA 63BB 63BC 63BD 22BCA 22BCE
63B2 3A8A 3A8B 3A8C 3A8D 3A8E 555F 654D 654E 654F
6550 6551 6552 6553 6554 6555 6556 6557 6558 6559
655A 655B 655D 658D 658E 658F 659B 659C 65AC 65AD
3ACB 65C7 65C8 65C9 65CB 65CC 65CD 65CE 65CF 65E3
3AF0 3AF2 52D7 6657 6658 6659 665A 665B 665C 665D
665E 6661 6662 6663 6664 6665 6666 6667 6668 6669
66FD 231B3 231C8 231C9 66F9 66FC 3B36 3B37 6716 6718
6719 671A 671B 3B68 3B69 3B6A 3B6B 3B6C
3B6D 3B6E 3B6F 3B70 3B71 3B72 3B73 3B74 3B75 3B77
686B 686C 686D 686E 686F 6870 6871 6872 6873 6874
6875 6876 6877 6878 6879 687A 687B 687C 687D 687E
687F 6880 6881 6882 6883 6884 6885 6886 6887 6888
6889 688A 688B 688C 688D 688E 688F 6890 6891 6892
6893 6894 6895 6896 6897 6898 6899 689A 689B 689C
689D 689E 689F 68A0 68A1 68A2 68A3 68A4 68A5 68A6
68A7 68A8 68A9 68AA 68AB 68AC 68AD 68AE 68AF 68B0
68B1 68B2 68B3 68B5 68B6 68B7 68B8 68B9 68BA 68BB
68BC 68BD 68BE 68BF 68C0 68C1 68C2 6956 3C2F 6B32
6B33 6B35 6B36 6B37 6B38 3C62 6B8C 6B8D 6B8E 6B8F
6B90 6B91 6B92 6B93 6BB8 6BB9 6BBA 6BBB 6BEB 6BEC
6BED 6BEE 6C2A 6C2B 23C63 3CEB 6D8E 3D00
3D04 6D99 6DAA 6DAB 6DAC 6DAD 6DAE 6DAF 6DB0 6DB1
6DB2 6DB3 6DB4 6DB5 6DB6 6DB7 6DB8 6DB9 6DBA 6DBB
6DBC 6DBD 6DBE 6DBF 6DC0 6DC1 6DC2 6DC3 6DC4 6DC5
6DC6 6DC7 6DC8 6DC9 6DCA 6DCB 6DCC 6DCD 6DCE 6DCF
6DD0 6DD1 6DD2 6DD3 6DD4 6DD5 6DD6 6DD7 6DD8 6DD9
6DDA 6DDB 6DDC 6DDD 6DDE 6DDF 6DE0 6DE1 6DE2 6DE3
6DE4 6DE5 6DE6 6DE7 6DE8 6DE9 6DEA 6DEB 6DEC 6DED
6DEE 6DEF 6DF0 6DF1 6DF2 6DF3 6DF4 6DF5 6DF6 6DF7
6DF8 6DF9 6DFA 6DFB 6DFD 6DFF 6E00 6E01 6E02 6E04
6E05 6E06 6E07 6E08 6E09 6E0A 6E0B 6E0C 6E0D 6E0E
6E0F 6E10 6E11 6E12 6E13 6E14 6E15 6E16 6E17 6E1A
6E74 23D7E 23D8F 23DB6 23DB7 23DB8 23DB9 23DBA 23DBB 23DBC
23DBD 23DE3 23DF8 3DBF 70EF 70F0 70F1 70F2
70F3 70F4 70F5 70F6 70F7 70F8 70F9 70FA 70FC 70FD
70FE 70FF 7100 7101 7102 7103 7104 7105 7106 7107
7108 7109 710A 710B 710C 710D 710E 710F 7110 7111
7112 7113 7115 7116 7117 7118 2424B 24256 24259 7114
723D 3E3A 3E3C 3E3E 3E40 727B 727C 727D 727E 727F
7281 24665 72FF 7304 7305 7307 7308 7309 730A 730D
730E 730F 7310 7311 7313 7314 7315 7316 7317 7318
7319 731A 731B 731C 731D 731E 731F 7320 7321 732A
7387 7388 3ECA 3ECC 3ED0 73F3 73F4 73F5 73F6 73F8
73FA 73FB 73FC 73FD 73FE 73FF 7400 7401 7402 7403
7404 7405 7406 7407 7408 740A 740B 740C 740D 740E
740F 7410 7411 7412 7413 24962 24963 24974 24975 24976
3F0E 74E0 3F26 74F6 74F7 74F8 751B 751C
7522 7523 7561 7562 7563 7564 7565 7566 7567 7569
7570 75B5 75CA 75CB 75CC 75CD 75CE 75CF 75D0 75D1
75D2 75D3 75D4 75D5 75D6 7689 768E 768F 7690 7691
76B2 4001 4002 76D2 76D3 76D4 76D5 76D6 76D7 76D8
76DB 7725 7726 772D 772E 772F 7730 7731 7732 7733
7734 7735 7736 7737 7738 7739 773A 773B 773C 773D
773E 7741 2517D 2517E 7740 77EA 77EB 7826 7842 7843
7844 7845 7846 7847 7848 7849 784A 784B 784C 784D
784E 784F 7850 7851 7852 7853 7854 7855 7856 7857
7858 7859 785A 785B 7961 4104 7963 7964 7965 7967
7968 7969 796A 796B 796C 796D 796E 796F 8996 25683
79BB 413B 4141 4143 4145 79F1 79F2 79F3 79F4 79F5
79F6 79F7 79F8 79F9 79FA 79FB 79FC 79FD
79FE 7A06 7A8F 7A90 7A91 7A92 7A93 7A94 7A95 7A9A
7AE1 7B16 7B17 7B18 7B19 7B1A 7B1B 7B1C 7B1D 7B1E
7B1F 7B20 7B21 7B22 7B23 7B24 7B25 7B26 7B27 7B28
7B29 7B2A 7B2B 7B2C 7B2D 7B2E 7B2F 7B30 7B31 7B32
7B33 7B34 7B35 7B36 7B37 7B38 7B39 7B3A 7B3B 7B3C
7B3D 7B3E 7568 7C92 7C93 7C94 7C95 7C96 7C97 7C98
7C99 7C9A 7C9B 7C9C 7C9D 7CA3 25E49 7D28 7D29 7D2C
7D2D 7D2E 7D2F 7D30 7D31 7D32 7D33 7D34 7D35 7D36
7D37 7D38 7D39 7D3A 7D3B 7D3C 7D3D 7D3E 7D3F 7D40
7D41 7D42 7D43 7D44 7D45 7D46 7D47 7D48 7D49 7D4A
7D4B 7D4C 25FE1 25FE2 7EE9 7EEA 7EEB 7EEC 7EED 7EEE
7EEF 7EF0 7EF1 7EF2 7EF3 7EF4 7EF5 7EF6 7EF7 7EF8
7EF9 7EFA 7EFB 7EFC 7EFD 7EFE 7EFF 7F00
7F01 4344 4345 7F3B 7F3D 7F63 7F95 7F9A 7F9B 7F9C
7F9D 7F9E 7F9F 7FC7 7FC8 7FC9 7FCA 7FCB 7FCC 7FCD
7FCE 7FCF 7FD0 7FD1 7FD2 8008 8009 801A 801B 801C
801D 801E 801F 8043 8044 8045 8046 8047 8048 8049
804A 804B 804C 804D 80EC 43F2 8115 8116 8117 8118
8119 811A 811B 811C 811D 811E 811F 8121 8122 8123
8124 8125 8126 8127 8128 8129 812A 812B 812C 812D
812E 812F 8130 8131 8132 8133 8134 8135 8136 8137
8138 266DA 266E8 266FC 2690E 8202 8211 8232 8233 8234
8235 8236 8237 8238 8239 823A 823B 26A2D 26A2E 8274
8359 836B 831D 8323 8370 8371 8372 8373 8374 8375
8376 8377 8378 8379 837A 837B 837C 837D 837E 837F
8380 8381 8382 8383 8384 8385 8386 8387
8388 8389 838A 838B 838C 838D 838E 838F 8390 8391
8392 8393 8394 8395 8396 8397 8398 8399 839B 839C
839D 839E 839F 83A0 83A1 83A2 83A3 83A4 83A5 83A6
83A7 83A8 83A9 83AA 83AB 83AC 83AE 83AF 83B0 83B1
83B2 83B3 83B4 83B5 83B6 83B7 83B8 83B9 83BA 83BC
83BD 26BC0 26BF7 26C21 83AD 5F6A 8655 8656 8657 8658
8659 865A 86AB 86AD 86AE 86AF 86B0 86B1 86B2 86B3
86B4 86B5 86B6 86B7 86B8 86B9 86BA 86BB 86BC 86BD
86BE 86BF 86C0 86C1 86C2 86C3 86C4 86C5 86C6 86C7
86C8 86C9 86CA 86CB 86CC 86CD 86CE 86CF FA21 27280
27285 2728B 8845 8850 8851 8852 8853 8854 4626 8888
8889 888A 888B 888C 888D 888E 888F 8890 8891 8892
8893 8894 8895 8896 8897 8898 8899 889A
889B 889C 889D 889E 889F 88A0 88A1 88A2 88A3 88A4
88A5 88A6 88A7 88A8 88A9 88AA 88AB 88AC 88AD 88AE
88B0 88AF 8982 467A 898F 8990 8991 8992 8993 8994
89CB 89D5 89D6 89D7 89D8 89D9 278C8 46BC 8A1B 8A1C
8A1D 8A1E 8A1F 8A20 8A21 8A22 8A23 8A24 8A25 8A26
8A27 8A28 8A29 8A2A 8A2B 8A2C 8A2D 8A2E 8A2F 8A30
8A31 8A32 8A33 2797A 8C0B 8C0C 8C0D 8C0E 8C0F 8C10
8C11 8C12 8C13 8C14 8C15 8C16 8C17 8C18 8C19 8C1A
8C1B 8C1C 8C1D 8C1E 8C39 8C3A 8C3B 8C49 4746 4747
8C58 8C59 8C5A 8C5B 8C5C 8C5D 8C7C 8C7D 8CA5 8CA6
8CA7 8CA8 8CA9 8CAA 8CAB 8CAC 8CAD 8CAE 8D47 8D48
8D49 8D4A 8D65 8D66 8D67 479B 8D79 8D7A 8D7B 8D7C
8D7D 8D7E 8D7F FA23 8DB9 8DBA 8DBB 8DBD
8DBE 8DBF 8DC0 8DC1 8DC2 8DC3 8DC4 8EAD 8EAE 8EAF
28218 2821A 4846 4847 4848 4849 484A 8ED6 8ED7 8ED8
8ED9 8EDA 8EDB 8EDC 8EDD 8EDE 8EDF 8EE0 8EE1 8EE2
8EE3 282CD 8F84 8F85 8F86 489B 900B 900C 900D 900E
900F 9010 9011 9012 9013 9014 9015 9016 9017 9018
9019 901A 901B 901C 901D 901E 901F 9020 9021 9022
9023 9024 9025 9026 9027 28501 90AB 90D4 90E8 90E9
90EA 90EB 90EC 90ED 90EE 90EF 90F0 90F2 90F3 90F4
90F7 90F8 90FD 2870F 28713 48ED 9153 9154 9155 9156
9157 9158 9159 915A 915B 915C 915D 915E 28804 91C8
91CE 91E3 91E4 91E5 91E6 91E7 91E8 91E9 91EA 91EB
91EC 91ED 91EE 91EF 91F0 91F1 91F2 91F3 91F4 91F5
91F6 91F7 91F8 91F9 91FA 91FB 91FC 28948
28949 94CF 94D0 94D1 94D2 94D3 94D4 94D5 94D6 94D7
94D8 94D9 94DA 94DB 94DC 94DD 94DE 94DF 94E0 94E1
94E2 94E3 94E4 94E5 94E6 94E7 94E8 94E9 94EA 94EB
94EC 94ED 94EE 94EF 94F0 94F1 94F2 94F3 94F4 94F5
94F6 94F7 9579 957A 9586 9587 9588 9589 958A 28CD2
9607 9608 9609 960A 960B 960C 960D 960E 960F 9610
966A 966B 966C 966D 966E 966F 9670 9671 9673 9674
9675 9676 9677 9678 9679 967A 967C 28EAC 28EB2 28EB3
96BF 96C0 96E9 96EA 96EB 291D5 291EB 976A 7ADF 7AE0
9802 9803 9804 9885 9886 9887 9888 98E1 98E5 98E6
9983 9984 9985 9986 9997 9A90 9A91 9A92 9A93 9A94
9A95 9A96 9AD9 9B5A 9C7E 9CE5 9E37 9E38 9E39 9E3A
9E3B 9E3C 9E3D 9E3E 9E3F 9E75 9E7F 9EA5
9EB8 9EBB 9ED2 9F81 9F9A 9F9B
FDD0-280C
20046 4E81 4EB4 4EB5 5068 3473 347A 5080 5081 5082
5083 5084 5085 5086 5087 5088 5089 508A 508B 508C
508D 508E 508F 5090 5091 5092 5093 5094 5095 5096
5097 5098 5099 509A 509B 509C 509D 509E 509F 50A0
50A1 50A2 50A3 50A4 50A5 50A6 50A7 50A8 50A9 2030A
20325 515F 5160 204A3 6700 51D2 51D3 51D4 51D5 51D6
51F1 51F2 51FF 34FB 5269 5272 5273 5274 5275 52DB
52DC 52DD 52DE 5311 5312 20923 20954 3538 535A 53A4
53A5 53A6 53A7 53A8 53C5 35BF 35C1 35C4 35C5 35C7
35CA 35CE 5559 557A 557B 557C 557D 557E 557F 5580
5581 5582 5583 5584 5585 5586 5587 5588 5589 558A
558B 558C 558E 558F 5590 5591 5592 5593 5594 5595
5596 5597 5598 5599 559A 559B 559C 559D
559E 559F 55A0 55A1 55A2 55A3 55A4 55A5 55A6 55A7
55A8 55A9 55AA 55AB 55AC 55AD 55AE 55AF 55B0 55B1
55B2 55B3 55B4 55B5 55B7 55B8 55B9 55BA 55BB 55BC
55BD 55BE 5645 20DC8 20E04 20E09 20E0A 20E0D 20E0E 20E0F
20E10 20E11 20E16 20E1D 55DE 570C 570D 570E 5710 364E
5816 5817 5818 5819 581A 581B 581C 581D 581E 581F
5820 5821 5822 5823 5824 5825 5826 5827 5828 5829
582A 582B 582C 582D 582E 582F 5830 5831 5833 5834
5836 5837 5838 5839 583A 583B 583C 583E 583F 5840
5841 5842 5844 5845 5846 5847 5848 21375 21376 2138E
21398 2139C 58F9 58FA 58FB 5921 5960 5961 5963 5964
5965 36F5 3700 3701 3703 3704 5A77 5A78 5A79 5A7A
5A7B 5A7C 5A7D 5A7E 5A7F 5A80 5A81 5A82
5A83 5A84 5A85 5A86 5A87 5A88 5A89 5A8A 5A8B 5A8C
5A8D 5A8F 5A91 5A92 5A93 5A94 5A95 5A96 5A97 5A98
5A99 5A9A 5A9B 5A9C 5A9D 5A9E 5A9F 5AA0 5AA1 5AA2
5AA3 5AA4 5AA5 5AA6 5AA7 5AA8 5AA9 5AAA 5AAB 5AAC
5AAD 5AAE 5AAF 5ACF 217AB 217B0 217B1 217B2 217B3 217B4
217B5 217C3 217C7 217D9 217DA 217DB 217DC 217DF 5B71 2197C
21980 5B73 3762 5BCA 5BCB 5BCC 5BCD 5BCE 5BCF 5BD0
5BD1 5BD2 5BD3 5BD4 5BD5 5BEA 21A45 21A4B 5C0A 5C0B
5C0C 5C1E 5C30 5C31 5C5E 5C5F 5C61 21CA2 21CA5 5D31
5D32 5D33 5D34 5D35 5D36 5D37 5D38 5D39 5D3A 5D3B
5D3C 5D3D 5D3E 5D3F 5D40 5D41 5D42 5D43 5D44 5D45
5D46 5D47 5D48 5D49 5D4B 5D4C 5D4D 5D4E 5D4F 5D50
5D51 5D52 5D53 5D54 5D55 5D56 5D57 5D58
5D59 5D5A 5D5B 5D5C 5D5D FA11 21E89 21EA4 21EA8 5D6B
382D 5DEF 5DFD 22045 22049 5E3D 5E3F 5E40 5E41 5E42
5E43 5E44 5E45 5E46 5E47 5E49 220FC 5E48 5E7E 5EBD
5EBF 5EC0 5EC1 5EC2 5EC3 5ECA 5EC4 5F11 5F3A 5F3B
5F3C 5F3D 5F3E 5F58 5F6D 5F9A 5FA5 5FA6 5FA7 5FA8
5FA9 5FAA 5FAB 22530 60B2 60B3 60B6 60B9 60C1 60C4
60C9 60CC 60CE 60D1 60D2 60D6 60E0 60E1 60E2 60E3
60E5 60E9 60EA 60EB 3961 3962 3965 60F0 60F1 60F2
60F4 60F5 60F6 60F8 60FA 60FB 60FC 60FD 60FE 60FF
6100 6103 6104 6105 6107 6109 610A 610B 610C 610E
6110 6111 6112 6113 6114 6115 6116 6118 611C 611D
611E 6120 6121 6122 6123 6124 6125 6126 6168 2272A
22775 22781 39B8 621E 621F 6249 624A 638C
6394 63A3 63B0 63B1 3A17 3A18 63BE 63BF 63C0 63C1
63C2 63C3 63C4 63C6 63C7 63C8 63C9 63CA 63CB 63CC
63CD 63CE 63CF 63D0 63D1 63D2 63D3 63D4 63D5 63D6
63D7 63D8 63D9 63DA 63DB 63DC 63DD 63DE 63DF 63E0
63E1 63E2 63E3 63E5 63E6 63E8 63E9 63EA 63EC 63ED
63EE 63EF 63F0 63F2 63F3 63F4 63F5 63F6 63F7 63F8
63F9 63FA 63FB 63FC 63FD 63FE 63FF 6400 6401 6402
6403 6404 6405 6452 22C26 22C27 22C38 22C4C 22C51 22C55
6461 6532 3A8F 3A90 3A97 655C 655E 655F 6560 6561
6562 6563 6564 6565 6566 6567 6568 6569 656A 22F74
658C 6590 6591 3AB8 659D 2307D 2307E 659E 3AC0 65AE
65AF 65B1 65D0 65D1 23103 3AFB 3B00 666A 666B 666C
666D 666E 666F 6670 6671 6672 6673 6674
6675 6676 6677 6679 667A 667B 667C 667D 667E 667F
6680 6681 6682 6683 6691 231F7 231F8 231F9 2320F 66FE
66FF 6701 6702 3B38 671C 671D 671E 671F 68B4 3B76
3B78 3B79 3B7A 3B7B 3B7C 3B7D 3B7E 3B7F 3B80 68C3
68C4 68C5 68C6 68C7 68C8 68C9 68CA 68CB 68CC 68CD
68CE 68CF 68D0 68D1 68D2 68D3 68D4 68D5 68D6 68D7
68D8 68D9 68DA 68DB 68DC 68DD 68DE 68DF 68E0 68E1
68E2 68E3 68E4 68E5 68E6 68E7 68E8 68E9 68EA 68EB
68EC 68ED 68EE 68EF 68F0 68F1 68F2 68F3 68F4 68F5
68F6 68F7 68F8 68F9 68FA 68FB 68FC 68FD 68FE 68FF
6900 6901 6902 6903 6904 6905 6906 6907 6908 6909
690A 690B 690C 690D 690E 690F 6910 6911 6912 6913
6914 6915 6916 6917 6918 6919 691A 691B
691C 691D 691E 691F 6920 6921 6922 6923 6924 6925
6926 6927 6928 6929 692A 692B 692C 692D 692E 234E5
23519 23530 696E 6970 6B39 6B3A 6B3B 6B3C 6B3D 6B3E
6B3F 6B6E 6B6F 3C64 3C66 3C68 3C69 6B94 6B95 6B96
6B97 6B98 6B99 6B9A 6BBC 6BBD 6BBE 6BEF 6BF0 6BF1
6BF2 6BF3 6BF4 6BF5 6BF6 6C2C 6C2D 6C2E 6C2F 6C30
6DFC 6DFE 3D13 3D1D 3D20 3D22 6E03 6E18 6E19 6E1B
6E1C 6E1D 6E1E 6E1F 6E20 6E21 6E22 6E23 6E24 6E25
6E26 6E27 6E28 6E29 6E2A 6E2B 6E2C 6E2D 6E2E 6E2F
6E30 6E31 6E32 6E33 6E34 6E35 6E36 6E37 6E38 6E39
6E3A 6E3B 6E3C 6E3D 6E3E 6E3F 6E40 6E41 6E42 6E43
6E44 6E45 6E46 6E47 6E48 6E49 6E4A 6E4B 6E4C 6E4D
6E4E 6E4F 6E50 6E51 6E52 6E53 6E54 6E55
6E56 6E57 6E58 6E59 6E5A 6E5B 6E5C 6E5D 6E5E 6E5F
6E60 6E61 6E62 6E63 6E64 6E65 6E66 6E67 6E68 6E69
6E6A 6E6B 6E6D 6E6E 6E6F 6E70 6E71 6E72 6E73 6E75
6E76 6E77 6E78 6E79 6E7A 6E7B 6E7C 6E7D 6E7E 6E7F
6E80 6E81 6E82 6E83 6E84 6E85 6E86 6E87 6E88 6E89
6E8A 6E8B 6E8C 6ECB 6EDE 23E11 23E2C 23E2D 23E2E 23E2F
23E30 23E31 23E39 70FB 3DC6 3DC7 3DC9 3DCC 3DCD 7119
711A 711B 711C 711D 711E 711F 7120 7121 7122 7123
7124 7125 7126 7127 7128 7129 712A 712B 712C 712D
712E 712F 7130 7131 7132 7133 7134 7135 7136 7137
7138 7139 713A 713B 713C 713D 713E 713F 7140 716E
24276 24277 24278 24284 24293 24295 242A5 7232 724B 724C
724D 245C8 725A 3E43 7280 7282 7283 7284
7285 7286 7287 7288 7289 728A 728B 24674 7306 730B
730C 7312 732B 7322 7323 7324 7325 7326 7327 7328
7329 732C 732D 732F 7330 7331 7332 7333 7334 7335
7336 7338 7339 247E0 3ED1 3ED6 3EDA 3EDB 73F7 7414
7415 7416 7417 7418 7419 741A 741B 741C 741D 741F
7420 7421 7422 7423 7424 7425 7426 7428 7429 742A
742B 742C 742D 742E 742F 7430 7431 7432 7433 7434
7435 7436 7437 7438 7439 743A 743B 743C 2497B 2497F
24982 24988 24989 2498A 2498B 2498C 2498D 2498E 2498F 24994
249A4 249A7 249A9 249AB 249AC 249AD 74F9 74FA 74FB 74FC
7524 7525 7526 752F 24C09 756A 756B 756C 756D 756E
756F 7572 7573 7574 7571 758E 758F 3F93 3F98 75D7
75D8 75D9 75DA 75DB 75DC 75DD 75DE 75DF
75E0 75E1 75E2 75E3 75E4 75E5 75E6 75E7 75E8 75E9
75EA 75EB 24DB8 767B 767C 7692 7693 7694 7695 7696
76B3 76B4 4003 76D9 76DA 76DC 4039 4040 7742 7743
7744 7745 7746 7747 7748 7749 774A 774B 774C 774D
774E 774F 7750 7751 77DE 77DF 77EC 77ED 785C 785D
785E 785F 7860 7861 7862 7863 7864 7865 7866 7867
7868 7869 786A 786B 786C 786D 786E 786F 7870 7871
7872 7873 7874 7875 7876 7877 4109 7966 7970 7971
7972 7973 7974 7975 7976 7977 7978 7984 25695 79BC
79FF 7A00 7A01 7A02 7A03 7A04 7A05 7A07 7A08 7A09
7A0A 7A0B 7A0C 7A0D 7A0E 257C7 7A96 7A97 7A98 7A99
7A9B 7A9C 7A9D 7AE2 7AE3 7AE4 7AE5 7AE6 7AE7 41ED
41EE 41EF 7B3F 7B40 7B41 7B42 7B43 7B44
7B45 7B46 7B47 7B48 7B49 7B4A 7B4B 7B4C 7B4D 7B4E
7B4F 7B50 7B51 7B52 7B53 7B54 7B55 7B56 7B57 7B58
7B59 7B5A 7B5B 7B5C 7B5D 7B6C 4283 7C9E 7C9F 7CA0
7CA1 7CA2 7CA4 7CA5 7CA6 7CA7 7CA8 7CA9 7CAA 7CAB
7CAC 7CAD 7D2A 7D2B 7D4D 7D4E 7D4F 7D50 7D51 7D52
7D53 7D54 7D55 7D56 7D57 7D58 7D59 7D5A 7D5C 7D5D
7D5E 7D5F 7D60 7D61 7D62 7D63 7D64 7D65 7D66 7D67
7D68 7D69 7D6A 7D6B 7D6C 7D6D 7D6E 7D6F 7D70 7D71
7D72 7D73 7D74 7D75 7D76 7D77 7D7E 433B 7F02 7F03
7F04 7F05 7F06 7F07 7F08 7F09 7F0A 7F0B 7F0C 7F0D
7F0E 7F0F 7F10 7F11 7F12 7F13 7F14 7F15 7F16 7F17
7F18 7F3E 7F3F 7F40 7F64 7F65 7F66 436E 436F 7FA0
7FA1 7FA2 7FD3 7FD4 7FD5 7FD6 7FD7 7FD8
7FD9 7FDA 800B 8020 804E 804F 8050 8051 8052 8053
8054 8060 26572 80D4 80FE 8114 8120 3B39 4400 4401
4402 8139 813A 813B 813C 813D 813E 813F 8140 8141
8142 8143 8144 8145 8146 8147 8148 8149 814A 814B
814C 814D 814E 814F 8151 8152 8153 8154 8155 8156
8157 8158 8159 815A 26716 26741 8174 81E6 81EE 81EF
81F0 26911 81F5 81F6 81F7 81F8 81F9 8203 8204 8212
821C 823C 823D 823E 823F 26A34 8275 44B0 44B7 44C0
8312 833B 8346 839A 83BE 83BF 83C0 83C1 83C2 83C3
83C4 83C5 83C6 83C7 83C8 83C9 83CA 83CB 83CC 83CD
83CE 83CF 83D0 83D1 83D2 83D3 83D4 83D5 83D6 83D7
83D8 83DA 83DB 83DC 83DD 83DE 83DF 83E0 83E1 83E2
83E3 83E4 83E5 83E6 83E7 83E8 83E9 83EA
83EB 83EC 83ED 83EE 83EF 83F0 83F1 83F2 83F3 83F4
83F5 83F6 83F7 83F8 83F9 83FA 83FB 83FC 83FD 83FE
83FF 8400 8401 8402 8403 8404 8405 8406 8407 8408
8409 840A 840B 840C 840D 840E 840F 8410 8411 8412
8413 8414 8415 8416 8417 8418 8419 841A 841B 841C
841D 841E 841F 8420 8421 8422 8423 8424 8426 8427
8457 26C40 26C7E 26C7F 26C80 26C81 26C82 26CA4 26CB7 26CB8
26CBD 26CC0 8438 865B 865D 45AD 45AF 8688 86D0 86D1
86D2 86D3 86D4 86D5 86D7 86D8 86D9 86DA 86DB 86DC
86DD 86DE 86DF 86E0 86E1 86E2 86E3 86E4 86E5 86E6
86E7 86E8 86E9 86EA 86EB 86EC 86ED 86EE 86EF 86F0
86F1 86F2 86F3 86F4 272B2 272B6 8846 8847 8848 8849
8855 8856 8857 88B1 88B2 88B3 88B4 88B5
88B6 88B7 88B8 88B9 88BA 88BB 88BC 88BD 88BE 88BF
88C0 88C1 88C2 88C3 88C4 88C5 88C6 88C7 88C8 88C9
88D7 27694 8901 8983 8984 8999 8995 8997 8998 899A
89CC 89CD 89DA 89DB 89DD 89DE 46CF 46D0 8A34 8A35
8A36 8A37 8A38 8A39 8A3A 8A3B 8A3C 8A3D 8A40 8A41
8A42 8A43 8A44 8A45 8A46 8A47 8A48 8A49 8A4A 8A4B
8A4C 8A4D 8A4E 8A4F 8A50 8A51 8A52 8A53 8A54 8A55
8A56 8A57 8A58 8A59 8A5A 8A5B 8A5C 8A5D 8A5E 8A5F
8A60 279A0 8C1F 8C20 8C21 8C22 8C23 8C24 8C25 8C26
8C27 4748 8C5E 8C5F 8C60 8C61 8C7E 8C7F 8C80 8C81
8C82 8C83 8CAF 8CB0 8CB1 8CB3 8CB4 8CB5 8CB6 8CB7
8CB8 8CB9 8CBA 8CBB 8CBC 8CBD 8CBE 8CBF 8CC0 8CC1
27D53 27D54 8D4B 8D4C 8D4D 8D4E 8D4F 8D50
8D51 8D52 8D53 8D54 8D55 47A1 47A3 8D80 8D81 8D82
8D83 8D84 8D85 8D86 8D87 8D88 8D89 8D8A 8D8B 47ED
8DC5 8DC6 8DC7 8DC8 8DC9 8DCA 8DCB 8DCC 8DCD 8DCE
8DCF 8DD1 8DD2 8DD3 8DD4 8DD5 8DD6 8DD7 8DD8 8DD9
8DDA 8DDB 8DDC 8DDD 8DDE 8DF5 27FF9 28002 28009 8EB0
4852 8EE4 8EE5 8EE6 8EE7 8EE8 8EE9 8EEA 8EEB 8EEC
8EEE 8EEF 8EF0 8EF1 8EF2 8EF3 8EF4 8EF5 8EF6 8EF7
8EF8 8EF9 8EFA 8EFB 8EFC 8EFD 282E2 8F87 8F88 8F89
8F8A 8F8B 8F8C 8F8D 8F8E 8F9C 8F9D 28412 9028 9029
902A 902B 902C 902D 902E 902F 9030 9031 9032 9033
9034 9035 9036 9037 9038 9039 903A 903B 2853C 2853D
90F5 48D0 90F9 90FB 90FC 90FE 90FF 9100 9101 9102
9103 9104 9105 9106 9107 9108 9109 910A
912C 48F3 915F 9160 9161 9162 9163 9164 9165 91C9
91CA 91CF 491E 4920 91FD 91FE 91FF 9200 9201 9202
9203 9204 9205 9206 9207 9208 9209 920A 920B 920C
920D 920E 920F 9210 9211 9212 9213 9214 9215 9216
9217 9218 9219 921A 921B 921C 921D 921E 921F 9220
9221 9222 9223 9224 9225 9226 9227 9228 9229 922A
922B 922C 28956 28964 94F8 94F9 94FA 94FB 94FC 94FD
94FE 94FF 9500 9501 9502 9503 9504 9505 9506 9507
9508 9509 950A 950B 950C 950D 950E 950F 9510 9511
9512 9513 9514 9515 957B 958B 958C 958D 958E 958F
9590 9591 9592 9593 9594 9595 9596 9597 9611 9612
9613 9614 9615 9672 967B 967D 967E 967F 9680 9681
9682 9683 9684 9685 9686 9687 9688 9689
968A 968B 968C 968D 968E 9690 28ED9 28EE7 96C1 96C2
96C3 96C4 96C5 96C6 96C7 96C8 96EC 96ED 96EE 96EF
96F0 96F1 96F2 96F3 29088 2908B 9753 9754 975F 976B
976C 976D 976E 976F 9770 9771 97CC 97E9 9805 9806
9807 9808 9889 988A 988B 988C 988D 988E 988F 98A9
98AA 2959E 98D3 98E7 98E8 98E9 98EA 98EB 98ED 98EF
98F0 98F2 9987 9988 998A 998B 99AD 99AE 9A97 9A98
9A99 9A9A 9A9B 9AA9 9AE0 9C7F 9C80 9C81 9C82 9C83
9CE6 9E40 9E41 9E42 9E43 9E44 9E45 9E46 9E47 9E48
9EC3 9EC4 9ECD 9ED1 9EF9 9F0B 9F82
FDD0-280D
4E82 4E83 4E84 4EB6 4EB7 347D 347E 50AA 50AB 50AC
50AD 50AE 50AF 50B0 50B1 50B2 50B3 50B4 50B5 50B6
50B7 50B8 50B9 50BA 50BB 50BC 50BD 50BE 50BF 50C0
50C1 50C2 50C3 50C4 50C5 50C6 50C7 50C8 50C9 50CA
50CB 50CC 50CD 20341 20345 20346 20347 203B5 5161 517E
517F 51D7 5277 5278 5279 527A 527B 527C 527D 527E
527F 52DF 52E0 52E1 52E2 52E3 52E4 52E5 52E6 52E7
3532 3533 3534 532F 5380 5381 53AA 53AB 53AF 53E0
FA0E 35D2 35D6 35DB 35DD 35E0 558D 55BF 55C0 55C1
55C2 55C3 55C4 55C5 55C6 55C7 55C8 55C9 55CA 55CB
55CC 55CD 55CE 55CF 55D0 55D1 55D2 55D3 55D4 55D5
55D6 55D7 55D8 55D9 55DA 55DB 55DC 55DD 55DF 55E0
55E1 55E2 55E3 55E4 55E5 55E6 55E7 55E8
55E9 55EA 55EB 55EC 55ED 55EE 55EF 55F0 55F1 55F2
55F3 55F4 55F5 20E4C 20E6D 20E73 20E75 20E76 20E77 20E78
20E79 20E7A 20E7B 20E8C 20E96 20E98 20E9D 20EA2 20EAA 20EAB
20EAC 20EB6 5711 5712 5713 5714 5715 365F 583D 5843
5849 584A 584B 584C 584D 584E 584F 5850 5851 5852
5853 5854 5855 5856 5857 5858 5859 585A 585B 585C
585D 585E 585F 5860 5861 5862 5863 5864 5865 5866
5867 5868 5869 586A 586B 586C 586D 586E 586F 5870
5871 213C5 213C6 58FC 5966 5967 5968 21661 3708 370A
370D 5A90 5AB0 5AB1 5AB2 5AB3 5AB4 5AB5 5AB6 5AB7
5AB8 5AB9 5ABA 5ABB 5ABC 5ABD 5ABE 5ABF 5AC0 5AC1
5AC2 5AC3 5AC4 5AC5 5AC6 5AC7 5AC8 5AC9 5ACA 5ACB
5ACC 5ACD 5ACE 5AD0 5AD1 5AD2 5AD3 5AD4
217EF 217F5 217F6 217F8 217F9 217FA 217FB 217FC 5B74 21983
21988 5B76 3766 5BD6 5BD7 5BD8 5BD9 5BDA 5BDB 5BDC
5BDD 5C1F 5C20 5C32 5C33 5C34 21CAC 37F2 37F8 5D4A
5D5E 5D5F 5D60 5D61 5D62 5D63 5D64 5D65 5D66 5D67
5D68 5D69 5D6A 5D6C 5D6D 5D6E 5D6F 5D70 5D71 5D72
5D73 5D74 5D75 5D76 21EC8 21ED5 5DF0 5E4A 5E4B 5E4C
5E4D 5E4E 5E4F 5E79 5EC5 5EC6 5EC7 5EC8 5EC9 5ECB
5ECC 2227C 5F12 5F3F 5F40 5F41 5F42 5F59 5F5A 22465
5F6E 5FAC 5FAD 5FAE 5FAF 5FB0 3963 3964 3966 60F3
60F7 60F9 6101 6102 6106 6108 610D 610F 6117 6119
611A 611B 611F 6127 6129 612A 612B 612D 612E 612F
6130 6131 6132 6134 6135 6136 6137 6139 613A 613C
613D 613E 6140 6143 6144 6145 6146 6149
614A 614C 614D 614E 614F 6151 227B4 227B5 227CD 6226
6220 6221 6222 6223 6224 6225 63E7 63EB 63F1 3A20
3A23 3A26 3A29 3A2A 6406 6407 6408 6409 640A 640B
640C 640D 640E 640F 6410 6411 6412 6413 6414 6415
6416 6417 6418 6419 641A 641B 641C 641D 641E 641F
6420 6421 6422 6423 6424 6425 6426 6427 6428 6429
642A 642C 642D 642E 642F 6430 6432 6433 6435 6436
6437 6438 6439 643A 643C 643D 643E 6440 6441 6442
6443 6444 6445 6446 6447 6448 6449 644A 22C62 22C88
22C9B 22CA1 22CA9 22CB2 63C5 6431 656D 656B 656C 656E
656F 6570 6592 295B0 659F 65B0 65D2 65D3 65D4 65D5
65E4 6678 3B01 3B02 3B03 3B04 3B05 3B06 3B07 3B08
3B09 3B0A 3B0B 3B0C 6684 6685 6686 6687
6688 6689 668A 668B 668C 668D 668E 668F 6690 6692
6693 6694 6695 6696 6697 6698 6699 23225 2322F 23231
23232 23233 23234 6703 6720 6721 3B96 3B99 692F 6930
6931 6932 6933 6934 6935 6936 6937 6938 6939 693A
693B 693C 693D 693E 693F 6940 6941 6942 6943 6944
6945 6946 6947 6948 6949 694A 694B 694C 694D 694E
694F 6950 6951 6952 6953 6954 6955 6957 6958 6959
695A 695B 695C 695D 695E 695F 6960 6961 6962 6963
6964 6965 6966 6967 6968 6969 696A 696B 696C 696D
696F 6971 6972 6973 6974 6975 6976 6977 6978 6979
697A 697B 697C 697D 697E 697F 6980 6981 6982 6983
6984 6985 6986 6987 6988 6989 698B 698C 6994 6998
FA13 2355A 23567 23595 23599 2359C 3C3C 3C3E
6B40 6B41 6B42 6B43 6B44 6B45 6B46 6B47 6B48 6B71
6B72 6B73 3C6E 6B9B 6B9C 6BBF 6BC0 6BC1 6BC2 23AA7
6BD3 6BF7 6BF8 6BF9 6BFA 6BFB 6BFC 6BFD 6C31 6E6C
3D26 3D32 3D33 3D3B 6E8D 6E8E 6E8F 6E90 6E91 6E92
6E93 6E94 6E95 6E96 6E97 6E98 6E99 6E9A 6E9B 6E9C
6E9D 6E9E 6E9F 6EA0 6EA1 6EA2 6EA3 6EA4 6EA5 6EA6
6EA7 6EA8 6EA9 6EAA 6EAB 6EAC 6EAD 6EAE 6EAF 6EB0
6EB1 6EB2 6EB3 6EB4 6EB5 6EB6 6EB7 6EB8 6EB9 6EBA
6EBB 6EBC 6EBD 6EBE 6EBF 6EC0 6EC1 6EC2 6EC3 6EC4
6EC5 6EC6 6EC7 6EC8 6EC9 6ECA 6ECD 6ECF 6ED0 6ED1
6ED2 6ED3 6ED4 6ED6 6ED7 6ED8 6ED9 6EDB 6EDC 6EDD
6EDF 6EE0 6EE1 6EE2 6EE3 6EE4 6EE5 6EE6 6EE7 6EE8
6EE9 6EEA 6F13 23E88 23E89 23E8A 23E8B 23EB9
23EBF 6EDA 3BA1 3DD3 3DDB 7141 7142 7143 7144 7145
7146 7147 7148 7149 714A 714B 714C 714D 714E 714F
7150 7151 7152 7153 7154 7156 7157 7158 7159 715A
715C 715D 715E 715F 7160 7161 7162 7163 7164 7165
7166 7167 7168 7169 716A 716B 716C 716D 716F 7170
7171 7172 7173 7174 7175 7176 7177 7178 717A 242C1
242C9 242CA 242EE 242FA 24521 723A 7243 724E 724F 7250
7251 7252 3E48 728C 728D 728E 728F 7290 7291 24697
732E 7337 7341 733A 733B 733C 733D 733E 733F 7340
7342 7345 7346 7347 7348 7349 734A 24812 24823 3ED7
7427 3EDE 3EE1 3EE2 741E 743D 743E 743F 7440 7441
7442 7443 7444 7445 7446 7447 7448 7449 744A 744B
744C 744D 744E 744F 7450 7451 7452 7453
7454 7455 7456 7457 7458 7459 745A 745B 745C 745D
745E 745F 249B7 249B8 249B9 249BA 249BB 249C5 249D0 249DA
249DE 249DF 249E3 249E5 746F 74E1 74FD 74FE 74FF 7501
751D 751E 3F63 7575 7576 7577 7578 7579 757A 24C9E
24C9F 75EC 75ED 75EE 75EF 75F0 75F1 75F2 75F3 75F4
75F5 75F6 75F7 75F8 75F9 75FA 75FB 75FC 75FD 75FE
75FF 7600 7601 7602 7603 7604 7605 7606 24DEA 24DEB
760F 7610 7697 7698 7699 24F82 24F86 76B5 4004 76DD
76DE 76DF 4045 7752 7753 7754 7755 7756 7757 7758
7759 775A 775B 775C 775D 775E 775F 7760 7762 7763
7764 7765 7766 7767 7768 7769 776A 776B 776C 776D
251CD 251E3 251E6 251E7 7761 7779 77E0 77EE 40BB 40BF
7878 7879 787A 787B 787C 787D 787F 7880
7881 7882 7883 7884 7885 7886 7887 7888 7889 788A
788B 788C 788D 788E 788F 7890 7891 7892 7893 7894
7895 7896 7897 7898 7899 789A 789B 789C 78B0 410E
7979 797A 797B 797C 797D 797E 797F 7980 7981 7982
7983 7985 7986 79BD 842C 7A0F 7A10 7A11 7A12 7A13
7A14 7A15 7A16 7A17 7A18 7A19 7A1A 7A1B 7A1C 7A1D
7A1E 7A1F 7A20 7A21 7A22 7A23 7A24 7A25 257DF 257E0
257E1 7A9E 7A9F 7AA0 7AA1 7AA2 7AA3 7AA4 7AA5 7AA6
7AA7 259AC 41CF 7AE8 7AE9 7AEA 7AEB 25A95 41F8 41F9
41FB 41FC 41FD 41FE 41FF 4200 7B5E 7B5F 7B60 7B61
7B62 7B63 7B64 7B65 7B66 7B67 7B68 7B69 7B6A 7B6B
7B6D 7B6E 7B6F 7B70 7B71 7B72 7B73 7B74 7B76 7B77
7B78 7B79 7B7A 7B7B 7B7C 7B7D 7B7E 7B7F
7B80 7BC0 25B74 25B89 428C 7CAE 7CAF 7CB0 7CB1 7CB2
7CB3 7CB4 7CB5 7CC0 25E81 25E82 25E83 7D5B 7D78 7D79
7D7A 7D7B 7D7C 7D7D 7D7F 7D80 7D81 7D82 7D83 7D84
7D85 7D86 7D87 7D88 7D89 7D8A 7D8B 7D8C 7D8D 7D8E
7D8F 7D90 7D91 7D92 7D93 7D94 7D95 7D97 7D98 7D99
7D9A 7D9B 26029 7F19 7F1A 7F1B 7F1C 7F1D 7F1E 7F1F
7F20 7F21 7F22 7F23 7F24 7F67 7F68 7F69 7F6A 7F6B
7F6C 7F6D 7F6E 7F72 262D0 7FA3 7FA4 7FA5 7FA6 7FA7
7FA8 7FA9 7FAA 7FDB 7FDC 7FDD 8021 8022 8055 8056
8057 8058 8084 8085 8086 4413 5E50 815B 815C 815D
815E 815F 8160 8161 8162 8163 8164 8165 8166 8167
8168 8169 816A 816B 816C 816D 816E 816F 8170 8171
8172 8173 8175 8176 8177 8178 8179 817A
817B 817C 817D 817E 8205 821D 8240 8241 8242 8243
8244 8245 8246 8247 8248 8249 26A42 44C5 44CE 83BB
83D9 8425 8428 8429 842A 842B 842D 842E 842F 8430
8431 8432 8433 8434 8435 8436 8437 8439 843A 843B
843C 843D 843E 843F 8440 8441 8442 8443 8444 8445
8446 8447 8448 8449 844A 844B 844C 844D 844E 844F
8450 8451 8452 8453 8454 8455 8456 8458 8459 845A
845B 845C 845D 845E 845F 8460 8461 8462 8463 8464
8465 8466 8467 8468 8469 846A 846B 846C 846D 846E
846F 8470 8471 8472 8473 8474 8475 8476 8477 8478
8479 847A 847B 847C 847D 847E 847F 8480 8481 8482
8483 8484 8485 8486 8487 8488 8489 848B 848C 848D
848E 848F 26CC3 26CD1 26D22 26D23 26D24 26D25
26D26 26D27 26D28 26D29 26D2A 26D51 84C5 84C8 84F1 8507
865C 865E 865F 86D6 86F5 86F6 86F7 86F8 86F9 86FA
86FB 86FC 86FD 86FE 86FF 8700 8701 8702 8703 8704
8705 8706 8707 8708 8709 870A 870B 870C 870D 870E
870F 8710 8713 8714 8715 8716 8717 8746 272E6 874D
8858 8859 88CA 88CB 88CC 88CD 88CE 88CF 88D0 88D1
88D2 88D3 88D4 88D5 88D6 88D8 88D9 88DA 88DB 88DC
88DD 88DE 88DF 88E0 88E1 88E3 88E4 88E5 8985 4680
899B 899C 89CE 89DC 89DF 89E0 89E1 89E2 89E3 89E4
89E5 89E6 89E7 8A3E 8A3F 8A61 8A62 8A63 8A64 8A65
8A66 8A67 8A68 8A69 8A6A 8A6B 8A6C 8A6D 8A6E 8A6F
8A70 8A71 8A72 8A73 8A74 8A75 8A76 8A77 8A78 8A79
8A7A 8A7B 8A7C 8A7D 8A7E 8A7F 8A80 8A81
8A82 8A83 8A84 8A85 8A86 8A87 8A88 8A89 8A8A 8AA0
279DD 8C28 8C29 8C2A 8C2B 8C2C 8C3C 8C4A 8C4B 8C62
8C63 8C64 8C65 8C66 8C84 8C85 8C86 8C87 8C88 8C89
8C8A 8CB2 8CC2 8CC3 8CC4 8CC5 8CC6 8CC7 8CC8 8CC9
8CCA 8CCB 8CCC 8CCD 8CCE 27D66 27D73 27D84 8D56 8D57
8D68 8D69 8D6A 47A6 8D8C 8D8D 8D8E 8D8F 8D90 8D91
8D92 8D93 8D94 8DD0 8DBC 8DDF 8DE0 8DE1 8DE2 8DE3
8DE4 8DE5 8DE6 8DE7 8DE8 8DE9 8DEA 8DEB 8DEC 8DED
8DEE 8DEF 8DF0 8DF1 8DF2 8DF3 8DF4 8DF6 8DF7 8DF8
8DF9 8DFA 8DFB 2801E 28023 28024 4837 8EB1 8EB2 8EED
8EFE 8EFF 8F00 8F01 8F02 8F03 8F04 8F05 8F06 8F07
8F08 8F09 8F0A 8F0B 8F0C 8F8F 8F90 8F91 8F92 8F93
8F94 8F9E 8F9F 8FA0 8FB2 903C 903D 903E
903F 9040 9041 9042 9043 9044 9045 9046 9047 9048
9049 904A 904B 904C 904D 904E 904F 9050 9051 9052
9053 9054 9055 9056 9057 2856C 90CC 910B 910C 910D
910E 910F 9110 9111 9112 9113 9114 9115 9116 9117
9166 9167 9168 9169 916A 916B 916C 916D 916E 916F
9170 9171 2882B 922E 922F 9230 9231 9232 9233 9234
9235 9236 9237 9238 9239 923A 923B 923C 923D 923E
923F 9240 9241 9242 9243 9244 9245 9246 9247 9248
9249 924A 924B 924C 924D 924E 924F 9250 9251 9252
9253 9254 9255 9256 9257 9258 9259 925A 925B 925C
925D 925E 925F 9260 9261 9262 9263 9264 9265 9266
9267 9268 9269 926A 926B 926C 926D 926E 926F 9270
9271 9272 9273 9274 928F 28968 2896C 2897E
9516 9517 9518 9519 951A 951B 951C 951D 951E 951F
9520 9521 9522 9523 9524 9525 9526 9527 9528 9529
952A 952B 952C 952D 952E 952F 9530 9531 4989 9598
9599 959A 959B 959C 959D 959E 959F 95A0 9616 9617
9618 9619 968F 49DF 9691 9692 9693 9694 9695 9696
9697 9698 FA29 96C9 96CA 96CB 96CD 96CE 96CF 96F4
96F5 96F6 96F7 96F8 96F9 96FA 96FB 96FC 96FD 96FE
29093 9755 9756 9772 9773 9774 9775 9776 9777 9778
9779 97EA 97EB 97EE 97F4 97F5 9809 4AB4 980A 980B
980C 980D 980E 980F 9810 9811 9812 9813 9890 9891
9892 9893 9894 9895 9896 98AB 98AC 98D4 4B26 98EC
98EE 98F1 98F3 98F4 98F5 98F6 98F7 98F9 98FB 98FC
98FD 98FE 98FF 9989 998C 998D 998E 998F
9990 999A 99AF 99B0 99B1 99B2 99B3 99B4 99B5 9A9C
9A9D 9A9E 9A9F 9AAA 9AAB 9AAC 9AAD 9AAE 29A28 9AE1
9AE2 9B3D 9B5B 9B5C 9B5D 9B5E 9C84 9C85 9C86 9C87
9C88 9C89 9C8A 9C8B 9C8C 9C8D 9C8E 9C8F 9C90 9CE7
9CE8 9CE9 9CEA 9CEB 9CED 9CEE 9CEF 9CF0 9E49 9E4A
9E4B 9E4C 9E4D 9E4E 9E4F 9E50 9E51 9E52 9E53 9E54
9E80 9E81 9E82 2A2FF 9EFD 9F0C 9F0E 9F13 9F14 9F20
9F83 9F84 9F85 9F86
FDD0-280E
2004E 221A1 50CE 50CF 50D0 50D1 50D2 50D3 50D4 50D5
50D6 50D7 50D8 50D9 50DA 50DB 50DC 50DD 50DE 50DF
50E0 50E1 50E2 50E3 50E4 50E5 50E6 50E7 50E8 50E9
50EA 50EB 50EC 50ED 50EE 50EF 50F0 50F1 50F3 50F4
50F7 2037E 2037F 20380 203A0 203A7 50F2 5162 51A9 51D8
51F3 51F4 3500 3504 3506 5280 5281 5282 5283 5284
3522 52E8 52E9 52EA 52EB 52EC 52ED 3535 5330 5331
5332 3551 53AC 53AD 53AE 53B0 53C6 20B0D 3561 55F6
55F7 55F8 55F9 55FA 55FB 55FC 55FD 55FE 55FF 5600
5601 5602 5603 5604 5605 5606 5607 5608 5609 560A
560B 560C 560D 560E 560F 5610 5611 5612 5613 5614
5615 5616 5617 5618 5619 561A 561B 561C 561D 561E
5621 5622 5623 5624 5625 5626 5627 5651
20ED7 20ED8 20EDD 20EF8 20EF9 20EFA 20EFB 20F1D 20F26 20F2D
20F2E 20F30 20F31 20F3B 20F4C 561F 5628 5716 5717 5718
5719 3665 3666 5872 5873 5874 5875 5876 5877 5878
5879 587A 587B 587C 587D 587E 587F 5881 5882 5883
5884 5885 5886 5887 5888 5889 588A 588B 588C 588D
588E 588F 5890 5891 5892 5893 5894 5895 5896 5897
5898 5899 589A 589B 213ED 213FE 21413 21416 58AD 58FD
58FE 5910 5922 5923 5924 5925 5969 596A 596B 596C
371C 371E 3720 3722 5AD5 5AD6 5AD7 5AD8 5AD9 5ADA
5ADB 5ADC 5ADD 5ADE 5ADF 5AE0 5AE1 5AE2 5AE3 5AE4
5AE5 5AE6 5AE7 5AE8 5AE9 5AEA 5AEB 5AEC 5AED 5AEE
5AEF 5AF0 5AF1 5AF2 21820 21828 21829 2182A 2182D 21839
2183A 2183B 21840 21845 5AF3 5B75 5B77 21996
5BDE 5BDF 5BE0 5BE1 5BE2 5BE3 5BE4 5BE5 5BE6 5BE7
5BE8 5C0D 5C21 5C62 5C63 37FB 3800 3801 3804 5D77
5D78 5D79 5D7A 5D7B 5D7C 5D7D 5D7E 5D7F 5D80 5D81
5D82 5D83 5D84 5D85 5D86 5D87 5D88 5D89 5D8A 5D8B
5D8C 5D8D 5D8E 5E51 5E52 5E53 5E54 5E55 5E56 5E57
5E58 5E59 5E5B 2212A 5E63 5ECD 5ECE 5ECF 5ED0 5ED1
5ED2 5ED3 5ED4 5ED5 5ED6 5ED7 5ED8 5ED9 5EDC 5F0A
38C3 5F43 5F44 5F45 5F46 38D1 5F6F 5F70 5FB1 5FB3
5FB4 22554 5FB6 6128 612C 6133 6138 613B 613F 6141
6142 6147 6148 614B 6150 3980 6152 6153 6154 6156
6158 615A 615B 615E 615F 6160 6161 6162 6163 6165
6169 616A 616C 616F 6171 6172 6173 6174 6175 6177
617A 617B 617D 6180 6181 6186 6188 22803
6227 6228 6229 622A 622B 622C 642B 6434 643B 643F
3A2F 3A31 3A32 3A33 3A34 3A35 3A36 3A37 3A38 3A39
644B 644C 644D 644E 644F 6450 6451 6453 6454 6455
6457 6458 6459 645A 645B 645C 645D 645E 645F 6460
6462 6463 6464 6465 6466 6467 646A 646B 646C 646D
6471 6472 6473 6474 6475 6476 6477 6478 647A 647B
647C 647D 647E 647F 6481 6482 6484 6487 22CB7 22CC2
22CC6 22CC9 22D07 22D08 22D12 6456 64A6 6571 6572 6573
65A0 65A1 65B2 230B7 65D6 65D7 3B0D 3B0E 3B0F 3B10
669A 669B 669C 669D 669E 669F 66A0 66A1 66A2 66A3
66A4 66A5 66A6 66A7 66A8 23262 6704 6705 3B3A 3B3B
6722 3BBC 698A 698D 698E 698F 6990 6991 6992 6993
6995 6996 6997 6999 699A 699B 699C 699D
699E 699F 69A0 69A1 69A2 69A3 69A4 69A5 69A6 69A7
69A8 69A9 69AA 69AB 69AC 69AD 69AE 69AF 69B0 69B1
69B2 69B3 69B4 69B5 69B6 69B7 69B8 69B9 69BA 69BB
69BC 69BD 69BE 69BF 69C0 69C1 69C2 69C3 69C4 69C5
69C6 69C7 69C8 69C9 69CA 69CB 69CC 69CD 69CE 69CF
69D0 69D1 69D2 69D3 69D4 69D5 69D6 69D7 69D8 69D9
69DA 69DB 69DC 69DD 69DE 69DF 69E0 69E1 6A2E FA14
235CD 235CE 235CF 235F3 23600 6A03 6B49 6B4A 6B4B 6B4C
6B4D 6B70 6B74 6B9D 6B9E 6B9F 6BA0 6BA1 6BC3 6BC4
23AFA 6BFE 6C32 6C33 6ECE 3D3D 3D46 6ECC 6EEB 6EEC
6EED 6EEE 6EEF 6EF0 6EF1 6EF2 6EF3 6EF4 6EF5 6EF6
6EF7 6EF8 6EF9 6EFA 6EFB 6EFC 6EFD 6EFE 6EFF 6F01
6F02 6F03 6F04 6F05 6F06 6F07 6F08 6F09
6F0A 6F0C 6F0D 6F0E 6F0F 6F11 6F12 6F14 6F15 6F16
6F17 6F18 6F19 6F1A 6F1B 6F1C 6F1D 6F1E 6F1F 6F20
6F21 6F22 6F23 6F24 6F25 6F27 6F28 6F29 6F2A 6F2B
6F2C 6F2D 6F2E 6F2F 6F30 6F31 6F32 6F33 6F34 6F35
6F36 6F37 6F38 6F39 6F3A 6F3B 6F3C 6F3E 6F40 6F42
6F43 6F44 6F45 6F46 6F47 6F48 6F49 6F4A 6F4B 6F4C
6F4D 23ED7 23EF7 23EF8 23EF9 23EFA 23EFB 23EFC 23F35 23F41
6F4E 6F73 7155 715B 3DE7 3DE8 7179 717B 717C 717D
717E 717F 7180 7181 7182 7183 7184 7185 7186 7187
7188 7189 718A 718B 718C 718D 718E 718F 7190 7191
7192 7193 7194 7195 7196 7197 7198 7199 84B8 2430D
2431A 24334 24348 7233 723E 7244 3E22 7253 7254 7292
7293 7294 7295 7296 7297 7353 7343 7344
734C 734D 734F 7350 7351 7352 7354 7355 3EE7 3EE9
7460 7461 7462 7463 7464 7465 7466 7467 7468 746A
746B 746D 746E 7470 7471 7472 7473 7474 7475 7476
7477 7478 249EC 249ED 249F6 249F7 249F8 249F9 249FB 24A0E
24A12 24A13 7500 7502 7503 7504 7505 7506 24B6E 7527
757B 757C 757D 7590 7591 7607 7608 7609 760A 760B
760C 760D 760E 7611 7612 7613 7614 7615 7616 7617
7618 7627 76B6 76B7 76B8 76B9 76E0 76E1 76E2 76E3
4053 776E 776F 7770 7771 7772 7773 7774 7775 7776
7777 7778 777A 777B 777C 777D 777E 777F 7780 7781
7782 7783 7784 7785 7786 25220 25221 40C8 787E 789D
789E 789F 78A0 78A1 78A2 78A3 78A4 78A5 78A6 78A7
78A8 78A9 78AA 78AB 78AC 78AD 78AE 78AF
78B1 78B2 78B3 78B4 78B5 78B6 78B7 78B8 78B9 78C1
7987 7988 7989 798A 798B 798C 798D 798E 798F 7990
7991 7992 7993 7994 7995 7996 7997 7998 7999 4167
7A26 7A27 7A28 7A29 7A2A 7A2B 7A2C 7A2D 7A2E 7A2F
7A30 7A31 7A32 7A33 7A4A 7A35 7AA8 7AA9 7AAA 7AAB
7AAC 7AAD 259CC 7AEC 7AED 7AEF 7AF0 25A9C 7AEE 7B75
4201 4202 4203 4204 4205 4206 4207 4208 4209 420A
420B 420C 420D 7B81 7B82 7B83 7B84 7B85 7B86 7B87
7B88 7B89 7B8A 7B8B 7B8C 7B8D 7B8E 7B8F 7B90 7B91
7B92 7B93 7B94 7B95 7B96 7B97 7B98 7B99 7B9A 7B9B
7B9C 7B9D 7B9E 7B9F 7BA0 7BA1 7BA2 7BA3 7BA4 7BA5
7BA6 7BA7 7BA8 7BA9 7BAA 7BAB 25BB3 25BB4 25BC6 7BB8
7CB6 7CB7 7CB8 7CB9 7CBA 7CBB 7CBC 7CBD
7CBE 7CBF 7CC1 25EA6 25EBC 7D96 7DD0 42E8 42E9 42EC
42ED 42F1 7D9C 7D9D 7D9E 7D9F 7DA0 7DA1 7DA2 7DA3
7DA4 7DA5 7DA6 7DA7 7DA8 7DA9 7DAA 7DAB 7DAC 7DAD
7DAE 7DAF 7DB0 7DB1 7DB2 7DB3 7DB4 7DB5 7DB6 7DB7
7DB8 7DB9 7DBA 7DBB 7DBC 7DBD 7DBE 7DBF 7DC0 7DC1
7DC2 7DC3 7DC4 7DC5 7DC6 7DC7 7DC8 7DC9 7DCA 7DCB
7DCC 7DCD 7DCE 7DCF 7DD1 7DD2 7DD4 7DD5 26048 26064
7F25 7F26 7F27 7F28 7F29 7F2A 7F2B 7F41 7F42 7F6F
7F70 7F71 7F73 7F74 7FAB 7FDE 7FDF 7FE0 7FE1 7FE2
7FE3 7FE4 2644A 7FE5 8023 8024 8025 43BA 8059 805A
805B 805C 805D 805E 805F 8061 8062 8063 265A0 8087
8088 8150 4420 4425 4426 817F 8180 8181 8182 8183
8184 8185 8186 8187 8188 8189 818A 818B
818C 818D 818E 818F 8191 26799 267B3 267B4 267CC 81E7
81FA 8207 8213 8214 8215 821E 824A 824B 824C 824D
26A51 26A52 44DD 44DF 44E4 44E9 44EA 44EB 44EC 8490
8491 8492 8493 8494 8495 8496 8497 8498 8499 849A
849B 849C 849D 849E 849F 84A0 84A1 84A2 84A3 84A4
84A5 84A6 84A7 84A8 84A9 84AA 84AB 84AC 84AD 84AE
84AF 84B0 84B1 84B2 84B3 84B4 84B5 84B6 84B7 84B9
84BA 84BB 84BC 84BD 84BE 84BF 84C0 84C1 84C2 84C3
84C4 84C6 84C7 84C9 84CA 84CB 84CC 84CD 84CE 84CF
84D0 84D1 84D2 84D3 84D4 84D5 84D6 84D7 84D8 84D9
84DA 84DB 84DC 84DD 84DF 84E1 84E2 84E3 84E4 84E6
26D74 26DA0 26DA1 26DA2 26DA3 26DA4 26DA5 26DA6 26DA7 26DAE
26DDC 26DEA 26DEB 26DF0 84E5 8660 8661 271CD
8711 8712 872B 8718 8719 871A 871B 871C 871D 871E
871F 8720 8721 8722 8723 8724 8725 8726 8727 8728
8729 872A 872C 872D 872E 872F 8730 8731 8732 8733
8734 8735 8736 8737 8738 8739 873A 873B 873C 873D
873E 873F 8740 8741 8742 8743 8744 8745 8747 8748
8749 874A 874B 8755 876B 88E2 463B 88E7 88E8 88E9
88EA 88EB 88EC 88ED 88EE 88EF 88F0 88F1 88F2 88F3
88F4 88F5 88F6 88F7 88F8 88F9 88FA 88FB 88FC 88FD
88FE 88FF 8900 8902 8903 8904 891A 899D 899E 899F
89A0 89A1 89CF 89E8 89E9 89EA 89EB 8A8B 8A8C 8A8D
8A8E 8A8F 8A90 8A91 8A92 8A93 8A94 8A96 8A97 8A98
8A99 8A9A 8A9B 8A9C 8A9D 8A9E 8A9F 8AA1 8AA2 8AA3
8AA4 8AA5 8AA6 8AA7 8AA8 8AA9 8AAA 8AAB
8AAC 8AAD 8AAE 279FD 27A0A 27A0E 8C2D 8C2E 8C2F 8C30
8C31 8C32 8C3D 8C67 8C68 8C69 8C6A 8C8B 8C8C 8C8D
3562 8CCF 8CD0 8CD1 8CD2 8CD3 8CD4 8CD5 8CD6 8CD7
8CD8 27D8F 27D98 8D58 8D59 8D5A 8D5B 8D6B 8D95 8D96
8D97 8D98 8D99 8D9A 47F4 8DFC 8DFD 8DFE 8DFF 8E00
8E01 8E02 8E03 8E04 8E05 8E06 8E07 8E08 8E09 8E0A
8E0B 8E0C 8E0D 8E0E 28048 8EB3 8EB4 8EB5 8F0D 8F0E
8F0F 8F10 8F11 8F12 8F13 8F14 8F15 28306 28318 8F95
8F96 8F97 8FA1 8FA2 8FA3 9058 9059 905A 905B 905C
905D 905E 905F 9060 9061 9062 9063 9064 9065 90D2
9118 9119 911A 911B 911C 911D 911E 911F 9120 9121
9122 9123 9124 9125 48FA 9172 9173 9174 9175 9176
9177 9178 9179 917A 917B 917C 917D 917E
917F 922D 4924 4925 4926 492A 9275 9276 9277 9278
9279 927A 927B 927D 927E 927F 9280 9281 9282 9283
9284 9285 9286 9287 9288 9289 928A 928B 928C 928D
928E 9290 9291 9292 9293 9294 9295 9296 9297 9298
9299 929A 929B 929C 929D 929E 929F 92A0 92A1 92A2
92A3 92A4 92A5 92A6 92A7 92A8 92A9 92AA 92AB 92AC
92AD 92AE 92AF 92B0 92B1 28989 289A8 289AA 289AB 92EE
927C 9532 9533 9534 9535 9536 9537 9538 9539 953A
953B 953C 953D 953E 953F 9540 9541 9542 9543 9544
9545 95A1 95A2 95A3 95A4 95A5 95A6 95A7 95A8 95A9
95AA 961A 9699 969A 969B 969C 969D 969E 969F 96A0
96A1 96CC 96D0 96D1 96D2 28FC5 4A0F 96FF 9700 9701
290AF 290B0 290B1 9757 9758 9759 9764 977A
977B 977C 977D 977E 977F 9780 9781 9782 9783 9784
9785 9786 97CD 97CE 97EC 97F6 97F7 2941D 9819 4AB8
9814 9815 9816 9817 9818 981A 9897 4AFF 4B00 98AD
98AE 98AF 98B0 98B1 295B8 98D6 98D5 98D7 4B2C 98F8
9900 9901 9902 9903 9904 9905 9906 9907 9909 990C
990E 990F 9991 9992 4B6F 999B 999C 999D 4B7B 4B7E
99B6 99B7 99B8 99B9 99BA 99BB 99BC 99BD 99BE 99BF
99C0 99C1 99C2 99C3 99C4 99C5 99C6 99C7 9AA0 9AA1
9AA2 4BC8 9AAF 9AB0 9AB1 9ADA 9AE3 9AE4 9AE5 9AE6
9AE7 9AE8 9AE9 9AEA 29B05 29B0E 9B26 9B3E 9B3F 9B40
9B41 9B42 29CAD 9B5F 9B60 9B61 9B62 29D5A 9C91 9C92
9C93 9C94 9C95 9C96 9C97 9C98 9C99 9C9A 9C9B 9C9C
9C9D 9C9E 9C9F 9CF1 9CF2 9CF3 9CF4 9CF5
9CF6 9E55 9E56 9E57 9E59 9E5A 9E5B 9E5C 9EA7 9EBC
9EBD 9F3B 9F4A 9F87 9F88
FDD0-280F
3493 3496 3498 50F5 50F6 50F8 50F9 50FA 50FB 50FC
50FD 50FE 50FF 5100 5101 5102 5103 5104 5105 5106
5107 5108 5109 510A 510B 510C 510D 510E 510F 203C9
203CB 5130 51D9 51DA 51DB 51DC 20611 5285 5286 5287
5288 5289 528A 528B 528C 528D 528E 528F 52EE 52EF
52F0 52F1 52F2 5314 5333 53B1 53B2 20AB4 3559 53C7
5653 35F1 35F2 35F3 5620 5629 562A 562B 562C 562D
562E 562F 5630 5631 5632 5633 5634 5635 5636 5637
5638 5639 563A 563B 563C 563D 563E 563F 5640 5641
5642 5643 5644 5646 5647 5648 5649 564A 564B 564C
564D 564E 564F 5650 5652 5654 5656 5657 5658 5659
565A 565B 565C 565D 5674 20F64 20F8D 20F90 20FAD 20FB4
20FB5 20FB6 20FBC 21014 571A 5880 589C 589D
589E 589F 58A0 58A1 58A2 58A3 58A4 58A5 58A6 58A7
58A9 58AA 58AB 58AC 58AE 58AF 58B0 58B1 58B2 58B3
58B4 58B5 58B6 58B7 58B8 58B9 21424 2143F 21452 21454
21455 58FF 5900 21577 5926 596D 3723 3725 3726 5AF4
5AF5 5AF6 5AF7 5AF8 5AF9 5AFA 5AFB 5AFC 5AFD 5AFE
5AFF 5B00 5B01 5B02 5B03 5B04 5B05 5B06 5B07 5B08
5B09 5B0A 5B0B 5B0C 5B0D 5B0E 5B0F 21852 2185E 21861
21862 21863 21864 21877 2187B 21883 21884 21885 376F 5BE9
5BEB 5BEC 5BED 5BEE 5C0E 21B44 5C35 37A0 5C64 5C65
5C66 5C67 380F 5D8F 5D90 5D91 5D92 5D93 5D94 5D95
5D96 5D97 5D98 5D99 5D9A 5D9B 5D9C 5D9D 5D9E 5D9F
5DA0 5DA1 5DA2 5DA3 5DA4 5DA5 21F0F 21F15 5DE4 3861
5E5A 5E5C 5E5D 5E5E 5E5F 5E60 5E61 5E62
5E64 5E65 5E69 5EDA 5EDB 5EDD 5EDE 5EDF 5EE0 5EE1
5EE2 5EE3 5EE4 5F47 5F48 5F49 5F71 5FB2 5FB5 5FB7
5FB8 5FB9 5FBA 6155 6157 6159 615C 615D 6164 6166
6167 616B 616E 6170 6176 6178 6179 617C 617E 617F
6182 6183 6184 6185 6187 3989 398A 3992 616D 6189
618B 618D 618E 618F 6190 6192 6193 6194 6195 6198
619A 619B 619C 619E 619F 61A1 61A2 61A3 61A4 61A6
61A7 61AA 61AB 61AC 61AD 61AE 61AF 61B0 61B1 61B3
2285F 22860 22871 622D 622E 622F 3A3C 6468 6469 646E
646F 6470 6479 6480 6483 3A4B 6485 6486 6488 648A
648B 648C 648D 648E 648F 6490 6491 6492 6493 6494
6495 6496 6497 6498 6499 649A 649B 649C 649D 649E
649F 64A0 64A1 64A2 64A3 64A4 64A5 64A7
64A8 64A9 64AA 64AB 64AC 64AD 64AE 64AF 64B0 64B1
64B2 64B3 64B4 64B5 64B6 64B7 64B8 64B9 64BA 64C6
22D44 22D4C 22D67 6575 6576 6577 6578 6579 657A 657B
22FCC 65B3 230BC 3B11 3B12 3B13 3B14 3B15 3B16 66A9
66AA 66AB 66AC 66AD 66AE 66AF 66B0 66B1 66B2 66B3
66B4 66B5 66B6 66B7 23281 2328A 66BC 3B3C 3B3D 3B3E
81A4 69E9 3BBE 3BC2 3BC4 69E2 69E3 69E4 69E5 69E6
69E7 69E8 69EA 69EB 69EC 69ED 69EE 69EF 69F0 69F1
69F2 69F3 69F4 69F5 69F6 69F7 69F8 69F9 69FA 69FB
69FC 69FD 69FE 69FF 6A00 6A01 6A02 6A04 6A05 6A06
6A07 6A08 6A09 6A0A 6A0B 6A0C 6A0D 6A0E 6A0F 6A10
6A11 6A12 6A13 6A14 6A15 6A16 6A17 6A18 6A19 6A1A
6A1B 6A1C 6A1D 6A1E 6A1F 6A20 6A21 6A22
6A23 6A24 6A25 6A26 6A27 6A29 6A2A 6A2B 6A2C 6A2D
6A2F 6A30 6A31 6A65 2361A 2363C 23640 23659 2365F 23677
6B4E 6B4F 6B50 6B51 6B52 6B53 6B75 6B76 3C73 3C74
6BA2 6BA3 6BA4 6BA5 6BA6 6BC5 6BC6 6BFF 6C00 6C01
6C02 6ED5 6F00 6F10 6F26 6F3F 6F41 3D4C 3D4E 3D51
6F0B 6F3D 6F4F 6F50 6F51 6F52 6F53 6F54 6F55 6F56
6F57 6F58 6F59 6F5A 6F5B 6F5C 6F5D 6F5F 6F60 6F61
6F62 6F63 6F64 6F65 6F66 6F67 6F68 6F69 6F6A 6F6B
6F6C 6F6D 6F6E 6F6F 6F70 6F71 6F72 6F74 6F75 6F76
6F77 6F78 6F79 6F7A 6F7B 6F7C 6F7D 6F7E 6F7F 6F81
6F82 6F84 6F85 6F86 6F87 6F88 6F89 6F8A 6F8B 6F8C
6F8D 6F8E 6F8F 6F90 6F91 6F92 6F93 6F94 6F95 6F96
6F97 6F98 6F9A 6F9B 6F9C 6F9D 6FD0 23F4A
23F7F 23F80 23F81 23F82 23F8F 23FB4 23FB7 6FC6 71A6 3DEB
719A 719B 719C 719D 719E 719F 71A0 71A1 71A2 71A3
71A4 71A5 71A7 71A8 71A9 71AA 71AB 71AC 71AD 71AE
71AF 71B0 71B1 71B2 71B3 71B4 71B5 9ED9 24362 24363
24364 24365 2438C 24396 2439C 5655 7234 7245 7255 7256
7257 7298 7299 729A 729B 246D4 734E 734B 7356 7357
7358 7359 735A 735B 735C 735D 735E 735F 7360 7361
7362 7364 24882 7469 746C 3EEB 3EF0 3EF3 3EF4 7479
747A 747B 747C 747D 747E 7480 7481 7482 7483 7484
7485 7486 7487 7488 7489 748A 748B 748C 748E 7493
24A15 24A21 24A22 24A23 24A24 24A25 24A26 24A27 24A28 24A29
24A2A 24A3E 7507 7508 7509 3F53 24BF5 757E 757F 761F
3FB7 3FC0 7619 761A 761B 761C 761D 761E
7620 7621 7622 7623 7624 7625 7626 7628 7629 762A
762B 24E3B 24E50 3FE5 769A 769B 769C 769D 769E 24F97
24F9A 76BA 76E4 4057 7787 7788 7789 778A 778B 778C
778D 778E 778F 7790 7791 7792 7793 25250 78BA 78BB
78BC 78BD 78BE 78BF 78C0 78C2 78C3 78C4 78C5 78C6
78C7 78C8 78C9 78CA 78CB 78CC 78CD 78CE 78CF 78D0
78D1 78D2 78D3 78D4 78D5 78D7 78D8 78D9 78E4 25531
25535 2553F 799A 799B 799C 799D 799E 799F 79A0 79A1
79A2 79A3 256E3 416C 416E 4175 7A34 7A36 7A37 7A38
7A39 7A3A 7A3B 7A3C 7A3D 7A3E 7A3F 7A40 7A41 7A42
7A43 25857 7AAE 7AAF 7AB0 7AB1 7AB2 7AB3 7AB4 420E
420F 4210 4211 4212 4213 4214 4215 421A 421C 4220
4222 4223 4226 4229 7BAC 7BAD 7BAE 7BAF
7BB0 7BB1 7BB2 7BB3 7BB4 7BB5 7BB6 7BB7 7BB9 7BBA
7BBB 7BBC 7BBD 7BBE 7BBF 7BC1 7BC2 7BC3 7BC4 7BC5
7BC6 7BC7 7BC8 7BCA 7BCB 7BCC 7BCD 7BCE 7BCF 7BD0
7BD1 7BD2 7BD3 25BE4 25BE8 25C01 25C06 4294 7CC2 7CC3
7CC4 7CC5 7CC6 7CC7 7CC8 7CC9 7CCA 7CCB 7CCC 7CCD
7CCE 25ED7 25ED8 42F4 42FB 42FC 4300 4301 4304 7DD3
7DD6 7DD7 7DD8 7DD9 7DDA 7DDB 7DDC 7DDD 7DDE 7DDF
7DE0 7DE1 7DE2 7DE3 7DE4 7DE5 7DE6 7DE7 7DE8 7DE9
7DEA 7DEB 7DEC 7DED 7DEE 7DEF 7DF0 7DF1 7DF2 7DF3
7DF4 7DF5 7DF6 7DF7 7DF8 7DF9 7DFA 7DFB 7DFC 7DFD
7DFE 7DFF 7E00 7E01 7E02 7E03 7E04 7E05 7E06 7E07
26083 26097 260A4 260A5 433E 7F2C 7F2D 7F2E 7F2F 7F75
7F76 7F77 7F78 7FAC 7FAD 7FAE 7FAF 7FB0
7FE6 7FE7 7FE8 7FE9 7FEA 7FEB 7FEC 7FED 26469 43AC
8026 8027 8064 8065 8066 8067 8068 8069 806A 806B
265AD 442D 4433 4434 8192 8193 8194 8195 8196 8197
8198 8199 819A 819B 819C 819D 819E 819F 81A0 81A1
81A2 81A3 81F1 8216 8217 447A 824E 824F 8250 8251
8252 8253 8254 44F4 848A 84E0 84E7 84E8 84E9 84EA
84EB 84EC 84ED 84EE 84EF 84F0 84F2 84F3 84F4 84F5
84F6 84F7 84F8 84F9 84FA 84FB 84FC 84FD 84FE 84FF
8500 8501 8502 8503 8504 8505 8506 8508 8509 850A
850B 850C 850D 850E 850F 8510 8511 8512 8513 8514
8515 8516 8517 8518 8519 851A 851B 851C 851D 851E
851F 8520 8521 8522 8523 8524 8525 8526 8527 8528
8529 852A 852B 852C 852D 852E 852F 8530
8531 8532 8533 8534 8535 8536 8537 8538 8539 853A
853B 853C 26E00 26E05 26E07 26E12 26E42 26E43 26E44 26E45
26E6E 26E72 26E77 26E84 853D 854F 8662 874C 874E 874F
8750 8751 8752 8753 8754 8756 8757 8758 8759 875A
875B 875C 875D 875E 875F 8760 8761 8762 8763 8764
8765 8766 8767 8768 8769 876A 876C 876D 876E 876F
8770 8771 8772 8773 8774 8775 8776 8777 8778 877A
877B 877C 877D 877E 877F 8780 87E1 2739A 8782 885A
885B 885C 885D 275FD 4645 4646 88E6 8905 8906 8907
8908 8909 890A 890B 890C 890D 890E 890F 8910 8911
8912 8913 8914 8915 8916 8917 8918 8919 891B 891C
891D 2770F 89A2 89A3 89A4 89A5 27858 89A9 89D0 89D1
89EC 89ED 89EE 89EF 89F0 8A95 46F5 8AAF
8AB0 8AB1 8AB2 8AB3 8AB4 8AB5 8AB6 8AB7 8AB8 8AB9
8ABA 8ABB 8ABC 8ABD 8ABE 8ABF 8AC0 8AC1 8AC2 8AC3
8AC4 8AC5 8AC6 8AC7 8AC8 8AC9 8ACA 8ACB 8ACC 8ACD
8ACE 8ACF 8AD0 8AD1 8AD2 8AD3 8AD4 8AD5 8AD6 8AD7
8AD8 8AD9 8ADA 8AE9 27A3E 27A53 27A59 8ADB 8AF8 8C33
8C34 8C35 8C3E 8C4C 8C4D 8C4E 27BF4 8C6C 8C8E 8C8F
477C 8CD9 8CDA 8CDB 8CDC 8CDD 8CDE 8CDF 8CE0 8CE1
8CE2 8CE3 8CE4 8CE5 8CE6 8CE7 8CE8 8CE9 8CEA 8CEB
8CEC 8CED 8D5C 8D6D 47B6 8D9B 8D9C 8D9D 8D9E 8D9F
8DA0 8DA1 8DA2 8DA3 8DA4 4800 4801 480B 8E0F 8E10
8E11 8E12 8E13 8E14 8E15 8E16 8E17 8E18 8E19 8E1A
8E1B 8E1C 8E1D 8E1E 8E1F 8E20 8E21 8E22 8E23 8E24
8E25 8E26 8E27 8E28 8E29 8E2A 8E2C 8E2D
8E2E 8E2F 8E3A 28083 28090 8E2B 8E37 8EB6 8EB7 8EB8
8EB9 8EBA 8EBB 8EBC 28256 485D 8F16 8F17 8F18 8F19
8F1A 8F1B 8F1C 8F1D 8F1E 8F1F 8F20 8F21 8F22 8F23
8F24 8F25 8F26 8F27 8F28 8F29 8F2A 8F2B 8F2C 2832F
2833A 8F98 8FA4 8FB3 9066 9067 9068 9069 906A 906B
906C 906D 906E 906F 9070 9071 9073 9077 90F6 48DD
9126 9127 9129 912A 912B 912D 912E 912F 9130 9131
9132 9180 9181 9182 9183 9184 9185 9186 9187 9188
9189 918A 918B 918C 492D 92B2 92B3 92B4 92B5 92B6
92B7 92B8 92B9 92BA 92BB 92BC 92BD 92BE 92BF 92C0
92C1 92C2 92C3 92C4 92C5 92C6 92C7 92C8 92C9 92CA
92CC 92CD 92CE 92CF 92D0 92D1 92D2 92D3 92D4 92D5
92D6 92D7 92D8 92D9 92DA 92DB 92DC 92DD
92DE 92DF 92E0 92E1 92E2 92E3 92E4 92E5 92E6 92E7
92E8 92E9 92EA 92EB 92EC 92ED 92EF 92F0 92F1 92F2
92F3 92F4 92F5 92F6 FA27 289B8 289BC 289C0 289DC 289DE
289E1 289E3 289E4 9546 9547 9548 9549 954A 954B 954C
954D 954E 954F 9550 9551 9552 9553 9554 9555 957C
95AB 95AC 95AD 95AE 95AF 95B0 95B1 95B2 95B3 95B4
28D34 49E5 96A2 96A3 96A4 96A5 96D3 9702 9703 9704
9705 9706 9707 9708 9709 970A 290C0 975A 9760 9765
9787 9788 9789 978A 978B 978C 978D 978E 978F 9790
9791 9792 97CF 97D0 97D1 97EF 29420 981B 981C 981D
981E 981F 9820 9821 9822 9823 9826 9827 9828 9829
982A 982B 982C 9898 9899 989A 989B 989C 989D 98B2
98B3 98D8 4B37 98FA 9908 990A 990B 990D
9911 9912 9913 9914 9915 9916 9917 9918 9919 9993
9994 99C8 99C9 99CA 99CB 99CC 99CD 99CE 99CF 99D0
99D1 99D2 99D3 99D4 99D5 99D6 99D7 99D8 99D9 99DA
99DB 99DC 99DD 99DE 99DF 99E0 298A4 9AA3 9AB2 9AB3
9AB4 9AB5 9AB6 9AB7 9ADB 9AEB 9AEC 9AEE 9AEF 9AF0
9AF1 9AF2 9AF3 9AF4 9B27 4C20 9B43 9B44 9B45 9B46
4C3B 4C3E 9B63 9B64 9B65 9B66 9B67 9B68 9B69 9B6A
9B6B 9B6C 9B6D 9B6E 9B6F 9B70 9B71 9B72 9B73 9B74
9B75 9B76 9B77 9B78 9B79 29D7C 9CA0 9CA1 9CA2 9CA3
9CA4 9CA5 9CA6 9CA7 9CA8 9CA9 9CAA 9CAB 9CAC 4CAE
4CB0 4CB7 9CF7 9CF8 9CF9 9CFA 9CFB 9CFC 9CFD 9CFE
9CFF 9D00 9D01 9D02 9D03 9D04 9D05 9D06 9D07 9D08
9D09 9D0B 9D0C 9D0D 9D0E 29FB7 9E58 9E5D
9E5E 9E5F 9E60 9E61 9E62 9E63 9E64 9E76 9E83 9E84
2A293 9EA8 9EA9 9EAA 9EAB 9EB9 9EBE 9ECE 58A8 9ED3
9F0F 9F10 9F11 9F51 9F52 9F89 9F8A
FDD0-2810
4EB8 5110 5111 5112 5113 5114 5115 5116 5117 5118
5119 511A 511B 511C 511D 511E 512B 203F5 203FC 5163
204FC 5180 51AA 51DD 51DE 20615 5290 5291 5292 5293
5294 52F3 5334 53E1 35FB 35FE 3600 3601 3602 3603
3604 565E 565F 5660 5661 5662 5663 5664 5665 5666
5667 5668 5669 566A 566B 566C 566D 566E 566F 5670
5671 5672 5673 5675 5676 5677 5678 5679 567A 567B
567C 20FDF 20FEA 20FEB 20FEC 20FED 2101D 2101E 571B 571C
58BA 58BB 58BC 58BD 58BE 58BF 58C0 58C1 58C2 58C3
58C4 58C5 58C6 58C7 58C8 58C9 58CA 58CB 58CC 5901
596E 596F 372B 372C 372D 5B10 5B11 5B12 5B13 5B14
5B15 5B16 5B17 5B18 5B19 5B1A 5B1B 5B1C 5B1D 5B1E
5B1F 5B20 5B21 5B22 5B34 2189E 2189F 218A0
218A1 218A2 218BE 218BF 5B28 5B78 5B79 5BEF 5BF0 5DA6
5DA7 5DA8 5DA9 5DAA 5DAB 5DAC 5DAD 5DAE 5DAF 5DB0
5DB1 5DB2 5DB3 5DB4 5DB5 5DB6 3862 3863 3864 5E66
5E67 5E68 2215B 5E6F 5EE5 5EE6 5EE7 5EE8 5EE9 5EEA
5F4A 5F4B 5F5B 5F5C 22471 224B0 5FBB 5FBC 618A 618C
6191 6196 6197 6199 619D 61A0 61A5 61A8 61A9 61B2
3999 61B4 61B6 61B7 61B8 61B9 61BA 61BD 61BE 61BF
61C0 61C1 61C4 61C5 61C6 61C8 61CA 61CC 61CD 61CE
61CF 61D0 61D2 61D3 61D4 228AD 228C1 61BB 6231 6230
6489 3A52 3A54 3A57 64BB 64BC 64BD 64BE 64BF 64C0
64C1 64C2 64C3 64C4 64C5 64C7 64C8 64C9 64CB 64CC
64CD 64CF 64D0 64D1 64D2 64D3 64D4 64D5 64D6 64D7
64D9 64DA 64DB 64DC 64DD 64DE 22D8D 22D95
22DA0 22DA3 22DA4 22DB7 64F3 6533 6574 657C 657D 657E
657F 22FE3 6593 65A2 65B4 65D8 65D9 3B17 3B18 3B19
3B1A 3B1B 3B1C 3B1D 3B1E 3B1F 66B8 66B9 66BA 66BB
66BD 66BE 66BF 66C0 66C1 66C2 66C3 66C4 66C5 66C6
66C7 66C8 66C9 66CA 66CB 66CC 66CD 23289 232AB 232AC
232AD 66CF 3B31 6706 3B3F 6723 6724 6725 6A28 6A74
3BD7 3BDD 6A32 6A33 6A34 6A35 6A36 6A37 6A38 6A39
6A3A 6A3B 6A3C 6A3D 6A3E 6A3F 6A40 6A41 6A42 6A43
6A44 6A45 6A46 6A47 6A48 6A49 6A4A 6A4B 6A4C 6A4D
6A4E 6A4F 6A50 6A51 6A52 6A53 6A54 6A55 6A56 6A57
6A58 6A59 6A5A 6A5B 6A5C 6A5D 6A5E 6A5F 6A60 6A61
6A62 6A63 6A64 6A66 6A67 6A68 6A69 6A6A 6A6B 6A6C
6A6D 6A6E 6A6F 6A70 6A71 6A72 6A73 6A75
6A76 6A77 6A78 6A79 6A7A 6A7B 6A7C 236A6 236AD 236BA
236DF 236EE 6B54 6B55 6B56 6B57 6B58 6B59 6B5A 6B77
6BA7 6BA8 6BA9 6BAA 6BAB 6BC7 6BC8 6C03 6C04 6C05
6C06 6C07 6F5E 6F83 3D5F 3D62 3D65 3D69 3D6A 6F99
6F9E 6F9F 6FA0 6FA1 6FA2 6FA3 6FA4 6FA5 6FA6 6FA7
6FA8 6FAA 6FAB 6FAC 6FAD 6FAE 6FAF 6FB0 6FB1 6FB2
6FB3 6FB4 6FB5 6FB6 6FB7 6FB8 6FB9 6FBA 6FBB 6FBC
6FBD 6FBE 6FBF 6FC0 6FC1 6FC2 6FC3 6FC4 6FC5 6FC7
6FC8 6FC9 6FCA 6FCB 6FCD 6FCE 6FCF 6FD1 6FD2 6FD3
6FD6 23FC5 23FEB 23FEC 23FED 23FEE 23FEF 23FF0 24011 7004
3DF3 3DF7 3DFC 3DFD 71B6 71B7 71B8 71B9 71BA 71BB
71BC 71BD 71BE 71BF 71C0 71C1 71C2 71C3 71C4 71C5
71C6 71C7 71C8 71C9 71CA 71CB 71CC 71CD
71CE 71CF 71D0 71D1 71D2 71D3 71D4 71D5 71D6 71D7
71D8 71D9 71DA 71DB 71DC 71DD 71DE 243BD 243C1 243E9
243EA 243F2 729C 729D 729E 729F 7363 7365 7366 7367
7368 7369 736A 736B 736C 736D 747F 3EFC 748D 748F
7491 7492 7494 7495 7496 7498 7499 749A 749B 749C
749D 749E 749F 74A0 74A1 74A3 74A4 24A42 24A45 24A4A
24A4E 24A4F 24A50 24A51 24A5D 24A65 24A66 24A67 74A2 74E2
750A 750B 750C 750D 750E 7580 7581 7582 24CC9 762C
762D 762E 762F 7630 7631 7632 7633 7634 7635 7636
7637 7638 7639 763A 763B 763C 763D 763E 763F 764A
769F 76A0 76A1 76BB 76E5 76E6 76E7 2509D 4062 4065
406A 7794 7795 7796 7797 7798 7799 779A 779B 779C
779D 779E 779F 77A0 77A1 77A2 77A3 25299
77A5 78D6 78DC 40D8 78DA 78DB 78DD 78DE 78DF 78E0
78E1 78E2 78E3 78E5 78E6 78E7 78E8 78E9 78EA 78EB
78EC 78ED 78EE 2555B 2555C 2555D 2555E 25562 25565 25566
79A4 79A5 79A6 79A9 256F6 7A44 7A45 7A46 7A47 7A48
7A4B 7A4C 7A4D 7A4E 7A4F 7A50 7A51 7A52 2585D 25872
7A53 41B2 7AB5 7AB6 7AB7 7AB8 7AB9 7ABA 7ABB 7ABC
7ABD 7AF1 25AAE 422A 422B 422D 422E 4230 4231 4232
7BC9 7BD4 7BD5 7BD6 7BD7 7BD8 7BD9 7BDA 7BDB 7BDC
7BDD 7BDE 7BDF 7BE0 7BE1 7BE2 7BE3 7BE4 7BE5 7BE6
7BE7 7BE8 7BE9 7BEA 7BEB 7BEC 7BED 7BEE 7BEF 7C11
25C21 25C4A 25C65 7BF9 4A00 7CCF 7CD0 7CD1 7CD2 7CD3
7CD4 7CD5 7CD6 7CD7 7CD8 7E08 7E09 7E0A 7E0B 7E0C
7E0D 7E0E 7E0F 7E10 7E11 7E12 7E13 7E14
7E15 7E16 7E17 7E18 7E19 7E1A 7E1B 7E1C 7E1D 7E1E
7E1F 7E20 7E21 7E22 7E23 7E24 7E25 7E26 7E27 7E28
7F30 7F31 7F32 7F33 7F34 7F43 7F79 7F7A 7F7B 7F7C
7FB1 7FB2 7FEE 7FEF 7FF0 7FF1 8028 8029 802A 26512
43C1 806C 806D 265BF 806E 8190 443B 81A6 81A7 81A8
81A9 81AA 81AB 81AC 81AD 81AE 81AF 81B0 81B1 81B2
81B3 81B4 81B5 81B6 2681C 2686E 81B7 81B9 81F2 81FB
8208 8206 8209 8218 8255 8256 8257 8258 8259 4500
4503 4504 4509 450B 84DE 853E 853F 8540 8541 8542
8543 8544 8545 8546 8547 8548 8549 854A 854B 854C
854D 854E 8550 8551 8552 8553 8554 8555 8556 8558
8559 855A 855B 855C 855D 855E 855F 8560 8561 8562
8563 8564 8565 8566 8567 8568 8569 856A
856B 856C 856D 856E 856F 8570 8571 8572 8573 8574
8575 26E99 26ED0 26ED1 26ED2 26ED3 26ED4 26ED5 26ED6 26ED7
26F26 858C 8663 8664 8665 8666 45DD 8779 8781 8783
8784 8785 8786 8787 8788 8789 878A 878B 878C 878D
878E 878F 8790 8791 8792 8793 8794 8795 8796 8797
8798 8799 879A 879B 879C 879D 879E 879F 87A0 87A1
87A2 87A3 87A4 87A5 87A6 87A7 87A8 87A9 4617 885E
885F 8860 8861 275FE 464F 891E 891F 8920 8921 8922
8923 8924 8925 8926 8927 8928 8929 892A 892B 892C
892D 892E 892F 8930 8931 8932 8934 27735 27736 27741
89A6 89A7 89A8 89AA 27870 89F1 8ADC 8ADD 8ADE 8ADF
8AE0 8AE1 8AE2 8AE3 8AE4 8AE5 8AE6 8AE7 8AE8 8AEA
8AEB 8AEC 8AED 8AEE 8AEF 8AF0 8AF1 8AF2
8AF3 8AF4 8AF5 8AF6 8AF7 8AF9 8AFA 8AFB 8AFC 8AFD
8AFE 8AFF 8B00 8B01 8B02 8B03 27A79 27A84 8B14 474E
8C6B 8C6D 8C6E 8C90 8C91 8C92 8C93 8CEE 8CEF 8CF0
8CF1 8CF2 8CF3 8CF4 8CF5 27DBD 8D5D 8D5E 8D5F 8D60
8D6C 8D6E 8DA5 8DA6 8DA7 8E30 8E31 8E32 8E33 8E34
8E35 8E36 8E38 8E39 8E3B 8E3C 8E3D 8E3E 8E3F 8E40
8E41 8E42 8E43 8E44 8E45 280BD 280BE 8EBD 8EBE 8F2D
8F2E 8F2F 8F30 8F31 8F32 8F33 8F34 8F35 8F36 8F37
8F38 8F39 8F3A 8F3B 8F3C 28365 8F99 8F9A 8FA5 8FA6
8FA7 8FA8 8FA9 8FAA 48AD 9072 9074 9075 9076 9078
9079 907A 907B 907C 9086 285E8 285F4 28600 90FA 9133
9134 9135 9136 9137 4900 4906 918D 918E 918F 9190
9191 9192 9193 9194 9195 9196 9197 92CB
4935 92F7 92F8 92F9 92FA 92FB 92FC 92FD 92FE 92FF
9300 9301 9302 9303 9304 9305 9306 9307 9308 9309
930A 930B 930C 930D 930E 930F 9310 9311 9312 9313
9314 9315 9316 9317 9318 9319 931A 931B 931C 931D
931E 931F 9320 9321 9322 9323 9324 9325 9326 9327
9329 932A 932B 932C 932D 932E 932F 9330 9331 9332
9333 9334 9335 9336 9337 9338 9339 933A 933B 933C
933D 933E 933F 9340 9341 9342 9343 9344 9345 9346
9348 FA28 289E7 289E8 289F9 289FA 289FB 289FC 28A0F 28A16
28A25 28A29 937A 9556 9557 9558 9559 955A 955B 955C
955D 955E 955F 9560 49A1 49A7 95B5 95B6 95B8 95B9
95BA 95BB 95BC 95BD 95BE 95BF 95C1 95C2 95CD 961B
49E7 96A6 96A7 96A8 96A9 96AA 96AB 96B7
96D4 96D5 4A1D 970B 970C 970D 970E 970F 9710 9711
9712 9713 9714 9715 9716 9717 290E4 290E5 290EC 290ED
975B 975C 9766 9793 9794 9795 9796 9797 9798 9799
97D2 97F0 97F8 9824 9825 982D 982E 982F 9830 9831
9832 9833 9834 9835 9836 9837 9838 9839 983A 983B
983C 983D 294D0 294D9 294DA 989E 989F 98A0 98A1 98B4
98B5 295D7 98D9 98DA 9910 991D 991A 991B 991C 991E
991F 9920 9921 9922 9923 9924 9926 9927 9928 9929
29720 9934 999E 999F 99A0 99E1 99E2 99E3 99E4 99E5
99E6 99E7 99E8 99E9 99EA 99EB 99EC 99ED 99EE 99EF
99F0 99F1 99F2 298D1 9AB8 9AB9 9ABA 9ABB 9ABC 29A4D
9ABF 9AED 9AF5 9AF6 9AF7 9AF8 9AF9 9AFA 9AFB 9B07
9B28 9B33 9B47 4C49 9B7A 9B7B 9B7C 9B7D
9B7E 9B7F 9B80 9B81 9B82 9B83 9B84 9B85 9B87 9B88
9B89 9B8A 9B8B 9B8C 9B8D 9B8E 9B8F 9B90 9B91 9B92
9B93 9B94 9B95 9B96 9B97 9B98 9BA3 29D98 29D9B 9CAD
9CAE 9CAF 9CB0 9CB1 9CB2 9CB3 9CB4 9CB5 9CB6 9CB7
9CB8 9CB9 9CBA 9CBB 9D0A 9D0F 9D10 9D11 9D12 9D13
9D14 9D15 9D16 9D17 9D18 9D19 9D1A 9D1B 9D1D 9D1E
9D1F 9D20 9D21 9D22 9D23 9D24 9D25 9D26 9D27 9D28
9D29 9D2A 9D2B 9D2C 29FDE 9E65 9E66 9E67 9E68 9E77
9E7E 9E85 9E86 9E87 9E88 2A29F 4D34 9EAC 9EAD 9EAE
9EBA 9EC5 9EC6 9ED4 9ED5 9ED6 9ED7 9ED8 9EFA 9F12
9F3C 9F3D 9F53 9F8D 9F9C
FDD0-2811
511F 5120 5121 5122 5123 5124 5125 5126 5127 5128
5129 512A 512C 20413 20414 2041F 5132 51DF 20619 2061A
5295 3525 3526 52F4 52F5 52F6 5335 3553 53B3 20B8F
567D 567E 567F 5680 5681 5682 5683 5684 5685 5686
5687 5688 5689 568A 568B 568C 568D 568E 568F 5690
5691 5692 5693 2104F 2105C 2106F 21075 21076 21077 21078
2107B 21088 367A 58CD 58CE 58CF 58D0 58D1 58D2 58D3
58D4 58D5 58D6 58D7 2148A 21497 21692 5B23 5B24 5B25
5B26 5B27 5B29 5B2A 5B2B 5B2C 5B2D 5B2E 5B2F 5B30
5B31 5B32 5B33 5B35 5B36 5B37 218D1 218D6 218D7 218D8
218D9 5B7A 5B7B 5BF1 5BF2 5C36 5C37 5C68 3819 5DB7
5DB8 5DB9 5DBA 5DBC 5DBD 5DBE 5DBF 21F6A 5DBB 3865
3866 5E6A 5E6B 5E6C 5F4C 5FBD 5FBE 2258D
61B5 61BC 61C2 61C3 61C7 61C9 61CB 61D1 61D7 61D9
61DA 61DB 61DC 61DD 61DE 61E0 61E1 61E2 61E4 61E5
61E6 61E7 61E8 228F7 6232 6234 64CA 64CE 64D8 3A5C
3A5E 64DF 64E0 64E1 64E2 64E3 64E4 64E6 64E8 64E9
64EB 64EC 64ED 64EE 64EF 64F0 64F1 22DEE 6580 6581
6582 6583 65A3 65B5 65B6 65DA 3B20 3B21 3B22 66CE
66D0 66D1 66D2 66D3 66D4 66D5 66D6 66D7 66DA 232D2
66D9 3B32 3B40 3BEC 3BF2 3BF3 3BF4 6A7D 6A7E 6A7F
6A80 6A81 6A82 6A83 6A84 6A85 6A86 6A87 6A88 6A89
6A8A 6A8B 6A8C 6A8D 6A8E 6A8F 6A90 6A91 6A92 6A93
6A94 6A95 6A96 6A97 6A98 6A99 6A9A 6A9B 6A9C 6A9D
6A9E 6A9F 6AA0 6AA1 6AA2 6AA3 6AA4 6AA5 6AA6 6AA7
6AA8 6AA9 6AAA 23703 23716 23720 2372D 2372F
2373F 6ADB 3C46 6B5B 6B5C 6B5D 6BAC 6BAD 6BAE 6BDA
6C08 6C09 6C0A 6FA9 6FCC 3D6F 3D73 3D75 6F80 6FD4
6FD5 6FD7 6FD8 6FD9 6FDA 6FDB 6FDC 6FDD 6FDE 6FDF
6FE0 6FE1 6FE2 6FE3 6FE4 6FE5 6FE6 6FE7 6FE8 6FE9
6FEA 6FEB 6FEC 6FED 6FEE 6FEF 6FF0 6FF1 6FF2 6FF4
6FF5 6FF6 6FF8 23FC0 24039 2403A 2403B 2403C 2403D 24057
3E00 3E01 3E02 71DF 71E0 71E1 71E2 71E3 71E4 71E5
71E6 71E7 71E8 71E9 71EA 71EB 71EC 71ED 71EE 71EF
71F0 71F1 71F2 71F3 71F4 71F5 71F6 71F7 243F8 24404
7235 7246 3E55 72A0 736E 736F 7370 7371 7372 7373
7374 3EFA 7490 7497 3EFF 3F00 3F01 74A5 74A6 74A8
74A9 74AA 74AB 74AC 74AD 74AE 74AF 74B0 74B1 74B2
74B3 74B4 24A71 24A77 24A78 24A79 24A7A 3F3F
750F 7510 7511 7512 7583 7584 24CD9 7640 7641 7642
7643 7644 7645 7646 7647 7648 7649 764B 764C 764D
764E 24EA5 24EA7 76A2 76A3 76A4 76A5 76BC 4009 76E8
76E9 76EA 406F 4071 77A4 77A6 77A7 77A8 77A9 77AA
77AB 77AC 77AD 77AE 77AF 77B0 77B1 77B2 77B3 77B4
77B5 77B6 77B7 252C7 77EF 77F0 40DF 78EF 78F0 78F1
78F2 78F3 78F4 78F5 78F6 78F7 78F8 78F9 78FA 78FB
78FC 78FD 78FE 78FF 7900 7901 7902 7903 7904 7905
25581 25584 2558F 79A7 79A8 79AA 79AB 25706 417F 7A49
7A54 7A55 7A56 7A57 7A58 7A59 7A5A 7A5B 7A5C 7A5D
7A5E 41B9 7ABE 7ABF 7AC0 7AC1 7AC2 7AF2 7AF3 7AF4
25AAF 7C15 423B 4240 4241 4242 4243 4244 4245 7BF0
7BF1 7BF2 7BF3 7BF4 7BF5 7BF6 7BF7 7BF8
7BFA 7BFB 7BFC 7BFD 7BFE 7BFF 7C00 7C01 7C02 7C03
7C04 7C05 7C06 7C07 7C08 7C09 7C0A 7C0B 7C0C 7C0D
7C0E 7C0F 7C10 7C12 7C13 7C14 7C16 7C17 25C91 25CA4
25CC0 7C18 42A2 7CD9 7CDA 7CDB 7CDC 7CDD 7CDE 7CDF
7CE0 7CE1 7CE2 7CE8 7E29 7E2A 7E2B 7E2C 7E2D 7E2E
7E2F 7E30 7E31 7E32 7E33 7E34 7E35 7E36 7E37 7E38
7E39 7E3A 7E3B 7E3C 7E3D 7E3E 7E3F 7E40 7E41 7E42
7E43 7E44 7E45 7E46 7E47 7E49 7E4A 7E4C 7E4D 26102
26121 7E48 7F44 7F45 7F46 7F7D 7F7E 7F7F 7F81 263BE
7FF2 7FF3 7FF4 7FF5 7FF6 7FFC 26484 26488 26489 802B
802C 806F 8070 8071 8072 8073 8074 26626 81A5 81B8
81BA 81BB 81BC 81BD 81BE 81BF 81C0 81C1 81C2 81C3
81C4 81C5 81C6 81C7 81C8 81C9 81CA 81CC
26846 2685E 81E8 81E9 269F2 825A 825B 825C 825D 8271
4516 451D 4520 4526 4527 8557 8576 8577 8578 8579
857A 857B 857C 857D 857E 857F 8580 8581 8582 8583
8584 8585 8586 8587 8588 8589 858A 858B 858D 858E
858F 8590 8591 8592 8593 8594 8595 8596 8597 8598
8599 859A 859B 859C 859D 859E 859F 85A0 85A1 85A2
85A3 85A4 85A5 85A6 85A7 85A8 85AA 85AB 85AC 85AE
26F73 26F74 26F9F 26FA1 85AD 85AF 8667 8668 45E9 45EE
87AA 87AB 87AC 87AD 87AE 87AF 87B0 87B1 87B2 87B3
87B4 87B5 87B6 87B7 87B8 87B9 87BA 87BB 87BC 87BD
87BE 87BF 87C0 87C1 87C2 87C3 87C4 87C5 87C6 87C7
87C8 87C9 87CA 87CB 87CC 87CD 87CE 87CF 87D0 87D1
87D2 27422 87DE 465B 8933 8935 8936 8937
8938 8939 893A 893B 893C 893D 893E 893F 8940 8941
8942 8943 8944 8945 8954 8952 2775E 89AB 89AC 89AD
89AE 89AF 89F2 89F3 27924 4700 8B04 8B05 8B06 8B07
8B08 8B09 8B0A 8B0B 8B0C 8B0D 8B0E 8B0F 8B10 8B11
8B12 8B13 8B15 8B16 8B17 8B18 8B19 8B1A 8B1B 8B1C
8B1D 8B1E 8B1F 8B20 8B21 8B22 27ABD 27ABE 4726 8C3F
8C40 8C41 8C4F 8C6F 8C70 8C71 8C72 8C73 8C94 8C95
8C96 8CF6 8CF7 8CF8 8CF9 8CFA 8CFB 8CFC 8CFD 27DDC
8D61 8D62 8D6F 8DA8 8E46 8E47 8E48 8E49 8E4A 8E4B
8E4C 8E4D 8E4E 8E4F 8E50 8E51 8E52 8E53 280E8 280E9
280F4 8F3D 8F3E 8F3F 8F40 8F41 8F42 8F43 8F44 8F45
2836D 2837D 8FAB 48AE 907D 907E 907F 9080 9081 9082
9083 9084 9085 9089 2860B 9138 9139 9198
9199 919A 919B 919C 919D 919E 919F 91A0 91A1 91A2
91A3 91A4 28933 493C 9328 9347 9349 934A 934B 934C
934D 934E 934F 9350 9351 9352 9353 9354 9355 9356
9357 9358 9359 935A 935B 935C 935D 935E 935F 9360
9361 9362 9363 9364 9365 9366 9367 9368 9369 936A
936B 936C 936D 936E 936F 9370 9371 9372 9373 9374
9375 9376 9377 9378 9379 937B 937C 937D 937E 937F
9380 9381 9382 9383 9384 9385 9386 9387 28A32 28A36
28A44 28A45 28A46 28A47 28A48 28A49 28A4A 28A4B 28A59 28A5A
28A81 28A82 28A83 93A1 93AF 9561 9562 9563 9564 9565
9566 9567 9568 9569 956A 956B 95C0 95B7 95C3 95C4
95C5 95C6 95C7 95C8 95C9 95CA 95CB 95CC 95CE 95CF
96AC 96AD 96AE 96AF 96B0 96B1 96B2 96B8
4A01 4A02 96D6 4A24 9718 9719 971A 971B 971C 971D
971E 971F 9720 2910D 29110 9721 4A4A 979A 979B 979C
979D 979E 979F 97A0 97A1 97D3 97D4 97D5 97F1 4AD1
9840 9841 9842 9843 9844 9845 9846 9847 9848 9849
984A 294E5 294E7 4B10 98B6 98B7 295E9 295F4 4B20 9925
992A 992B 992C 992D 992F 9930 9931 9932 9933 9935
9937 29732 9942 9946 9998 4B70 4B72 99A1 99A2 99A3
4B8E 4B90 99F4 99F5 99F6 99F7 99F8 99F9 99FA 99FB
99FC 99FD 99FE 99FF 9A00 9A01 9A02 9A03 298EA 99F3
9AA4 9ABD 9ABE 9AFC 9AFD 9AFE 9AFF 9B00 9B01 9B02
9B34 9B48 9B49 9B86 4C4B 4C4C 4C4D 9B99 9B9A 9B9B
9B9C 9B9D 9B9E 9B9F 9BA0 9BA1 9BA2 9BA4 9BA5 9BA6
9BA7 9BA8 9BA9 9BAA 9BAB 9BAC 9BAD 9BAE
9BAF 9BB0 9BB1 9BB2 9BB3 9BB4 9BBA 9BCE 9CBC 9CBD
9CBF 9CC0 9CC1 9CC2 9CC3 9CC4 9CC5 9CC6 9CC7 9CC8
9CC9 9CCA 9CCB 9CBE 9D1C 4CCD 4CD4 9D2D 9D2E 9D2F
9D30 9D31 9D32 9D33 9D34 9D35 9D36 9D37 9D38 9D39
9D3A 9D3B 9D3C 9D3D 9D3E 9D3F 9D40 9D41 9D42 9D43
9D44 9D45 9D46 9D47 9D48 9D49 2A014 9D67 9E69 9E6A
9E6B 9E6C 9E89 9E8A 9E8B 2A2B2 9EAF 9EB0 9EC7 9EC8
9EC9 9ECF 9EDA 9EDB 9EDC 9EDD 9EDE 2A434 9EFB 9EFF
9F22 9F23 9F24 9F3E 9F3F 9F4B 2A5C6 9F54 9F62 9F8B
9F8C 9FA0
FDD0-2812
512D 512E 512F 5131 34AF 204FE 5181 205A5 20AC2 53E2
3609 5694 5695 5696 5697 5698 5699 569A 569B 569C
569D 569E 569F 56A0 56A1 56A2 56A3 56A4 21096 2109D
210B4 210BF 210C0 210C1 210C7 210C8 210C9 210CF 210D3 56AE
58D8 58D9 214B6 5911 5913 5970 3730 3731 5B38 5B3A
5B3B 5B3C 218FA 5C69 5C6A 5DC0 5DC1 5DC2 5E6D 5E6E
5EEB 5F4D 5F5D 5F5E 399B 61D5 61D6 61D8 61DF 61E3
39A1 61E9 61EA 61EB 61ED 61EE 61F0 61F1 61F3 61F4
6233 64E7 64EA 3A61 3A66 3A67 64E5 64F2 64F4 64F5
64F6 64F7 64F8 64F9 64FA 64FB 64FC 64FD 64FE 64FF
6501 6502 6503 6504 6505 6506 22E0D 3AAB 8D01 23066
6594 2308E 65B7 65DB 3B23 3B24 3B25 3B26 3B27 3B28
66D8 66DB 66DC 232E0 232E1 6726 6AAB 6AAC
6AAD 6AAE 6AAF 6AB0 6AB1 6AB2 6AB3 6AB4 6AB5 6AB6
6AB7 6AB8 6AB9 6ABA 6ABB 6ABC 6ABD 6ABE 6ABF 6AC0
6AC1 6AC2 6AC3 6AC4 6AC5 6AC6 6AC7 6AC8 6AC9 6ACA
23766 23781 6AE1 6AED 6B5E 6B5F 6B78 6BAF 6BC9 6C0B
6FF7 3D7D 6FF9 6FFA 6FFB 6FFC 6FFD 6FFE 6FFF 7000
7001 7002 7003 7005 7006 7007 7008 7009 700A 700B
700C 700D 700E 700F 7010 7011 7012 7013 7014 24085
2408B 2408C 2408D 24091 7026 3E04 71F8 71F9 71FA 71FB
71FC 71FD 71FE 71FF 7200 7201 7203 24435 24436 269A8
7375 7376 7377 74A7 74B5 74B6 74B8 74B9 74BB 74BC
74BE 74BF 74C0 74C1 74C2 24A8C 24A93 24A94 24A95 24A96
24AA4 24AA5 24AA6 24AA7 7513 7514 7515 7585 764F 7650
7651 7652 7653 7654 7655 7656 7657 7658
7659 765A 765B 765C 765D 765E 7664 76A6 76A7 76A8
24FA9 3FF9 76BD 76EB 76EC 77B8 77B9 77BA 77BB 77BC
77BD 77BE 77BF 77C0 77C1 77C2 252D8 7906 7907 7908
7909 790A 790B 790C 790D 790E 790F 7910 7911 7912
7913 7914 7915 7916 4120 79AC 79AD 79AE 79AF 7A5F
7A60 7A61 7A62 7A63 258C8 7AC4 7AC5 7AF5 424E 4255
7C19 7C1A 7C1B 7C1C 7C1D 7C1E 7C1F 7C20 7C21 7C22
7C23 7C24 7C25 7C26 7C27 7C28 7C29 7C2A 7C2B 7C2D
7C2E 7C2F 7C30 7C31 7C32 25CC1 25CFE 25D20 42A6 7CE3
7CE4 7CE5 7CE6 7CE7 25F1A 4318 7E4E 7E4F 7E50 7E51
7E52 7E53 7E54 7E55 7E56 7E57 7E58 7E59 7E5A 7E5B
7E5C 7E5D 7E5E 7E5F 7E60 7E61 7E62 7E63 7E64 7E65
7E67 7E71 26159 2615A 2615B 2615C 7F47 7F48
7F49 26258 7F80 7F82 7FB3 7FB4 7FB5 4397 7FF7 7FF8
7FF9 7FFA 7FFB 2648D 26498 802D 802E 8075 8076 8077
4443 4444 4445 81CD 81CE 81CF 81D0 81D1 81D2 81D3
26888 820A 8219 825E 825F 8260 448F 452E 4533 453B
453D 453F 4540 4543 4544 85A9 85B0 85B1 85B2 85B3
85B4 85B5 85B6 85B7 85B8 85B9 85BA 85BB 85BC 85BD
85BE 85BF 85C0 85C1 85C2 85C3 85C4 85C5 85C6 85C7
85C8 85C9 85CA 85CB 85CC 85CD 85CE 85CF 85D0 85D2
85D3 26FBE 26FDE 26FDF 2700E 459B 8669 87D7 87D3 87D4
87D6 87D8 87D9 87DA 87DB 87DC 87DD 87DF 87E0 87E2
87E3 87E4 87E5 87E6 87E7 87E8 87E9 87EA 87EB 87EC
87ED 87EE 87EF 87F0 87F1 87F2 87F3 87F4 87F5 880E
27450 27484 27486 8946 8947 8948 8949 894A
894B 894C 894D 894E 894F 8950 8951 8953 8955 27784
27785 8986 468D 89B0 89B1 89B2 89B3 89F4 9D64 8B23
8B24 8B25 8B26 8B27 8B28 8B29 8B2A 8B2B 8B2C 8B2D
8B2E 8B2F 8B30 8B31 8B32 8B33 8B34 8B35 8B36 8B37
8B38 8B39 8B3A 8B3B 8B3C 8B3D 8B3E 27AF4 8B47 8C42
8C50 8C74 8C75 8C97 8C98 8C99 8CFE 8CFF 8D00 8D02
8D03 8D04 8D05 8DA9 4820 8E54 8E55 8E56 8E57 8E58
8E59 8E5A 8E5B 8E5C 8E5D 8E5E 8E5F 8E60 8E61 8E62
8E63 8E64 8E65 8E66 8E67 8E6E 8E80 2812E 8E69 8EBF
8EC0 8EC1 2827C 4871 8F46 8F47 8F48 8F49 8F4A 8F4B
8F4C 2838A 8FAC 9087 9088 28625 9128 913A 913B 913C
913D 913E 91A5 91A6 91A7 91A8 91A9 91AA 91AB 91AC
91D0 493E 4944 4945 4947 9388 9389 938A
938B 938C 938D 938E 938F 9390 9391 9392 9393 9394
9395 9396 9397 9398 9399 939A 939B 939C 939D 939E
939F 93A0 93A2 93A3 93A4 93A5 93A6 93A7 93A8 93AA
93AB 93AC 93AD 93AE 93B0 93B1 93B2 93B3 93B4 93B5
93B6 93B7 93B8 93B9 93BA 93BB 93BC 93BD 93BE 93BF
28A9A 28A9B 28A9C 28AC0 28AC6 28ACB 28ACC 28ACE 956C 956D
956E 956F 9570 9571 95D0 95D1 95D2 95D3 95D4 95D5
95D6 95D7 95D8 28D99 96B3 4A03 96D7 96D8 96D9 96DA
96DB 96DC 96DD 96DE 96DF 96E0 96E2 4A26 9722 9723
9724 9725 975D 97A2 97A3 97A4 97A5 97A6 97A7 97A8
97A9 97AA 97AB 97AC 97AD 97AE 97AF 97B0 4A96 97D6
97D7 97D8 97D9 97DA 97F9 97FA 29433 983F 983E 984B
984C 984D 984E 984F 9850 9851 9852 9853
9854 9855 98A2 98A3 98B8 98B9 98BA 4B49 4B4B 4B4C
992E 9936 9938 9939 993A 993B 993C 993D 993E 993F
9940 9941 99A4 99A5 4B93 4B96 4B97 9A04 9A05 9A06
9A07 9A08 9A09 9A0A 9A0B 9A0C 9A0D 9A0E 9A0F 9A10
9A11 9A12 9A13 298F1 29903 29905 9AC0 9AC1 9ADC 4C00
4C01 9B03 9B04 9B05 9B06 9B08 9B29 9B35 9B36 4C26
9B4A 9B4B 9B4C 9B4D 9B4E 9B4F 9BB5 9BB6 9BB7 9BB8
9BB9 9BBB 9BBC 9BBD 9BBE 9BBF 9BC0 9BC1 9BC2 9BC3
9BC4 9BC6 9BC7 9BC8 9BC9 9BCA 9BCB 9BCC 9BCD 9BCF
9BD0 9BD1 9BD2 9BD3 9BFD 29DF6 9CCC 9CCD 9CCE 9CCF
9CD0 9CD1 9CD2 9D4A 9D4B 9D4C 9D4D 9D4E 9D4F 9D50
9D51 9D52 9D53 9D54 9D55 9D56 9D57 9D58 9D59 9D5A
9D5B 9D5C 9D5D 9D5E 9D5F 9D60 9D62 9D63
9D65 9E6D 9E6E 9E6F 9E70 4D26 9E8C 9E8D 9E8E 9E8F
9E90 2A2B4 2A2B6 2A2BA 2A2BD 4D36 9EB1 9EB2 9EBF 9ECA
9ECB 9EDF 9EE0 9EE1 9F00 9F01 9F02 9F15 9F16 9F25
9F26 9F27 9F28 9F29 9F2A 9F2B 9F2C 9F4C 9F55 9F8E
FDD0-2813
3426 34A3 5133 5134 5135 5296 52F7 52F8 5336 53B4
58E1 56A5 56A6 56A7 56A8 56A9 56AA 56AB 56AC 56AD
56AF 56B0 210E4 210F4 210F5 210F6 58DA 58DB 58DC 58DD
58DE 58DF 58E0 58E2 5912 5B3D 3732 3733 3734 3735
5B39 5B3E 5B3F 21903 21904 21905 5B7C 5BF3 5BF4 5BF5
5C6B 3820 3822 5DC3 5DC4 5DC5 21F9E 21FA1 5E70 22173
5EEC 5EED 9F90 5F5F 5FBF 61EC 61EF 61F2 61F5 61F6
61F7 22926 22939 61FB 6500 6507 6508 6509 650A 650B
650C 650D 650E 650F 6510 6512 22E36 22E42 6584 65DC
65DD 65DE 3B29 3B2A 66DD 66DE 66DF 66E0 66E1 66E2
3C00 3C01 3C02 3C04 6ACB 6ACC 6ACD 6ACE 6ACF 6AD0
6AD1 6AD2 6AD3 6AD4 6AD5 6AD6 6AD7 6AD8 6AD9 6ADA
6ADC 6ADD 6ADE 6ADF 6AE0 6AE2 6AE3 6AE4
6AE5 6AE6 6AEB 237A2 237BC 237C2 6AE7 6B60 6BB0 6BB1
23ADB 6C0C 3D85 3D8A 6FF3 7015 7016 7017 7018 7019
701A 701B 701C 701D 701E 701F 7020 7021 7022 7023
7024 7025 7027 7028 7029 702B 702C 702D 702E 240C9
240E1 7202 3E06 7204 7205 7206 7207 7208 7209 720A
720C 720D 720E 7215 2445A 2445B 7258 72A1 72A2 72A3
72A4 72A5 72A6 7378 7379 737A 74B7 74BD 3F04 3F06
74C3 74C4 74C5 74C6 74C7 74C8 74C9 74CA 74CB 24AB1
24AB2 24AB3 24ABA 24ABB 24ABC 74E3 7516 7586 7587 765F
7660 7661 7663 76A9 2502C 77C3 77C4 77C5 77C6 77C7
77C8 77C9 77CA 2530E 25311 25313 77F1 7917 7918 7919
791A 791B 791C 791D 791E 791F 7920 7921 255B9 79B0
79B1 2571D 7A64 7A65 7A66 7A67 7A68 7A69
7A6A 7A6B 258E1 7AC6 7C2C 424F 4260 7C33 7C34 7C35
7C36 7C37 7C38 7C39 7C3A 7C3B 7C3C 7C3D 7C3E 7C3F
7C40 7C41 7C42 25D30 25D43 7CE9 7CEA 7CEB 7CEC 7CED
25F4B 4320 7E4B 7E66 7E68 7E69 7E6A 7E6B 7E6C 7E6D
7E6E 7E6F 7E70 7E72 7E73 7E74 7E75 7E76 7E77 7E78
7E79 7E7A 7F35 7F4A 7F4B 7F83 7F84 7F85 7F86 7FB6
7FB7 7FB8 7FB9 7FFD 7FFE 8078 81CB 4446 81D4 81D5
81D7 81D8 2688A 26893 26926 820B 821A 8261 8262 8263
8264 8265 8276 4545 4546 4551 FA1F 85D1 85D5 85D6
85D7 85D8 85D9 85DA 85DB 85DC 85DD 85DE 85DF 85E0
85E1 85E2 85E3 85E4 85E5 85E6 85E7 85E8 85E9 85EA
85EB 85EC 85ED 85EF 85F0 85F1 85F2 85F3 85F4 85F5
2704B 27052 27053 85F7 85F8 8801 87D5 87F6
87F7 87F8 87F9 87FA 87FB 87FC 87FD 87FE 87FF 8800
8802 8803 8804 8805 8806 8807 8808 8809 880A 880B
880C 880D 880F 881E 8956 8957 8958 8959 895A 895B
895C 895D 895E 895F 8960 8961 8962 8987 8988 89B4
89B5 89B6 89B7 89B8 89F5 89F6 8B3F 8B40 8B41 8B42
8B43 8B44 8B46 8B48 8B49 8B4A 8B4B 8B4C 8B4E 8B4F
8B50 8B51 8B52 8B53 8B54 8B55 8B56 8B57 8B58 8B59
8B5A 8B5B 8B5C 27B06 27B0B 27B18 8C36 8C43 8C77 8C76
8C9A 8D06 8D07 8D08 8D09 8D0A 8D0B 8D0C 8DAA 8DAB
8DAC 8DAD 4826 8E68 8E6A 8E6B 8E6C 8E6D 8E6F 8E70
8E71 8E72 8E73 8E74 8E75 8E76 8E77 8E78 8E79 8E7A
8E7B 8E7C 8E7D 8E7E 8E7F 2814F 2815D 2816F 8E87 8EC2
8EC3 8EC4 8EC5 8F4D 8F4E 8F4F 8F50 8F51
8F52 8F53 8F54 8FAD 8FB4 908A 908B 908C 2863B 913F
9140 9142 4911 91AD 91AE 91AF 91B0 91B1 4949 4951
4953 93A9 93C0 93C1 93C2 93C3 93C4 93C5 93C6 93C7
93C8 93C9 93CA 93CB 93CC 93CD 93CE 93CF 93D0 93D1
93D2 93D3 93D4 93D5 93D6 93D7 93D8 93D9 93DA 93DB
93DC 93DD 93DE 93DF 93E0 93E1 93E2 93E3 93E4 93E5
93E6 93E7 93E8 93E9 93EA 93EB 93EC 93ED 93EE 93EF
93F0 93F1 93F2 93F9 28ADE 28ADF 28AE0 28AE1 28AE2 28AE3
28AE5 28AEA 28AFC 28B0C 9572 957D 95D9 95DA 95DB 95DC
95DD 96B4 4A04 96E1 96E3 9726 9727 9728 9729 972A
972B 972C 972D 2913C 2914D 2915B 9761 97B1 97B2 97B3
97B4 97B5 97B6 97B7 97DC 97DD 97DE 97DF 97F2 97FB
97FC 4AE4 9856 9857 9858 9859 985A 985B
985C 985D 985E 98A4 4B19 98BB 98BC 98BD 98BE 98BF
98C0 4B53 9943 9944 9945 9947 9948 9949 99A6 99A7
29857 4B9D 9A14 9A15 9A16 9A17 9A18 9A19 9A1A 9A1B
9A1C 9A1D 9A1E 9A1F 9A20 9A21 9A22 9A23 9A24 9A25
9A26 9A27 9A28 2992F 9AA5 9AC2 9AC3 9AC5 4C04 4C07
9B09 9B0A 9B0B 9B0C 9B0D 9B0E 9B0F 9B37 9BC5 4C5B
9BD4 9BD5 9BD6 9BD7 9BD8 9BD9 9BDA 9BDB 9BDC 9BDD
9BDE 9BDF 9BE0 9BE1 9BE2 9BE3 9BE4 9BE5 9BE6 9BE7
9BE8 9BE9 9BEA 9BEB 9BEC 9BED 9BEE 9BEF 9BF0 9BF1
9BF2 9BF3 9BF4 9BF5 29E06 29E2D 9BFA 9CD3 9CD4 9CD5
9CD6 9CD7 9CD8 9CD9 9CDA 9CDB 9D61 4CE1 9D66 9D68
9D69 9D6A 9D6B 9D6C 9D6D 9D6E 9D6F 9D70 9D71 9D72
9D73 9D74 9D75 9D76 9D77 9D78 9D79 9D7A
9D7B 9D7C 9D7D 9D7E 9D7F 9D80 9D81 9D82 9D83 9D84
9D85 9D86 9D87 9D88 9D89 9D8A 9D8B 9D8C 9D8D 9D8E
9D8F 9D91 2A087 9E71 9E72 9E78 9E91 9E92 9E93 9E94
9E95 9E96 9E97 9EB3 9EB4 9EC0 4D4C 9EE2 9EE3 9EFC
9F03 9F04 9F17 9F2D 9F40 9F41 9F4D 9F56 9F57 9F58
9F8F 2A6A9
FDD0-2814
34A5 5136 5337 56B1 56B2 56B3 56B4 56B5 56B6 56B7
56B8 56B9 2112F 56BC 58E3 58E4 58E5 214E8 3736 3737
3738 5B40 5B41 5B42 5B43 5B44 5B45 5B46 21910 21911
21912 21915 5B7D 5B7E 5BF6 5DC6 5DC7 5DC8 5DC9 5DCA
5DCC 5E71 2217A 5EEE 5EEF 5EF0 5FC0 5FC1 39A4 61F8
61F9 61FA 2294F 3A70 6513 6514 6515 6516 6517 6518
6519 651A 6585 6586 65DF 3B2B 66E3 66E4 66E5 66E6
66E7 66E8 23300 6727 3C09 3C0A 3C0D 3C11 6AE8 6AE9
6AEA 6AEC 6AEE 6AEF 6AF0 6AF1 6AF2 6AF3 6AF4 6AF5
6AF6 237D5 237D6 237D7 6AF9 702A 3D8F 3D91 702F 7030
7031 7032 7033 7034 7035 7036 7037 7038 7039 703A
703B 703C 703D 703E 703F 7040 7041 24104 7042 3E0A
720B 720F 7210 7211 7212 7213 7214 7216
7217 7218 24473 24487 24488 72A7 72A8 24706 737B 737C
737D 74BA 74CC 74CD 74CE 74CF 74D0 74D1 74D2 24AC0
24AC7 7588 7589 7662 7665 7666 76AA 76AB 3FFA 76BE
76ED 77CB 77CC 77CD 77CE 77CF 77F2 7922 7923 7924
7925 7926 7927 7928 7929 792A 792B 792C 79B2 7A6C
7A6D 7A6E 7A6F 7AC7 7AF6 7AF7 7C43 7C44 7C45 7C46
7C47 7C48 7C49 7C4A 7C4B 7C4C 7C4D 7C4E 7C4F 7C55
42AE 7CEE 7CEF 7CF0 4326 7E7B 7E7C 7E7D 7E7E 7E7F
7E80 7E81 7E82 7E83 261AD 261AE 261B2 7F4C 26335 7FBA
7FFF 8000 802F 8079 807A 807B 807C 81D6 81D9 81DA
81DB 81DC 269B5 8266 8267 8268 8269 4552 4554 4555
4557 455C 8624 85EE 85F6 85F9 85FA 85FB 85FC 85FD
85FE 85FF 8600 8601 8602 8603 8604 8605
8606 8607 8608 8609 860A 860B 860C 860D 860E 860F
8610 8611 8613 8614 8622 27088 270AD 270AE 270AF 8612
861B 8630 4600 4601 8810 8811 8812 8813 8814 8815
8816 8817 8818 8819 8963 8964 8965 8966 8967 8968
89B9 89BA 89BB 89F7 89F8 89F9 4713 4718 8B4D 8B5D
8B5E 8B5F 8B60 8B61 8B62 8B63 8B64 8B65 8B66 8B67
8B68 8B69 8B6A 8B6B 8B6C 8B6D 8B6E 8B6F 8B70 8B71
8B72 27B38 27B39 27B3A 8C51 27C12 8D0D 8D0E 8D0F 8DAE
8E81 8E82 8E83 8E84 8E85 8E86 8E88 8E89 28189 8EC6
8F55 8F56 8F57 8F58 8F59 8F5A 8FAE 908D 9141 9143
91B2 91B3 91B4 91B5 91B6 91B7 91B8 91CB 93F3 93F5
93F6 93F7 93F8 93FA 93FB 93FC 93FD 93FE 93FF 9400
9401 9402 9403 9404 9405 9406 9407 9408
9409 940A 940B 940C 940D 940E 940F 9410 9411 9412
9413 9414 9415 9416 9417 9418 9419 941A 941B 941C
941D 941E 941F 9420 9421 9422 9423 9424 9425 9426
9427 9428 28B13 28B21 28B22 28B2B 28B2C 28B2D 28B2F 28B46
28B4C 28B4E 28B50 942F 943C 9573 9574 95DE 95DF 95E0
95E1 28DB9 96B5 972E 972F 9730 9731 9733 9734 2915E
29170 4A4B 97B8 97B9 97BA 97BB 97DB 97E0 97FD 97FE
97FF 985F 9860 9861 9862 9863 98A5 98C1 98C2 98C3
98C4 994A 994B 994C 994D 994E 9950 9951 9952 9953
9959 99A8 9A29 9A2A 9A2B 9A2C 9A2D 9A2E 9A2F 9A30
9A31 9A32 9A33 9A34 9A35 9A36 9A37 9A38 29947 29948
29949 2995D 2996A 9AA6 9AA7 9AC4 9AC6 9AC7 9AC8 9AC9
9ACA 9ACB 9ACC 9B10 9B11 9B12 9B13 9B2A
9B38 9B50 9BFB 4C6D 9BF6 9BF7 9BF8 9BF9 9BFC 9BFE
9BFF 9C00 9C01 9C02 9C03 9C04 9C05 9C06 9C07 9C08
9C09 9C0A 9C0B 9C0C 9C0D 9C0E 9C0F 9C10 9C11 9C12
9C13 9C14 9C15 9C16 9C17 9C18 9C19 9C1A 9C1B 9C20
29E68 9C40 9CDC 9CDD 9CDE 9CDF 4CED 9D90 9D92 9D93
9D94 9D95 9D96 9D97 9D98 9D99 9D9A 9D9B 9D9C 9D9D
9D9E 9D9F 9DA0 9DA1 9DA2 9DA3 9DA4 9DA5 9DA6 9DA7
9DA8 9DA9 9DAA 9DAB 2A0B9 2A0E1 2A0ED 2A0F3 2A0F8 9DBF
9E79 9E98 9E99 9E9A 9E9B 9EB5 9EC1 2A3A9 4D4D 9EE4
9EE5 9EE6 9EE7 9EE8 9EE9 9EEA 2A45B 9F0D 9F2E 9F2F
9F30 2A5CB 9F59 9F5A 9F5B 9F5D 9F5E 9F5F 9F60 9F61
9F63 9F91
FDD0-2815
34A7 5137 5138 5139 513A 5164 5297 5298 2082C 535B
56BA 56BB 56BD 56BE 56BF 56C0 56C1 56C2 56C3 56C4
56CD 2113B 2113D 21145 21148 2114F 58E6 214FD 5914 3739
5B47 5B48 5B49 2191C 5BF7 5C6C 5DCB 3826 5DCD 5DCF
5DD0 5EF1 5FC2 61FC 61FD 61FE 6511 651B 651C 651D
22E78 6595 66E9 2330A 6707 3C15 6AF8 6AFA 6AFB 6AFC
6AFD 6AFE 6AFF 6B00 6B01 6B02 6B03 6B04 6B05 6B0C
6BB2 7043 7044 7045 7046 7047 7048 7049 704A 704B
704C 704D 704F 7050 2410F 24119 3E0D 7219 721A 721B
244B9 721D 737E 74D3 74D4 74D6 24ACA 24AD1 7517 3FD7
7667 7668 7669 766A 766B 76AC 24FB8 77D0 77D1 77D2
77D3 792D 792E 792F 7930 7931 7932 7933 7934 255D5
255DB 25725 25903 7AC3 7AC8 7AC9 7C56 426A
7C50 7C51 7C52 7C53 7C54 7CF2 7E84 7E85 7E86 7E87
7E88 7E89 7E8A 7E8B 7E8C 7E8D 7E8E 7E8F 7E90 7F4D
7FBB 7FBC 8030 81DD 826A 4562 85D4 8615 8616 8617
8618 8619 861A 861C 861D 861E 861F 8620 8621 8623
8625 8626 8627 8628 8629 862A 862B 862D 862E 862F
270CD 270F8 27109 2710C 4602 881A 881B 881C 881D 881F
8820 8821 8822 8823 8824 8829 882B 884A 8969 896A
896B 896C 896D 896E 89BC 89BD 89FA 8B45 8B73 8B74
8B75 8B76 8B77 8B78 8B79 8B7A 8B7B 8B7C 8B7D 27B48
8D10 8D11 8D12 8D13 8D14 8D63 8DAF 8DB0 8E8A 8E8B
8E8C 8E8D 8E8E 8E8F 281AF 281BC 8EC7 8F5B 8F5C 8F5D
8F5E 8F5F 8FAF 908E 9144 9145 9146 91B9 91BA 91BB
4965 93F4 9429 942A 942B 942C 942D 942E
9430 9431 9432 9433 9434 9435 9436 9437 9438 9439
943A 943B 943D 943E 943F 9440 9441 28B63 28B64 28B65
28B66 28B6C 28B8F 95E2 95E3 95E4 95E5 95E6 96E4 9732
9735 9736 9737 9738 9739 973A 973B 9767 97BC 97BD
97BE 97BF 97E1 97E2 2943F 9864 9865 9866 9867 9868
98A6 98C5 98C6 98C7 98C8 98C9 98CA 98DC 994F 9956
9957 9958 99A9 9A39 9A3A 9A3B 9A3C 9A3D 9A3E 9A3F
9A40 9A41 9A42 9A43 9A44 9A45 9A46 9A47 9ACD 9ACE
9ACF 9B14 9B15 9B16 9B17 9B18 9B39 9B3A 9B51 9B52
9B53 9B54 4C7D 9C1C 9C1D 9C1E 9C1F 9C21 9C22 9C23
9C24 9C25 9C26 9C27 9C28 9C29 9C2A 9C2B 9C2C 9C2D
9C2E 9C2F 9C30 29EAC 4CA3 4CA4 9CE0 9CE1 9CE2 9CE3
9DAC 9DAD 9DAE 9DAF 9DB0 9DB1 9DB2 9DB3
9DB4 9DB5 9DB6 9DB7 9DB8 9DB9 9DBA 9DBB 9DBC 9DBD
9DBE 9DC0 9DC1 9DC2 9DC3 9DC4 9DC5 9DC6 9DC7 9DC8
9DC9 9DCA 9DCC 9DCD 9DCE 9DCF 2A0FE 2A107 2A123 9E7A
9E7B 9E9C 9E9D 4D4E 9EEB 9EEC 9EED 9EEE 9EEF 9F05
9F18 9F19 9F1A 9F1B 9F31 9F4E 9F5C 9F64 9F65 9F66
9F67 9F68 9F69 2A601 9F92 9F9D 9FA1
FDD0-2816
4EB9 513B 513C 20465 20979 3618 56C5 56C6 56C7 56C8
56C9 56CA 56CB 56CE 571D 5971 373A 5B4A 5B4B 5B4C
21922 21927 5B7F 5DCE 5DD1 5DD2 5DD3 5DD4 5DD5 5DD7
5EF2 5F4E 5F72 61FF 6202 22967 2296B 6235 651E 651F
6520 6521 6522 6524 6526 6527 22E8B 23033 3B2C 3B2D
3C18 6AF7 6B06 6B07 6B08 6B09 6B0A 6B0B 6B0D 6B0E
6B61 6C0D 7051 7052 7054 7055 7056 7057 7058 2413F
24140 24144 721C 721E 721F 7220 72A9 737F 7380 74D5
74D7 74D8 74D9 74E4 758A 766C 766D 766E 24F0E 76AD
7935 255E0 79B3 79B4 7A70 7A71 7ACA 7AF8 7C57 7C58
7C59 7C5A 7C5B 7C5C 7C5D 7C5F 7C60 7C61 7CF1 7CF4
432B 7E91 7E92 261DD 7F4E 7F4F 26261 7F87 8031 8032
807D 807E 81DE 81DF 269FA 826B 4567 456A
862C 8632 8633 8634 8635 8636 8637 2710D 27126 27127
4606 8825 8826 8827 8828 882A 882C 896F 8970 8971
8972 89BE 89BF 2789D 89FB 89FC 4720 8B7E 8B7F 8B80
8B81 8B82 8B83 8B84 8B85 8B86 8C44 8D15 8D16 8D17
8D18 8E90 8E91 8E92 8E93 8E94 8E95 8E96 8E97 8E9A
8F60 8F61 8F62 9147 9148 496A 496D 9442 9443 9444
9445 9446 9447 9448 9449 944A 944B 944C 944D 944E
944F 9450 9451 9452 9453 9454 9467 28B99 28B9C 28B9D
9575 9576 957E 95E7 973C 973D 973E 973F 9740 2919C
97C0 97C1 97C2 97C3 97E3 9869 986A 986B 98CB 9954
9955 995A 995B 297D4 9A48 9A49 9A4A 9A4B 9A4C 9A4D
9A4E 9A4F 9A50 9A51 9A52 9A53 9A54 9A55 2999D 9AD0
9AD2 9ADD 9B1D 4C0E 9B19 9B1A 9B1B 9B1C
29BD5 9B2B 9B3B 9B55 9B56 4C81 9C31 9C32 9C33 9C34
9C35 9C36 9C37 9C38 9C39 9C3A 9C3B 9C3C 9C3D 9C3E
9C3F 9C41 9C42 9C43 9C44 9C45 9C46 9C47 9C48 9DE0
29EC3 9C49 9CE4 9DCB 9DD0 9DD1 9DD2 9DD3 9DD4 9DD5
9DD6 9DD7 9DD8 9DD9 9DDA 9DDB 9DDC 9DDD 9DDE 9DDF
2A133 2A134 2A150 9DE9 9DF5 9E73 9E74 9E9E 2A2DF 9EB6
9ED0 9EF0 9EF1 9F32 9F33 9F34 9F35 9F42 4D9C 9F6A
9F6B 9F6C 9F93 9F94 9F95 9FA2
FDD0-2817
513D 5299 529A 20ACD 361A 56CC 56CF 56D0 58E7 58E8
21582 5972 5B4D 5DD6 5DD8 5DDA 5F4F 6200 6201 6203
6204 6523 3A77 6525 6528 6529 652A 652B 6596 3B2E
66EA 66EB 66EC 6B0F 6B10 6B11 6B12 2383A 6BCA 7053
7059 705A 705B 705C 2414E 24155 24156 24157 2415C 2415F
7221 7222 244BC 244CE 244D3 244D6 7381 7382 7383 74DA
24ADF 766F 7670 77D4 7936 7937 79B5 7C5E 4274 7C62
7C63 7C64 7C65 7C66 7C67 7C68 7CF5 7E93 7E94 7E95
7E96 81E2 268C7 826C 4577 8631 8638 8639 863A 863B
863C 863D 863E 863F 8640 8641 2721B 8834 882D 882E
882F 8830 8831 8832 8833 8973 8974 8976 8989 89FD
89FE 8B87 8B88 8B89 8B8A 8B8B 8B8C 8B8D 8B8E 8B8F
8B90 8C45 8D19 8D1A 8DB1 8E98 8E99 8E9B
8E9C 2829B 8F63 8F64 908F 9090 91BC 4972 9455 9456
9457 9458 9459 945A 945B 945C 945D 945E 945F 9460
9461 9462 9463 9464 9465 9466 28BC2 28BC5 28BD4 28BD7
28BD9 28BDA 4A35 9741 291A8 9768 97C4 97C5 9800 986C
986D 986E 986F 98A7 995C 99AA 9A56 9A57 9A58 9A59
9A5A 9A5B 9A5C 9AD1 9AD3 9AD4 9ADE 9B1E 9B1F 9B20
9C4A 9C4B 9C4C 9C4D 9C4E 9C4F 9C50 9C51 9C52 9C53
9C54 9C55 9C56 9C57 9C58 9C59 9C5A 9C5B 29EF8 9C6A
4D00 9DE1 9DE2 9DE3 9DE4 9DE5 9DE6 9DE7 9DE8 9DEA
9DEB 9DEC 9DED 9DEE 9DEF 9DF0 9DF1 9DF2 9DF3 9DF4
9DF6 9DF7 9DF8 9DFB 9DFC 2A192 2A193 2A1AB 9E9F 9EC2
9EF2 9EF3 9EF4 9F06 9F07 9F1C 9F36 9F37 9F38 9F39
9F43 9F44 9F4F 9F6D 9F6E 9F6F 9F70 9F71
2A632
FDD0-2818
513E 204D7 56D1 56D2 56D3 21180 3681 58E9 5B4E 5B4F
5C6D 5DD9 22980 652C 652D 66ED 66EE 6B13 6B14 6B15
705D 705E 705F 7060 7061 7223 74DB 74E5 7671 7672
24FC2 77D5 77D7 77D6 40FA 7938 79B6 79B7 7A73 7A72
4276 7C6A 7E97 7F50 7F88 7F89 826D 8277 8643 8645
27164 27165 8835 8836 8837 8838 8839 883A 884B 8862
8975 8977 277CC 8B91 8B92 8B93 8B94 8B95 8B96 8D1B
8E9D 8E9E 8E9F 8EA0 8EC8 91BD 91BE 91BF 91C0 91C2
9468 9469 946A 946B 946C 28BE7 28BE8 28BE9 28BEA 28BEB
28BEC 28BF5 96E5 96E6 9742 9743 9744 9745 9746 9747
9748 97C6 97C7 97C8 97E4 97E5 29448 9870 995D 9A5D
9A5E 9A5F 9AD5 9B21 9B22 9B2C 9B2D 9B57 9B58 9B59
29D3E 9C5C 9C5D 9C5E 9C5F 9C60 9C61 9C62
9C63 9C64 9C65 9C66 9C67 9C69 9C6B 29F23 9C70 9DFA
4D09 9DF9 9DFD 9DFE 9DFF 9E00 9E01 9E02 9E03 9E04
9E05 9E06 9E07 9E08 9E09 9E0A 2A1B4 9E7C 9E7D 9EA0
9F1E 9F45 9F46 9F72 9F73 9F74 9F75 9F76 9F77
FDD0-2819
56D4 56D5 21187 58EA 5EF3 6205 6206 652E 65B8 3B2F
66EF 6B16 6B17 6B18 6B19 6B1A 6B1B 6B1D 7062 7063
7224 7225 7226 72AA 24D06 77D8 77D9 77E1 7939 7C69
7C6B 7C6C 7C6D 7C6E 7CF6 7E98 7E99 7E9A 7E9B 81E0
81E1 8642 8646 8647 8648 8649 883B 27574 8978 8979
897A 897B 897C 898A 89C0 89FF 8B97 8B98 8B99 8C52
8C9B 8D1C 27E4D 8EA1 8EA2 8EA3 8EA4 8EA5 91C1 946D
946E 946F 9470 9471 9472 9473 28BFF 28C03 9749 9871
9872 995E 995F 9995 4BBD 299C3 299C9 9AD6 9B23 9C68
9C6C 9C6D 9C6E 9C6F 29F30 9E0B 9E0C 9E0D 9E0E 9E0F
9E10 9E11 9E12 2A1DF 9EA1 9ECC 9EF5 9F08 9F09 9F1D
9F1F 9F47 9F78 9F79 9F7A 9F7B 2A64A 9FA3
FDD0-281A
3536 571E 373B 5F60 6B1C 6C0E 704E 7064 7066 24AE2
7673 77DA 7C6F 7C70 25E0E 7CF3 8644 866A 883C 8B9A
8B9B 27E4F 8DB2 8EA6 8EA7 91C3 91C4 9474 9475 9476
9477 9478 9479 947A 28C09 28C1C 28C1D 974A 97C9 4BBE
9A60 9A61 9A62 9A63 9A65 9AD7 9C71 9C72 9C73 9C74
9C75 9C76 9E13 9E14 2A1F5 9EF6 9F0A 2A65B 9FA4 9FA5
FDD0-281B
7065 7067 7068 24177 2417A 72AB 7CF7 7E9C 7E9D 864A
883D 883E 883F 897D 8B9C 8B9D 8B9E 8C53 8C9C 8EA9
8EAA 8EC9 8F65 91C5 947B 947C 947D 947E 28C23 28C26
974B 974C 974D 974E 9873 9874 98CC 98CD 98DD 9960
9961 99AB 9A64 9A66 9A67 9B24 9B2E 9B30 9C77 9C78
9E15 9E16 9E17 9EF7 9F48
FDD0-281C
56D6 6207 22EB3 2331F 6B1E 6B1F 7227 24AE9 3FDC 7674
27175 864C 8C54 8EA8 28207 947F 9480 9481 9482 28C2B
28C30 96E7 4BC0 9A68 9A69 9E18 9E19 9E1A 2A220 9EA2
9EF8 9F3A 9F7C 9F7D 9F9E
FDD0-281D
7228 7E9E 864B 8B9F 4979 9483 9484 974F 9A6A 9B31
9C79 9E1B 9E1C 9EB7
FDD0-281E
53B5 7675 4190 7C71 4585 28C39 97CA 9962 9A6B 29C73
9C7A 9E1D 9E1E 2A233 4D91
FDD0-281F
7069 275A3 4D10 9EA3
FDD0-2820
706A 7C72 28C3B 9F96
FDD0-2821
2193B 7229 9C7B 9EA4 9F97
FDD0-2823
9F7E
FDD0-2824
9F49
FDD0-2827
9750
FDD0-2830
9F98
__END__

=head1 NAME

Unicode::Collate::CJK::Stroke - weighting CJK Unified Ideographs
for Unicode::Collate

=head1 SYNOPSIS

    use Unicode::Collate;
    use Unicode::Collate::CJK::Stroke;

    my $collator = Unicode::Collate->new(
        overrideCJK => \&Unicode::Collate::CJK::Stroke::weightStroke
    );

=head1 DESCRIPTION

C<Unicode::Collate::CJK::Stroke> provides C<weightStroke()>,
that is adequate for C<overrideCJK> of C<Unicode::Collate>
and makes tailoring of CJK Unified Ideographs in the order
of CLDR's stroke ordering.

=head1 CAVEAT

The stroke ordering includes some characters that are not
CJK Unified Ideographs and can't utilize C<weightStroke()>
for collation. For them, use C<entry> instead.

=head1 SEE ALSO

=over 4

=item CLDR - Unicode Common Locale Data Repository

L<http://cldr.unicode.org/>

=item Unicode Locale Data Markup Language (LDML) - UTS #35

L<http://www.unicode.org/reports/tr35/>

=item L<Unicode::Collate>

=item L<Unicode::Collate::Locale>

=back

=cut
