package Text::ASCII::Convert;
use strict;
use warnings FATAL => 'all';
use Encode qw(decode);
use base 'Exporter';
our @EXPORT = qw(convert_to_ascii);
our @EXPORT_OK = qw(convert_to_ascii);
our $VERSION = '0.30';
my %char_map;

=encoding utf8

=head1 NAME

Text::ASCII::Convert - Perl module to convert non-ASCII characters to their ASCII equivalents

=head1 SYNOPSIS

  use Text::ASCII::Convert;

  print convert_to_ascii("Ýou hãve a nèw vòice-mãil");

  # prints "You have a new voice-mail"

=head1 DESCRIPTION

This module attempts to convert non-ASCII characters in a string to their closet ASCII homoglyph. The input
can be a string of Unicode characters or a string of UTF-8 octets. The output is always a string of ASCII characters
in the range 0x00 to 0x7F.

This is most useful for catching spam that uses non-ASCII characters to obfuscate words. For example,

    Ýou hãve a nèw vòice-mãil
    You havé Reꞓeìved an Enꞓryptéd Company Maíl

would be converted to

    You have a new voice-mail
    You have ReCeived an EnCrypted Company Mail

Unlike other transliteration software, this plugin converts non-ASCII characters
to their ASCII equivalents based on appearance instead of meaning. For example, the
German eszett character 'ß' is converted to the Roman letter 'B' instead of 'ss'
because it resembles a 'B' in appearance. Likewise, the Greek letter Sigma ('Σ') is
converted to 'E' and a lower case Omega ('ω') is converted to 'w' even though these
letters have different lexical meanings.

Not all non-ASCII characters are converted. For example, the Japanese Hiragana
character 'あ' is not converted because it does not resemble any ASCII character.
Characters that have no ASCII equivalent are replaced by spaces. To avoid long runs
of spaces, multiple spaces are collapsed into a single space. For example,

    Find 💋💘Singles💋💘 in your Area

would be converted to

    Find Singles in your Area

The plugin also removes zero-width characters such as the zero-width
space (U+200B) and zero-width non-joiner (U+200C) that are often used to
obfuscate words.

Control characters such as tabs, newlines, and carriage returns are retained.

=head1 AUTHORS

Kent Oyer <kent@mxguardian.net>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2023 MXGuardian LLC

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the LICENSE
file included with this distribution for more information.

You should have received a copy of the GNU General Public License
along with this program.  If not, see https://www.gnu.org/licenses/.

=cut

UNITCHECK {
    # build character map from __DATA__ section
    while (<DATA>) {
        chomp;
        my ($key,$value) = split /\s+/;
        my $ascii = join('', map { chr(hex($_)) } split /\+/, $value);
        $char_map{chr(hex($key))} = $ascii;
    }
    close DATA;
};

# Converts a string of Unicode characters (or UTF-8 encoded bytes) to a string of ASCII characters
# in the range 0x00 to 0x7F. Non-ASCII characters are replaced with their ASCII equivalents.
# Zero-width characters and combining marks are removed. Multiple spaces are collapsed into a single space.
#
sub convert_to_ascii {
    my $str = is_valid_utf_8($_[0]) ? decode('UTF-8', $_[0]) : $_[0];
    # remove zero-width characters and combining marks
    $str =~ s/[\xAD\x{034F}\x{200B}-\x{200F}\x{202A}\x{202B}\x{202C}\x{2060}\x{FEFF}]|\p{Combining_Mark}//g;
    # replace non-ascii characters with ascii equivalents
    $str =~ s/([^[:ascii:]])/defined($char_map{$1})?$char_map{$1}:' '/eg;
    # collapse spaces
    $str =~ s/\x{20}+/ /g;
    return $str;
}

# returns true if the provided string of octets represents a syntactically
# valid UTF-8 string, otherwise a false is returned.
# Copied from Mail::SpamAssassin::Util::is_valid_utf8
#
sub is_valid_utf_8 {
    return undef if !defined $_[0];
    #
    # RFC 6532: UTF8-non-ascii = UTF8-2 / UTF8-3 / UTF8-4
    # RFC 3629 section 4: Syntax of UTF-8 Byte Sequences
    #   UTF8-char   = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4
    #   UTF8-1      = %x00-7F
    #   UTF8-2      = %xC2-DF UTF8-tail
    #   UTF8-3      = %xE0 %xA0-BF UTF8-tail /
    #                 %xE1-EC 2( UTF8-tail ) /
    #                 %xED %x80-9F UTF8-tail /
    #                   # U+D800..U+DFFF are utf16 surrogates, not legal utf8
    #                 %xEE-EF 2( UTF8-tail )
    #   UTF8-4      = %xF0 %x90-BF 2( UTF8-tail ) /
    #                 %xF1-F3 3( UTF8-tail ) /
    #                 %xF4 %x80-8F 2( UTF8-tail )
    #   UTF8-tail   = %x80-BF
    #
    # loose variant:
    #   [\x00-\x7F] | [\xC0-\xDF][\x80-\xBF] |
    #   [\xE0-\xEF][\x80-\xBF]{2} | [\xF0-\xF4][\x80-\xBF]{3}
    #
    $_[0] =~ /^ (?: [\x00-\x7F] |
                  [\xC2-\xDF] [\x80-\xBF] |
                  \xE0 [\xA0-\xBF] [\x80-\xBF] |
                  [\xE1-\xEC] [\x80-\xBF]{2} |
                  \xED [\x80-\x9F] [\x80-\xBF] |
                  [\xEE-\xEF] [\x80-\xBF]{2} |
                  \xF0 [\x90-\xBF] [\x80-\xBF]{2} |
                  [\xF1-\xF3] [\x80-\xBF]{3} |
                  \xF4 [\x80-\x8F] [\x80-\xBF]{2} )* \z/xs ? 1 : 0;
}


1;

__DATA__
00A0 20
00A2 63
00A3 45
00A5 59
00A9 28+43+29
00AA 61
00AE 28+52+29
00B2 32
00B3 33
00B4 27
00B5 75
00B7 2E
00B8 2C
00B9 31
00BA 6F
00BC 31+2F+34
00BD 31+2F+32
00BE 33+2F+34
00C0 41
00C1 41
00C2 41
00C3 41
00C4 41
00C5 41
00C6 41+45
00C7 43
00C8 45
00C9 45
00CA 45
00CB 45
00CC 49
00CD 49
00CE 49
00CF 49
00D0 44
00D1 4E
00D2 4F
00D3 4F
00D4 4F
00D5 4F
00D6 4F
00D7 78
00D8 4F
00D9 55
00DA 55
00DB 55
00DC 55
00DD 59
00DF 42
00E0 61
00E1 61
00E2 61
00E3 61
00E4 61
00E5 61
00E6 61+65
00E7 63
00E8 65
00E9 65
00EA 65
00EB 65
00EC 69
00ED 69
00EE 69
00EF 69
00F0 6F
00F1 6E
00F2 6F
00F3 6F
00F4 6F
00F5 6F
00F6 6F
00F8 6F
00F9 75
00FA 75
00FB 75
00FC 75
00FD 79
00FF 79
0100 41
0101 61
0102 41
0103 61
0104 41
0105 61
0106 43
0107 63
0108 43
0109 63
010A 43
010B 63
010C 43
010D 63
010E 44
010F 64
0110 44
0111 64
0112 45
0113 65
0114 45
0115 65
0116 45
0117 65
0118 45
0119 65
011A 45
011B 65
011C 47
011D 67
011E 47
011F 67
0120 47
0121 67
0122 47
0123 67
0124 48
0125 68
0126 48
0127 68
0128 49
0129 69
012A 49
012B 69
012C 49
012D 69
012E 49
012F 69
0130 49
0131 69
0132 49+4A
0133 69+6A
0134 4A
0135 6A
0136 4B
0137 6B
0138 6B
0139 4C
013A 6C
013B 4C
013C 6C
013D 4C
013E 6C
013F 4C
0140 6C
0141 4C
0142 6C
0143 4E
0144 6E
0145 4E
0146 6E
0147 4E
0148 6E
0149 6E
014A 4E
014B 6E
014C 4F
014D 6F
014E 4F
014F 6F
0150 4F
0151 6F
0152 4F+45
0153 6F+65
0154 52
0155 72
0156 52
0157 72
0158 52
0159 72
015A 53
015B 73
015C 53
015D 73
015E 53
015F 73
0160 53
0161 73
0162 54
0163 74
0164 54
0165 74
0166 54
0167 74
0168 55
0169 75
016A 55
016B 75
016C 55
016D 75
016E 55
016F 75
0170 55
0171 75
0172 55
0173 75
0174 57
0175 77
0176 59
0177 79
0178 59
0179 5A
017A 7A
017B 5A
017C 7A
017D 5A
017E 7A
017F 66
0180 62
0181 42
0182 62
0183 62
0184 62
0185 62
0187 43
0188 63
0189 44
018A 44
018B 61
018C 61
018D 67
018E 33
0190 45
0191 46
0192 66
0193 47
0194 56
0196 6C
0197 49
0198 4B
0199 6B
019A 6C
019C 57
019D 4E
019E 6E
019F 4F
01A0 4F
01A1 6F
01A4 50
01A5 70
01A6 52
01A7 32
01A8 32
01A9 45
01AB 74
01AC 54
01AD 74
01AE 54
01AF 55
01B0 75
01B1 55
01B2 56
01B3 59
01B4 79
01B5 5A
01B6 7A
01B7 33
01B8 45
01B9 45
01BB 32
01BC 35
01BD 73
01BF 70
01C0 6C
01C3 21
01C4 44+5A
01C5 44+7A
01C6 64+7A
01C7 4C+4A
01C8 4C+6A
01C9 6C+6A
01CA 4E+4A
01CB 4E+6A
01CC 6E+6A
01CD 41
01CE 61
01CF 49
01D0 69
01D1 4F
01D2 6F
01D3 55
01D4 75
01D5 55
01D6 75
01D7 55
01D8 75
01D9 55
01DA 75
01DB 55
01DC 75
01DE 41
01DF 61
01E0 41
01E1 61
01E2 41+45
01E3 61+65
01E4 47
01E5 67
01E6 47
01E7 67
01E8 4B
01E9 6B
01EA 51
01EB 51
01EC 51
01ED 51
01EE 33
01EF 33
01F0 6A
01F1 44+5A
01F2 44+7A
01F3 64+7A
01F4 47
01F5 67
01F6 48
01F7 50
01F8 4E
01F9 6E
01FA 41
01FB 61
01FC 41+45
01FD 61+65
01FE 4F
01FF 6F
0200 41
0201 61
0202 41
0203 61
0204 45
0205 65
0206 45
0207 65
0208 49
0209 69
020A 49
020B 69
020C 4F
020D 6F
020E 4F
020F 6F
0210 52
0211 72
0212 52
0213 72
0214 55
0215 75
0216 55
0217 75
0218 53
0219 73
021A 54
021B 74
021C 33
021D 33
021E 48
021F 68
0220 6E
0221 64
0222 38
0223 38
0224 5A
0225 7A
0226 41
0227 61
0228 45
0229 65
022A 4F
022B 6F
022C 4F
022D 6F
022E 4F
022F 6F
0230 4F
0231 6F
0232 59
0233 79
0234 6C
0235 6E
0236 74
0237 6A
0238 64+62
0239 71+70
023A 41
023B 43
023C 63
023D 4C
023E 54
023F 73
0240 7A
0241 3F
0242 32
0243 42
0244 55
0246 45
0247 65
0248 4A
0249 6A
024A 51
024B 71
024C 52
024D 72
024E 59
024F 79
0251 61
0253 62
0255 63
0256 64
0257 64
025B 65
025C 33
025D 33
025E 67
025F 6A
0260 67
0261 67
0262 47
0263 79
0265 75
0266 68
0267 68
0268 69
0269 69
026A 49
026B 6C
026C 6C
026D 6C
026F 77
0270 77
0271 6D
0272 6E
0273 6E
0274 4E
0275 6F
0276 4F+45
027C 72
027D 72
027E 72
0280 52
0282 73
0284 66
0288 74
0289 75
028B 75
028D 6D
028F 79
0290 7A
0291 7A
0292 33
0293 33
0294 3F
0297 43
0298 4F
0299 42
029B 47
029C 48
029D 6A
029F 4C
02A0 71
02A1 3F
02B0 68
02B1 68
02B2 6A
02B3 72
02B7 77
02B8 79
02B9 27
02BA 22
02BB 27
02BC 27
02BD 27
02BE 27
02C2 3C
02C3 3E
02C4 5E
02C6 5E
02CA 27
02CB 27
02D0 3A
02D7 2D
02DB 69
02DC 7E
02DD 22
02E0 79
02E1 6C
02E2 73
02E3 78
02EE 22
02F4 27
02F6 22
02F8 3A
0372 74
0373 74
0374 27
0376 4E
0377 6E
037A 69
037C 63
037E 3B
037F 4A
0384 27
0386 41
0387 2E
0388 45
0389 48
038A 49
038C 4F
038E 59
0390 69
0391 41
0392 42
0395 45
0396 5A
0397 48
0399 49
039A 4B
039C 4D
039D 4E
039F 4F
03A1 50
03A3 45
03A4 54
03A5 59
03A6 4F
03A7 58
03A8 57
03AA 49
03AB 59
03AC 61
03AD 65
03AE 6E
03AF 69
03B0 75
03B1 61
03B2 42
03B3 79
03B4 64
03B5 65
03B6 7A
03B7 6E
03B8 6F
03B9 69
03BA 6B
03BC 75
03BD 76
03BE 65
03BF 6F
03C0 6E
03C1 70
03C2 63
03C3 6F
03C4 74
03C5 75
03C7 78
03C8 77
03C9 77
03CA 69
03CB 75
03CC 6F
03CD 75
03CE 77
03CF 4B
03D0 42
03D1 6F
03D2 59
03D3 59
03D4 59
03D6 77
03D7 6B
03D8 4F
03D9 6F
03DA 43
03DB 63
03DC 46
03DD 66
03DE 73
03E2 77
03E3 77
03E4 34
03E5 34
03E6 62
03E7 73
03E8 32
03E9 32
03EC 36
03ED 36
03EE 74
03EF 74
03F0 6B
03F1 70
03F2 63
03F3 6A
03F4 4F
03F5 65
03F9 43
03FA 4D
03FB 6D
03FC 70
03FE 43
0400 45
0401 45
0405 53
0406 49
0407 49
0408 4A
040C 4B
040D 4E
0410 41
0412 42
0415 45
0417 33
041A 4B
041C 4D
041D 48
041E 4F
0420 50
0421 43
0422 54
0425 58
042C 62
0430 61
0431 36
0432 42
0433 72
0434 41
0435 65
0437 33
0438 6E
0439 6E
043A 6B
043C 4D
043D 48
043E 6F
043F 6E
0440 70
0441 63
0442 54
0443 79
0445 78
0446 75
0447 34
0448 77
0449 77
044A 62
044C 62
0450 65
0451 65
0453 72
0454 65
0455 73
0456 69
0457 69
0458 6A
045C 6B
045D 6E
045E 79
0461 77
0463 62
0474 56
0475 76
0476 56
0477 76
048D 62
048F 70
0491 72
0493 66
0498 33
0499 33
049B 4B
049C 4B
049D 4B
049F 6B
04A1 4B
04A3 48
04A4 48
04A5 48
04AB 63
04AD 54
04AE 59
04AF 79
04B1 59
04B3 78
04B8 34
04B9 34
04BB 68
04BC 65
04BD 65
04BE 65
04BF 65
04C0 6C
04C4 4B
04C8 48
04CA 48
04CE 4D
04CF 69
04D0 41
04D1 61
04D2 41
04D3 61
04D4 41+45
04D5 61+65
04D6 45
04D7 65
04DE 33
04DF 33
04E0 33
04E1 33
04E2 4E
04E3 6E
04E4 4E
04E5 6E
04E6 4F
04E7 6F
04EC 33
04ED 33
04EE 59
04EF 79
04F0 59
04F1 79
04F2 59
04F3 79
0501 64
050B 48
050C 47
050D 47
050F 54
0511 45
051B 71
051C 57
051D 77
051E 4B
051F 4B
0527 68
0529 48
0531 55
0536 32
0545 33
054D 55
054F 53
0555 4F
055A 27
055D 27
0561 77
0563 71
0566 71
0570 68
0575 6A
0578 6E
057C 6E
057D 75
0581 67
0584 70
0585 6F
0587 75
0589 3A
05C3 3A
05D0 4E
05D5 69
05D8 76
05D9 27
05DF 6C
05E1 6F
05E2 59
05F0 6C+6C
05F2 22
05F3 27
05F4 22
0609 25
060A 25
0622 49
0623 49
0625 49
0627 49
0639 45
0660 2E
0665 6F
066A 25
066B 2C
066D 2A
0671 49
0672 49
0673 49
06A0 45
06AC 4A
06AE 4A
06B6 4A
06B7 4A
06B8 4A
06B9 55
06BD 55
06D4 2E
06F0 2E
0701 2E
0702 2E
0703 3A
0704 3A
075D 45
075E 45
075F 45
076B 6A
07C0 30
07CA 6C
07F4 27
07F5 27
07FA 5F
0903 3A
0966 30
097D 3F
09E6 30
09EA 38
09ED 39
0A66 30
0A67 39
0A6A 38
0A83 3A
0AE6 30
0B03 38
0B20 4F
0B66 30
0B68 39
0BD0 43
0BE6 30
0C02 6F
0C66 30
0C82 6F
0CE6 30
0D02 6F
0D20 6F
0D66 30
0D82 6F
0E50 30
0ED0 30
101D 6F
1040 30
10A6 42
10AF 34
10CD 5A
10E7 79
10FD 53
13A0 44
13A1 52
13A2 54
13A5 69
13A6 53
13A9 59
13AA 41
13AB 4A
13AC 45
13AF 41
13B3 57
13B6 47
13B7 4D
13BB 48
13BD 59
13C0 47
13C2 68
13C3 5A
13C6 49
13C7 77
13CE 34
13CF 62
13D2 52
13D4 57
13D5 53
13D9 56
13DA 53
13DE 4C
13DF 43
13E2 50
13E3 43
13E4 56
13E5 68
13E6 4B
13E7 64
13E9 47
13EE 36
13F0 42
13F3 47
13F4 42
13F5 47
142F 56
144C 55
146D 50
146F 64
148D 4A
14AA 4C
14BF 32
1541 78
157C 48
157D 78
1587 52
15AF 62
15B4 46
15C5 41
15DE 44
15EA 44
15F0 4D
15F7 42
1616 32
1633 43
1640 55
166D 58
166E 78
1680 20
1735 2F
1803 3A
1809 3A
180E 20
1D00 41
1D01 41+45
1D03 42
1D04 43
1D05 44
1D07 45
1D09 21
1D0A 4A
1D0B 4B
1D0C 4C
1D0D 4D
1D0E 4E
1D0F 4F
1D11 4F
1D13 4F
1D18 50
1D19 52
1D1A 52
1D1B 54
1D1C 55
1D20 56
1D21 57
1D22 5A
1D26 72
1D28 6E
1D2C 41
1D2D 41+45
1D2E 42
1D30 44
1D31 45
1D33 47
1D34 48
1D35 49
1D36 4A
1D37 4B
1D38 4C
1D39 4D
1D3A 4E
1D3C 4F
1D3D 38
1D3E 50
1D3F 52
1D40 54
1D41 55
1D42 57
1D43 61
1D45 61
1D47 62
1D48 64
1D49 65
1D4B 65
1D4C 33
1D4D 67
1D4F 6B
1D50 6D
1D51 6E
1D52 6F
1D56 70
1D57 74
1D58 75
1D5A 77
1D5B 76
1D5D 42
1D5E 79
1D5F 64
1D61 78
1D62 69
1D63 72
1D64 75
1D65 76
1D66 42
1D67 79
1D68 70
1D6A 78
1D6C 62
1D6D 64
1D6E 66
1D6F 6D
1D70 6E
1D71 70
1D72 66
1D73 66
1D74 73
1D75 74
1D76 7A
1D7B 49
1D7D 70
1D7E 75
1D80 62
1D81 64
1D82 66
1D83 67
1D85 6C
1D86 6D
1D87 6E
1D88 70
1D89 72
1D8A 73
1D8C 79
1D8D 78
1D8E 7A
1D8F 61
1D91 64
1D92 65
1D96 69
1D99 75
1D9C 63
1D9D 63
1D9E 6F
1D9F 33
1DA0 66
1DA1 6A
1DA2 67
1DA3 75
1DA4 69
1DA5 69
1DA6 69
1DA7 49
1DA8 6A
1DA9 6C
1DAA 6C
1DAB 4C
1DAC 6D
1DAD 77
1DAE 6E
1DAF 6E
1DB0 4E
1DB1 6F
1DB3 73
1DB5 74
1DB6 75
1DB8 75
1DB9 75
1DBB 7A
1DBC 7A
1DBD 7A
1DBE 33
1DBF 6F
1E00 41
1E01 61
1E02 42
1E03 62
1E04 42
1E05 62
1E06 42
1E07 62
1E08 43
1E09 63
1E0A 44
1E0B 64
1E0C 44
1E0D 64
1E0E 44
1E0F 64
1E10 44
1E11 64
1E12 44
1E13 64
1E14 45
1E15 65
1E16 45
1E17 65
1E18 45
1E19 65
1E1A 45
1E1B 65
1E1C 45
1E1D 65
1E1E 46
1E1F 66
1E20 47
1E21 67
1E22 48
1E23 68
1E24 48
1E25 68
1E26 48
1E27 68
1E28 48
1E29 68
1E2A 48
1E2B 68
1E2C 49
1E2D 69
1E2E 49
1E2F 69
1E30 4B
1E31 6B
1E32 4B
1E33 6B
1E34 4B
1E35 6B
1E36 4C
1E37 6C
1E38 4C
1E39 6C
1E3A 4C
1E3B 6C
1E3C 4C
1E3D 6C
1E3E 4D
1E3F 6D
1E40 4D
1E41 6D
1E42 4D
1E43 6D
1E44 4E
1E45 6E
1E46 4E
1E47 6E
1E48 4E
1E49 6E
1E4A 4E
1E4B 6E
1E4C 4F
1E4D 6F
1E4E 4F
1E4F 6F
1E50 4F
1E51 6F
1E52 4F
1E53 6F
1E54 50
1E55 70
1E56 50
1E57 70
1E58 52
1E59 72
1E5A 52
1E5B 72
1E5C 52
1E5D 72
1E5E 52
1E5F 72
1E60 53
1E61 73
1E62 53
1E63 73
1E64 53
1E65 73
1E66 53
1E67 73
1E68 53
1E69 73
1E6A 54
1E6B 74
1E6C 54
1E6D 74
1E6E 54
1E6F 74
1E70 54
1E71 74
1E72 55
1E73 75
1E74 55
1E75 75
1E76 55
1E77 75
1E78 55
1E79 75
1E7A 55
1E7B 75
1E7C 56
1E7D 76
1E7E 56
1E7F 76
1E80 57
1E81 77
1E82 57
1E83 77
1E84 57
1E85 77
1E86 57
1E87 77
1E88 57
1E89 77
1E8A 58
1E8B 78
1E8C 58
1E8D 78
1E8E 59
1E8F 79
1E90 5A
1E91 7A
1E92 5A
1E93 7A
1E94 5A
1E95 7A
1E96 68
1E97 74
1E98 77
1E99 79
1E9A 61
1E9B 73
1E9D 66
1EA0 41
1EA1 61
1EA2 41
1EA3 61
1EA4 41
1EA5 61
1EA6 41
1EA7 61
1EA8 41
1EA9 61
1EAA 41
1EAB 61
1EAC 41
1EAD 61
1EAE 41
1EAF 61
1EB0 41
1EB1 61
1EB2 41
1EB3 61
1EB4 41
1EB5 61
1EB6 41
1EB7 61
1EB8 45
1EB9 65
1EBA 45
1EBB 65
1EBC 45
1EBD 65
1EBE 45
1EBF 65
1EC0 45
1EC1 65
1EC2 45
1EC3 65
1EC4 45
1EC5 65
1EC6 45
1EC7 65
1EC8 49
1EC9 69
1ECA 49
1ECB 69
1ECC 4F
1ECD 6F
1ECE 4F
1ECF 6F
1ED0 4F
1ED1 6F
1ED2 4F
1ED3 6F
1ED4 4F
1ED5 6F
1ED6 4F
1ED7 6F
1ED8 4F
1ED9 6F
1EDA 4F
1EDB 6F
1EDC 4F
1EDD 6F
1EDE 4F
1EDF 6F
1EE0 4F
1EE1 6F
1EE2 4F
1EE3 6F
1EE4 55
1EE5 75
1EE6 55
1EE7 75
1EE8 55
1EE9 75
1EEA 55
1EEB 75
1EEC 55
1EED 75
1EEE 55
1EEF 75
1EF0 55
1EF1 75
1EF2 59
1EF3 79
1EF4 59
1EF5 79
1EF6 59
1EF7 79
1EF8 59
1EF9 79
1EFE 59
1EFF 79
1F00 61
1F01 61
1F02 61
1F03 61
1F04 61
1F05 61
1F06 61
1F07 61
1F08 41
1F09 41
1F0A 41
1F0B 41
1F0C 41
1F0D 41
1F0E 41
1F0F 41
1F10 65
1F11 65
1F12 65
1F13 65
1F14 65
1F15 65
1F18 45
1F19 45
1F1A 45
1F1B 45
1F1C 45
1F1D 45
1F20 6E
1F21 6E
1F22 6E
1F23 6E
1F24 6E
1F25 6E
1F26 6E
1F27 6E
1F28 48
1F29 48
1F2A 48
1F2B 48
1F2C 48
1F2D 48
1F2E 48
1F2F 48
1F30 69
1F31 69
1F32 69
1F33 69
1F34 69
1F35 69
1F36 69
1F37 69
1F38 49
1F39 49
1F3A 49
1F3B 49
1F3C 49
1F3D 49
1F3E 49
1F3F 49
1F40 6F
1F41 6F
1F42 6F
1F43 6F
1F44 6F
1F45 6F
1F48 4F
1F49 4F
1F4A 4F
1F4B 4F
1F4C 4F
1F4D 4F
1F50 75
1F51 75
1F52 75
1F53 75
1F54 75
1F55 75
1F56 75
1F57 75
1F59 59
1F5B 59
1F5D 59
1F5F 59
1F60 77
1F61 77
1F62 77
1F63 77
1F64 77
1F65 77
1F66 77
1F67 77
1F70 61
1F71 61
1F72 65
1F73 65
1F74 6E
1F75 6E
1F76 69
1F77 69
1F78 6F
1F79 6F
1F7A 75
1F7B 75
1F7C 77
1F7D 77
1F80 61
1F81 61
1F82 61
1F83 61
1F84 61
1F85 61
1F86 61
1F87 61
1F88 41
1F89 41
1F8A 41
1F8B 41
1F8C 41
1F8D 41
1F8E 41
1F8F 41
1F90 6E
1F91 6E
1F92 6E
1F93 6E
1F94 6E
1F95 6E
1F96 6E
1F97 6E
1F98 48
1F99 48
1F9A 48
1F9B 48
1F9C 48
1F9D 48
1F9E 48
1F9F 48
1FA0 77
1FA1 77
1FA2 77
1FA3 77
1FA4 77
1FA5 77
1FA6 77
1FA7 77
1FB0 61
1FB1 61
1FB2 61
1FB3 61
1FB4 61
1FB6 61
1FB7 61
1FB8 41
1FB9 41
1FBA 41
1FBB 41
1FBC 41
1FBD 27
1FBE 69
1FBF 27
1FC0 7E
1FC2 6E
1FC3 6E
1FC4 6E
1FC6 6E
1FC7 6E
1FC8 45
1FC9 45
1FCA 48
1FCB 48
1FCC 48
1FCD 22
1FCE 22
1FD0 69
1FD1 69
1FD2 69
1FD3 69
1FD6 69
1FD7 69
1FD8 49
1FD9 49
1FDA 49
1FDB 49
1FDD 22
1FDE 22
1FE0 75
1FE1 75
1FE2 75
1FE3 75
1FE4 70
1FE5 70
1FE6 75
1FE7 75
1FE8 59
1FE9 59
1FEA 59
1FEB 59
1FEC 50
1FEF 27
1FF2 77
1FF3 77
1FF4 77
1FF6 77
1FF7 77
1FF8 4F
1FF9 4F
1FFD 27
1FFE 27
2000 20
2001 20
2002 20
2003 20
2004 20
2005 20
2006 20
2007 20
2008 20
2009 20
200A 20
2010 2D
2011 2D
2012 2D
2013 2D
2018 27
2019 27
201A 2C
201B 27
201C 22
201D 22
201F 22
2022 2A
2023 3E
2024 2E
2025 2E+2E
2026 2E+2E+2E
2028 20
2029 20
202F 20
2030 25
2032 27
2033 22
2034 27+27+27
2035 27
2036 22
2037 27+27+27
2039 3C
203A 3E
203C 21+21
2041 2F
2043 2D
2044 2F
2047 3F+3F
2048 3F+21
2049 21+3F
204E 2A
2052 25
2053 7E
2057 27+27+27+27
205A 3A
205F 20
2070 30
2071 69
2074 34
2075 35
2076 36
2077 37
2078 38
2079 39
207A 2B
207B 2D
207C 3D
207D 28
207E 29
207F 6E
2080 30
2081 31
2082 32
2083 33
2084 34
2085 35
2086 36
2087 37
2088 38
2089 39
208A 2B
208B 2D
208C 3D
208D 28
208E 29
2090 61
2091 65
2092 6F
2093 78
2095 68
2096 6B
2097 6C
2098 6D
2099 6E
209A 70
209B 73
209C 74
20A8 52+73
2100 25
2101 25
2102 43
2103 43
2105 25
2106 25
2107 45
2109 4F+46
210A 67
210B 48
210C 48
210D 48
210E 68
210F 68
2110 4A
2111 4A
2112 4C
2113 6C
2115 4E
2116 4E+6F
2117 28+50+29
2118 50
2119 50
211A 51
211B 52
211C 52
211D 52
2120 28+53+4D+29
2121 54+45+4C
2122 28+54+4D+29
2124 5A
2128 33
212A 4B
212B 41
212C 42
212D 43
212E 65
212F 65
2130 45
2131 46
2133 4D
2134 6F
2139 69
213B 46+41+58
213C 6E
213D 79
2140 45
2145 44
2146 64
2147 65
2148 69
2149 6A
2150 31+2F+37
2151 31+2F+39
2152 31+2F+31+30
2153 31+2F+33
2154 32+2F+33
2155 31+2F+35
2156 32+2F+35
2157 33+2F+35
2158 34+2F+35
2159 31+2F+36
215A 35+2F+36
215B 31+2F+38
215C 33+2F+38
215D 35+2F+38
215E 37+2F+38
215F 31+2F
2160 49
2161 49+49
2162 49+49+49
2163 49+56
2164 56
2165 56+49
2166 56+49+49
2167 56+49+49+49
2168 49+58
2169 58
216A 58+49
216B 58+49+49
216C 4C
216D 43
216E 44
216F 4D
2170 69
2171 69+69
2172 69+69+69
2173 69+76
2174 76
2175 76+69
2176 76+69+69
2177 76+69+69+69
2178 69+78
2179 78
217A 78+69
217B 78+69+69
217C 4C
217D 63
217E 64
217F 6D
2189 30+2F+33
2208 45
2209 45
220A 45
2211 45
2212 2D
2215 2F
2216 5C
2217 2A
2219 2E
221F 4C
2223 6C
2224 6C
2228 76
222B 53
222C 53+53
222D 53+53+53
2236 3A
223C 7E
2241 7E
2260 21+3D
226E 3C
226F 3E
2282 43
2284 43
2295 28+2B+29
2296 28+2D+29
2297 28+78+29
2298 28+2F+29
2299 28+2E+29
229B 28+2A+29
229C 28+3D+29
229D 28+2D+29
229E 5B+2B+5D
229F 5B+2D+5D
22A0 5B+78+5D
22A1 5B+2E+5D
22C1 76
22C3 55
22C5 2E
22FF 45
2329 3C
232A 3E
2373 69
2374 70
2375 77
2379 77
237A 61
23B8 4C
2460 28+31+29
2461 28+32+29
2462 28+33+29
2463 28+34+29
2464 28+35+29
2465 28+36+29
2466 28+37+29
2467 28+38+29
2468 28+39+29
2469 28+31+30+29
246A 28+31+31+29
246B 28+31+32+29
246C 28+31+33+29
246D 28+31+34+29
246E 28+31+35+29
246F 28+31+36+29
2470 28+31+37+29
2471 28+31+38+29
2472 28+31+39+29
2473 28+32+30+29
2474 28+31+29
2475 28+32+29
2476 28+33+29
2477 28+34+29
2478 28+35+29
2479 28+36+29
247A 28+37+29
247B 28+38+29
247C 28+39+29
247D 28+31+30+29
247E 28+31+31+29
247F 28+31+32+29
2480 28+31+33+29
2481 28+31+34+29
2482 28+31+35+29
2483 28+31+36+29
2484 28+31+37+29
2485 28+31+38+29
2486 28+31+39+29
2487 28+32+30+29
2488 31+2E
2489 32+2E
248A 33+2E
248B 34+2E
248C 35+2E
248D 36+2E
248E 37+2E
248F 38+2E
2490 39+2E
2491 31+30+2E
2492 31+31+2E
2493 31+32+2E
2494 31+33+2E
2495 31+34+2E
2496 31+35+2E
2497 31+36+2E
2498 31+37+2E
2499 31+38+2E
249A 31+39+2E
249B 32+30+2E
249C 28+61+29
249D 28+62+29
249E 28+63+29
249F 28+64+29
24A0 28+65+29
24A1 28+66+29
24A2 28+67+29
24A3 28+68+29
24A4 28+69+29
24A5 28+6A+29
24A6 28+6B+29
24A7 28+6C+29
24A8 28+6D+29
24A9 28+6E+29
24AA 28+6F+29
24AB 28+70+29
24AC 28+71+29
24AD 28+72+29
24AE 28+73+29
24AF 28+74+29
24B0 28+75+29
24B1 28+76+29
24B2 28+77+29
24B3 28+78+29
24B4 28+79+29
24B5 28+7A+29
24B6 28+41+29
24B7 28+42+29
24B8 28+43+29
24B9 28+44+29
24BA 28+45+29
24BB 28+46+29
24BC 28+47+29
24BD 28+48+29
24BE 28+49+29
24BF 28+4A+29
24C0 28+4B+29
24C1 28+4C+29
24C2 28+4D+29
24C3 28+4E+29
24C4 28+4F+29
24C5 28+50+29
24C6 28+51+29
24C7 28+52+29
24C8 28+53+29
24C9 28+54+29
24CA 28+55+29
24CB 28+56+29
24CC 28+57+29
24CD 28+58+29
24CE 28+59+29
24CF 28+5A+29
24D0 28+61+29
24D1 28+62+29
24D2 28+63+29
24D3 28+64+29
24D4 28+65+29
24D5 28+66+29
24D6 28+67+29
24D7 28+68+29
24D8 28+69+29
24D9 28+6A+29
24DA 28+6B+29
24DB 28+6C+29
24DC 28+6D+29
24DD 28+6E+29
24DE 28+6F+29
24DF 28+70+29
24E0 28+71+29
24E1 28+72+29
24E2 28+73+29
24E3 28+74+29
24E4 28+75+29
24E5 28+76+29
24E6 28+77+29
24E7 28+78+29
24E8 28+79+29
24E9 28+7A+29
24EA 28+30+29
24EB 28+31+31+29
24EC 28+31+32+29
24ED 28+31+33+29
24EE 28+31+34+29
24EF 28+31+35+29
24F0 28+31+36+29
24F1 28+31+37+29
24F2 28+31+38+29
24F3 28+31+39+29
24F4 28+32+30+29
24F5 28+28+31+29+29
24F6 28+28+32+29+29
24F7 28+28+33+29+29
24F8 28+28+34+29+29
24F9 28+28+35+29+29
24FA 28+28+36+29+29
24FB 28+28+37+29+29
24FC 28+28+38+29+29
24FD 28+28+39+29+29
24FE 28+28+31+30+29+29
24FF 28+30+29
2571 2F
2573 78
25AE 4C
25AF 4C
25CB 4F
25CC 6F
25CF 2A
25D8 2A
25D9 4F
25E6 2A
25EF 4F
2613 58
2686 6F
2687 6F
26AA 4F
26AB 2A
26DD 5B+58+5D
26E3 4F
2731 2A
2732 2A
2753 3F
2754 3F
2755 21
2758 4C
2759 4C
275A 4C
275B 27
275C 27
275D 22
275E 22
2768 28
2769 29
276A 28
276B 29
276C 3C
276D 3E
276E 3C
276F 3E
2770 3C
2771 3E
2772 5B
2773 5D
2774 7B
2775 7D
2776 28+31+29
2777 28+32+29
2778 28+33+29
2779 28+34+29
277A 28+35+29
277B 28+36+29
277C 28+37+29
277D 28+38+29
277E 28+39+29
277F 28+31+30+29
2780 28+31+29
2781 28+32+29
2782 28+33+29
2783 28+34+29
2784 28+35+29
2785 28+36+29
2786 28+37+29
2787 28+38+29
2788 28+39+29
2789 28+31+30+29
278A 28+31+29
278B 28+32+29
278C 28+33+29
278D 28+34+29
278E 28+35+29
278F 28+36+29
2790 28+37+29
2791 28+38+29
2792 28+39+29
2793 28+31+30+29
27CE 5B+5E+5D
27CF 5B+76+5D
27D9 54
27E8 3C
27E9 3E
27EA 3C+3C
27EB 3E+3E
27EC 5B
27ED 5D
2801 2E
2802 2E
2804 2E
2810 2E
2820 2E
2840 2E
2880 2E
28C0 2E+2E
292B 78
292C 78
2981 2E
2991 3C
2992 3E
2997 5B
2998 5D
29B6 28+7C+29
29B7 28+7C+7C+29
29B8 28+5C+29
29BE 28+2A+29
29BF 28+2A+29
29C0 28+3C+29
29C1 28+3E+29
29C4 5B+2F+5D
29C5 5B+5C+5D
29C6 5B+2A+5D
29C7 5B+6F+5D
29F5 5C
29F8 2F
29F9 5C
29FC 3C
29FD 3E
2A0C 53+53+53+53
2A2F 78
2A74 3A+3A+3D
2A75 3D+3D
2A76 3D+3D+3D
2B24 2A
2B2F 4F
2B55 4F
2B58 4F
2B59 28+58+29
2BBE 28+58+29
2BBF 28+58+29
2C60 4C
2C61 6C
2C62 4C
2C63 50
2C64 52
2C65 61
2C66 74
2C67 48
2C68 68
2C69 4B
2C6A 6B
2C6B 5A
2C6C 7A
2C6D 61
2C6E 4D
2C71 76
2C72 57
2C73 77
2C74 76
2C78 65
2C7A 6F
2C7C 6A
2C7D 56
2C7E 53
2C7F 5A
2C82 42
2C83 42
2C84 72
2C85 72
2C88 45
2C89 45
2C8C 7A
2C8D 7A
2C8E 48
2C8F 48
2C90 4F
2C91 4F
2C92 49
2C93 49
2C94 4B
2C95 6B
2C98 4D
2C99 4D
2C9A 4E
2C9B 4E
2C9C 33
2C9D 33
2C9E 6F
2C9F 6F
2CA0 4E
2CA1 4E
2CA2 50
2CA3 70
2CA4 43
2CA5 63
2CA6 54
2CA7 54
2CA8 59
2CA9 59
2CAC 58
2CAD 58
2CB0 57
2CB1 57
2CBA 2D
2CC4 33
2CC5 33
2CC6 2F
2CCA 39
2CCC 33
2CD0 4C
2CD2 36
2D06 62
2D0F 34
2D38 56
2D39 45
2D3A 33
2D49 45
2D4A 49
2D4B 58
2D4C 23
2D4E 43
2D4F 49
2D51 21
2D54 4F
2D55 51
2D56 59
2D5C 74
2D5D 58
2D61 55
2D62 53
2E2C 3A+3A
2E31 2E
2E33 2E
2F02 5C
2F03 2F
3000 20
3003 22
3007 30
3008 3C
3009 3E
300A 3C+3C
300B 3E+3E
3014 5B
3015 5D
3018 5B
3019 5D
3033 2F
309B 22
30FB 2E
31D3 2F
31D4 5C
3248 5B+28+31+30+29+5D
3249 5B+28+32+30+29+5D
324A 5B+28+33+30+29+5D
324B 5B+28+34+30+29+5D
324C 5B+28+35+30+29+5D
324D 5B+28+36+30+29+5D
324E 5B+28+37+30+29+5D
324F 5B+28+38+30+29+5D
3250 50+54+45
3251 28+32+31+29
3252 28+32+32+29
3253 28+32+33+29
3254 28+32+34+29
3255 28+32+35+29
3256 28+32+36+29
3257 28+32+37+29
3258 28+32+38+29
3259 28+32+39+29
325A 28+33+30+29
325B 28+33+31+29
325C 28+33+32+29
325D 28+33+33+29
325E 28+33+34+29
325F 28+33+35+29
32B1 28+33+36+29
32B2 28+33+37+29
32B3 28+33+38+29
32B4 28+33+39+29
32B5 28+34+30+29
32B6 28+34+31+29
32B7 28+34+32+29
32B8 28+34+33+29
32B9 28+34+34+29
32BA 28+34+35+29
32BB 28+34+36+29
32BC 28+34+37+29
32BD 28+34+38+29
32BE 28+34+39+29
32BF 28+35+30+29
32C0 31
32C1 32
32C2 33
32C3 34
32C4 35
32C5 36
32C6 37
32C7 38
32C8 39
32C9 31+30
32CA 31+31
32CB 31+32
32CC 48+67
32CD 65+72+67
32CE 65+56
32CF 4C+54+44
3358 30
3359 31
335A 32
335B 33
335C 34
335D 35
335E 36
335F 37
3360 38
3361 39
3362 31+30
3363 31+31
3364 31+32
3365 31+33
3366 31+34
3367 31+35
3368 31+36
3369 31+37
336A 31+38
336B 31+39
336C 32+30
336D 32+31
336E 32+32
336F 32+33
3370 32+34
3371 68+50+61
3372 64+61
3373 41+55
3374 62+61+72
3375 6F+56
3376 70+63
3377 64+6D
3378 64+6D+32
3379 64+6D+33
337A 49+55
3380 70+41
3381 6E+41
3382 75+41
3383 6D+41
3384 6B+41
3385 4B+42
3386 4D+42
3387 47+42
3388 63+61+6C
3389 6B+63+61+6C
338A 70+46
338B 6E+46
338C 75+46
338D 75+67
338E 6D+67
338F 6B+67
3390 48+7A
3391 6B+48+7A
3392 4D+48+7A
3393 47+48+7A
3394 54+48+7A
3395 75+6C
3396 6D+6C
3397 64+6C
3398 6B+6C
3399 66+6D
339A 6E+6D
339B 75+6D
339C 6D+6D
339D 63+6D
339E 6B+6D
339F 6D+6D+32
33A0 63+6D+32
33A1 6D+32
33A2 6B+6D+32
33A3 6D+6D+33
33A4 63+6D+33
33A5 6D+33
33A6 6B+6D+33
33A7 6D+2F+73
33A8 6D+73+32
33A9 50+61
33AA 6B+50+61
33AB 4D+50+61
33AC 47+50+61
33AD 72+61+64
33AE 72+61+64+2F+73
33AF 72+61+64+2F+73+32
33B0 70+73
33B1 6E+73
33B2 75+73
33B3 6D+73
33B4 70+56
33B5 6E+56
33B6 75+56
33B7 6D+56
33B8 6B+56
33B9 4D+56
33BA 70+57
33BB 6E+57
33BC 75+57
33BD 6D+57
33BE 6B+57
33BF 4D+57
33C0 6B
33C1 4D
33C2 61+2E+6D+2E
33C3 42+71
33C4 63+63
33C5 63+64
33C6 43+2F+6B+67
33C7 43+6F+2E
33C8 64+42
33C9 47+79
33CA 68+61
33CB 48+50
33CC 69+6E
33CD 4B+4B
33CE 4B+4D
33CF 6B+74
33D0 6C+6D
33D1 6C+6E
33D2 6C+6F+67
33D3 6C+78
33D4 6D+62
33D5 6D+69+6C
33D6 6D+6F+6C
33D7 50+48
33D8 70+2E+6D+2E
33D9 50+50+4D
33DA 50+52
33DB 73+72
33DC 53+76
33DD 57+62
33DE 56+2F+6D
33DF 41+2F+6D
33E0 31
33E1 32
33E2 33
33E3 34
33E4 35
33E5 36
33E6 37
33E7 38
33E8 39
33E9 31+30
33EA 31+31
33EB 31+32
33EC 31+33
33ED 31+34
33EE 31+35
33EF 31+36
33F0 31+37
33F1 31+38
33F2 31+39
33F3 32+30
33F4 32+31
33F5 32+32
33F6 32+33
33F7 32+34
33F8 32+35
33F9 32+36
33FA 32+37
33FB 32+38
33FC 32+39
33FD 33+30
33FE 33+31
33FF 67+61+6C
4E36 5C
4E3F 2F
A4D0 42
A4D1 50
A4D2 64
A4D3 44
A4D4 54
A4D6 47
A4D7 4B
A4D9 4A
A4DA 43
A4DC 5A
A4DD 46
A4DF 4D
A4E0 4E
A4E1 4C
A4E2 53
A4E3 52
A4E6 56
A4E7 48
A4EA 57
A4EB 58
A4EC 59
A4EE 41
A4F0 45
A4F2 49
A4F3 4F
A4F4 55
A4F8 2E
A4FB 2E+2C
A4FD 3A
A4FF 3D
A60E 2A
A644 32
A64F 62
A663 41
A667 4D
A681 41
A683 53
A689 41
A731 73
A733 61+61
A740 4B
A741 6B
A742 4B
A743 6B
A744 4B
A745 6B
A748 4C
A749 6C
A74A 4F
A74B 6F
A74C 4F
A74D 6F
A750 50
A751 70
A752 50
A753 70
A754 50
A755 70
A756 51
A757 71
A758 51
A759 71
A75A 32
A75B 72
A75E 56
A75F 76
A76A 33
A76E 39
A778 26
A789 3A
A78C 27
A78E 6C
A78F 2E
A790 4E
A791 6E
A792 43
A793 43
A794 63
A795 68
A796 42
A797 62
A798 46
A799 66
A7A0 47
A7A1 67
A7A2 4B
A7A3 6B
A7A4 4E
A7A5 6E
A7A6 52
A7A7 66
A7A8 53
A7A9 73
A7AA 48
A7AD 4C
A7B2 4A
A7B6 77
A7B7 77
A7B8 55
A7B9 75
A7C4 43
A7C5 53
A7C6 5A
A7C7 44
A7C8 64
A7C9 53
A7CA 73
A7F2 43
A7F3 46
A7F4 51
A7F8 48
A7F9 6F+65
A7FE 49
AB30 61
AB31 61+65
AB32 65
AB33 65
AB34 65
AB35 66
AB36 67
AB37 6C
AB38 6C
AB39 6C
AB3A 6D
AB3B 6E
AB3E 6F
AB46 52
AB47 72
AB49 72
AB4E 75
AB4F 75
AB52 75
AB53 58
AB54 58
AB55 58
AB56 78
AB57 78
AB58 78
AB59 78
AB5A 79
AB5D 6C
AB5E 6C
AB5F 75
AB64 61
AB69 6D
AB70 44
AB71 52
AB72 54
AB74 4F
AB75 69
AB76 73
AB79 79
AB7A 41
AB7B 4A
AB7C 45
AB7E 3F
AB7F 41
AB83 57
AB85 70
AB86 47
AB87 4D
AB8B 48
AB8D 79
AB90 47
AB92 68
AB93 7A
AB96 49
AB9B 45
AB9E 34
AB9F 62
ABA2 52
ABA4 57
ABA5 67
ABA9 76
ABAA 73
ABAE 4C
ABAF 43
ABB2 50
ABB3 43
ABB4 76
ABB5 68
ABB6 4B
ABB7 64
ABB8 43
ABB9 47
ABBE 36
ABBF 77
FB00 66+66
FB01 66+69
FB02 66+6C
FB03 66+66+69
FB04 66+66+6C
FB05 66+74
FB06 73+74
FB1D 27
FB1F 22
FB29 2B
FB35 69
FB38 76
FB39 27
FB41 6F
FB4B 69
FD3E 28
FD3F 29
FE10 2C
FE13 3A
FE14 3B
FE15 21
FE16 3F
FE19 2E+2E+2E
FE30 3A
FE31 4C
FE32 4C
FE33 4C
FE34 4C
FE35 28
FE36 29
FE37 7B
FE38 7D
FE39 5B
FE3A 5D
FE47 5B
FE48 5D
FE4D 5F
FE4E 5F
FE4F 5F
FE50 2C
FE52 2E
FE54 3B
FE55 3A
FE56 3F
FE57 21
FE58 2D
FE59 28
FE5A 29
FE5B 7B
FE5C 7D
FE5D 5B
FE5E 5D
FE5F 23
FE60 26
FE61 2A
FE62 2B
FE63 2D
FE64 3C
FE65 3E
FE66 3D
FE68 5C
FE69 24
FE6A 25
FE6B 40
FF01 21
FF02 22
FF03 23
FF04 24
FF05 25
FF06 26
FF07 27
FF08 28
FF09 29
FF0A 2A
FF0B 2B
FF0C 2C
FF0D 2D
FF0E 2E
FF0F 2F
FF10 30
FF11 31
FF12 32
FF13 33
FF14 34
FF15 35
FF16 36
FF17 37
FF18 38
FF19 39
FF1A 3A
FF1B 3B
FF1C 3C
FF1D 3D
FF1E 3E
FF1F 3F
FF20 40
FF21 41
FF22 42
FF23 43
FF24 44
FF25 45
FF26 46
FF27 47
FF28 48
FF29 49
FF2A 4A
FF2B 4B
FF2C 4C
FF2D 4D
FF2E 4E
FF2F 4F
FF30 50
FF31 51
FF32 52
FF33 53
FF34 54
FF35 55
FF36 56
FF37 57
FF38 58
FF39 59
FF3A 5A
FF3B 5B
FF3C 5C
FF3D 5D
FF3E 5E
FF3F 5F
FF40 60
FF41 61
FF42 62
FF43 63
FF44 64
FF45 65
FF46 66
FF47 67
FF48 68
FF49 69
FF4A 6A
FF4B 6B
FF4C 4C
FF4D 6D
FF4E 6E
FF4F 6F
FF50 70
FF51 71
FF52 72
FF53 73
FF54 74
FF55 75
FF56 76
FF57 77
FF58 78
FF59 79
FF5A 7A
FF5B 7B
FF5C 7C
FF5D 7D
FF5E 7E
FF65 2E
FFE8 4C
FFEE 6F
10300 41
10301 42
10302 43
10303 44
10304 45
10305 46
10306 49
10309 49
1030A 4B
1030B 4C
1030F 4F
10310 52
10311 4D
10312 51
10313 50
10314 53
10315 54
10316 59
10317 58
1031A 38
1031B 70
1031C 62
10320 49
10322 58
10331 42
10332 72
10333 4A
10334 45
10335 55
10336 5A
10337 6E
10338 57
10339 49
1033A 52
1033C 4D
1033D 4E
1033E 47
1033F 4E
10341 55
10342 52
10343 53
10344 54
10345 59
10346 46
10347 58
10348 4F
1040E 57
10415 43
10419 50
1041A 42
1041B 4C
1041E 36
1041F 44
10420 53
10429 45
1042C 4F
10432 72
10436 77
1043D 63
10441 70
10442 42
10443 4C
10444 59
10445 38
10446 36
10447 44
10448 73
10496 53
104B2 41
10781 3A
10783 61+65
10784 42
10785 62
1078B 64
1078C 64
1078D 64
1078F 67
10792 47
10793 67
10794 47
10795 68
10796 48
10797 68
10798 66
1079B 6C
1079D 6C
107A2 6F
107A3 4F+45
107A5 71
107A8 72
107A9 72
107AA 52
107AF 74
107B0 76
107B2 79
107B3 3F
107B5 4F
107B6 6C
107BA 73
10840 58
1084F 55
10A30 34
10A50 2E
10B46 53
10B66 53
1BC0D 44
1D16D 2E
1D400 41
1D401 42
1D402 43
1D403 44
1D404 45
1D405 46
1D406 47
1D407 48
1D408 49
1D409 4A
1D40A 4B
1D40B 4C
1D40C 4D
1D40D 4E
1D40E 4F
1D40F 50
1D410 51
1D411 52
1D412 53
1D413 54
1D414 55
1D415 56
1D416 57
1D417 58
1D418 59
1D419 5A
1D41A 61
1D41B 62
1D41C 63
1D41D 64
1D41E 65
1D41F 66
1D420 67
1D421 68
1D422 69
1D423 6A
1D424 6B
1D425 6C
1D426 6D
1D427 6E
1D428 6F
1D429 70
1D42A 71
1D42B 72
1D42C 73
1D42D 74
1D42E 75
1D42F 76
1D430 77
1D431 78
1D432 79
1D433 7A
1D434 41
1D435 42
1D436 43
1D437 44
1D438 45
1D439 46
1D43A 47
1D43B 48
1D43C 49
1D43D 4A
1D43E 4B
1D43F 4C
1D440 4D
1D441 4E
1D442 4F
1D443 50
1D444 51
1D445 52
1D446 53
1D447 54
1D448 55
1D449 56
1D44A 57
1D44B 58
1D44C 59
1D44D 5A
1D44E 61
1D44F 62
1D450 63
1D451 64
1D452 65
1D453 66
1D454 67
1D456 69
1D457 6A
1D458 6B
1D459 6C
1D45A 6D
1D45B 6E
1D45C 6F
1D45D 70
1D45E 71
1D45F 72
1D460 73
1D461 74
1D462 75
1D463 76
1D464 77
1D465 78
1D466 79
1D467 7A
1D468 41
1D469 42
1D46A 43
1D46B 44
1D46C 45
1D46D 46
1D46E 47
1D46F 48
1D470 49
1D471 4A
1D472 4B
1D473 4C
1D474 4D
1D475 4E
1D476 4F
1D477 50
1D478 51
1D479 52
1D47A 53
1D47B 54
1D47C 55
1D47D 56
1D47E 57
1D47F 58
1D480 59
1D481 5A
1D482 61
1D483 62
1D484 63
1D485 64
1D486 65
1D487 66
1D488 67
1D489 68
1D48A 69
1D48B 6A
1D48C 6B
1D48D 6C
1D48E 6D
1D48F 6E
1D490 6F
1D491 70
1D492 71
1D493 72
1D494 73
1D495 74
1D496 75
1D497 76
1D498 77
1D499 78
1D49A 79
1D49B 7A
1D49C 41
1D49E 43
1D49F 44
1D4A2 47
1D4A5 4A
1D4A6 4B
1D4A9 4E
1D4AA 4F
1D4AB 50
1D4AC 51
1D4AE 53
1D4AF 54
1D4B0 55
1D4B1 56
1D4B2 57
1D4B3 58
1D4B4 59
1D4B5 5A
1D4B6 61
1D4B7 62
1D4B8 63
1D4B9 64
1D4BB 66
1D4BD 68
1D4BE 69
1D4BF 6A
1D4C0 6B
1D4C1 6C
1D4C2 6D
1D4C3 6E
1D4C5 70
1D4C6 71
1D4C7 72
1D4C8 73
1D4C9 74
1D4CA 75
1D4CB 76
1D4CC 77
1D4CD 78
1D4CE 79
1D4CF 7A
1D4D0 41
1D4D1 42
1D4D2 43
1D4D3 44
1D4D4 45
1D4D5 46
1D4D6 47
1D4D7 48
1D4D8 49
1D4D9 4A
1D4DA 4B
1D4DB 4C
1D4DC 4D
1D4DD 4E
1D4DE 4F
1D4DF 50
1D4E0 51
1D4E1 52
1D4E2 53
1D4E3 54
1D4E4 55
1D4E5 56
1D4E6 57
1D4E7 58
1D4E8 59
1D4E9 5A
1D4EA 61
1D4EB 62
1D4EC 63
1D4ED 64
1D4EE 65
1D4EF 66
1D4F0 67
1D4F1 68
1D4F2 69
1D4F3 6A
1D4F4 6B
1D4F5 6C
1D4F6 6D
1D4F7 6E
1D4F8 6F
1D4F9 70
1D4FA 71
1D4FB 72
1D4FC 73
1D4FD 74
1D4FE 75
1D4FF 76
1D500 77
1D501 78
1D502 79
1D503 7A
1D504 55
1D505 42
1D507 44
1D508 45
1D509 46
1D50A 47
1D50D 4A
1D50E 4B
1D50F 4C
1D510 4D
1D511 4E
1D512 4F
1D513 42
1D514 51
1D516 47
1D517 49
1D518 55
1D519 42
1D51A 57
1D51B 58
1D51C 4E
1D51E 61
1D51F 62
1D520 63
1D521 64
1D522 65
1D523 66
1D524 67
1D525 68
1D526 69
1D527 6A
1D528 6B
1D529 6C
1D52A 6D
1D52B 6E
1D52C 6F
1D52D 70
1D52E 71
1D52F 72
1D530 73
1D531 74
1D532 75
1D533 76
1D534 77
1D535 78
1D536 6E
1D537 33
1D538 41
1D539 42
1D53B 44
1D53C 45
1D53D 46
1D53E 47
1D540 49
1D541 4A
1D542 4B
1D543 4C
1D544 4D
1D546 4F
1D54A 53
1D54B 54
1D54C 55
1D54D 56
1D54E 57
1D54F 58
1D550 59
1D552 61
1D553 62
1D554 63
1D555 64
1D556 65
1D557 66
1D558 67
1D559 68
1D55A 69
1D55B 6A
1D55C 6B
1D55D 6C
1D55E 6D
1D55F 6E
1D560 6F
1D561 70
1D562 71
1D563 72
1D564 73
1D565 74
1D566 75
1D567 76
1D568 77
1D569 78
1D56A 79
1D56B 7A
1D56C 55
1D56D 42
1D56E 43
1D56F 44
1D570 45
1D571 46
1D572 47
1D573 48
1D574 49
1D575 4A
1D576 4B
1D577 4C
1D578 4D
1D579 4E
1D57A 4F
1D57B 42
1D57C 51
1D57D 52
1D57E 47
1D57F 49
1D580 55
1D581 42
1D582 57
1D583 58
1D584 4E
1D585 33
1D586 61
1D587 62
1D588 63
1D589 64
1D58A 65
1D58B 66
1D58C 67
1D58D 68
1D58E 69
1D58F 6A
1D590 6B
1D591 6C
1D592 6D
1D593 6E
1D594 6F
1D595 70
1D596 71
1D597 72
1D598 73
1D599 74
1D59A 75
1D59B 76
1D59C 77
1D59D 78
1D59E 6E
1D59F 33
1D5A0 41
1D5A1 42
1D5A2 43
1D5A3 44
1D5A4 45
1D5A5 46
1D5A6 47
1D5A7 48
1D5A8 49
1D5A9 4A
1D5AA 4B
1D5AB 4C
1D5AC 4D
1D5AD 4E
1D5AE 4F
1D5AF 50
1D5B0 51
1D5B1 52
1D5B2 53
1D5B3 54
1D5B4 55
1D5B5 56
1D5B6 57
1D5B7 58
1D5B8 59
1D5B9 5A
1D5BA 61
1D5BB 62
1D5BC 63
1D5BD 64
1D5BE 65
1D5BF 66
1D5C0 67
1D5C1 68
1D5C2 69
1D5C3 6A
1D5C4 6B
1D5C5 6C
1D5C6 6D
1D5C7 6E
1D5C8 6F
1D5C9 70
1D5CA 71
1D5CB 72
1D5CC 73
1D5CD 74
1D5CE 75
1D5CF 76
1D5D0 77
1D5D1 78
1D5D2 79
1D5D3 7A
1D5D4 41
1D5D5 42
1D5D6 43
1D5D7 44
1D5D8 45
1D5D9 46
1D5DA 47
1D5DB 48
1D5DC 49
1D5DD 4A
1D5DE 4B
1D5DF 4C
1D5E0 4D
1D5E1 4E
1D5E2 4F
1D5E3 50
1D5E4 51
1D5E5 52
1D5E6 53
1D5E7 54
1D5E8 55
1D5E9 56
1D5EA 57
1D5EB 58
1D5EC 59
1D5ED 5A
1D5EE 61
1D5EF 62
1D5F0 63
1D5F1 64
1D5F2 65
1D5F3 66
1D5F4 67
1D5F5 68
1D5F6 69
1D5F7 6A
1D5F8 6B
1D5F9 6C
1D5FA 6D
1D5FB 6E
1D5FC 6F
1D5FD 70
1D5FE 71
1D5FF 72
1D600 73
1D601 74
1D602 75
1D603 76
1D604 77
1D605 78
1D606 79
1D607 7A
1D608 41
1D609 42
1D60A 43
1D60B 44
1D60C 45
1D60D 46
1D60E 47
1D60F 48
1D610 49
1D611 4A
1D612 4B
1D613 4C
1D614 4D
1D615 4E
1D616 4F
1D617 50
1D618 51
1D619 52
1D61A 53
1D61B 54
1D61C 55
1D61D 56
1D61E 57
1D61F 58
1D620 59
1D621 5A
1D622 61
1D623 62
1D624 63
1D625 64
1D626 65
1D627 66
1D628 67
1D629 68
1D62A 69
1D62B 6A
1D62C 6B
1D62D 6C
1D62E 6D
1D62F 6E
1D630 6F
1D631 70
1D632 71
1D633 72
1D634 73
1D635 74
1D636 75
1D637 76
1D638 77
1D639 78
1D63A 79
1D63B 7A
1D63C 41
1D63D 42
1D63E 43
1D63F 44
1D640 45
1D641 46
1D642 47
1D643 48
1D644 49
1D645 4A
1D646 4B
1D647 4C
1D648 4D
1D649 4E
1D64A 4F
1D64B 50
1D64C 51
1D64D 52
1D64E 53
1D64F 54
1D650 55
1D651 56
1D652 57
1D653 58
1D654 59
1D655 5A
1D656 61
1D657 62
1D658 63
1D659 64
1D65A 65
1D65B 66
1D65C 67
1D65D 68
1D65E 69
1D65F 6A
1D660 6B
1D661 6C
1D662 6D
1D663 6E
1D664 6F
1D665 70
1D666 71
1D667 72
1D668 73
1D669 74
1D66A 75
1D66B 76
1D66C 77
1D66D 78
1D66E 79
1D66F 7A
1D670 41
1D671 42
1D672 43
1D673 44
1D674 45
1D675 46
1D676 47
1D677 48
1D678 49
1D679 4A
1D67A 4B
1D67B 4C
1D67C 4D
1D67D 4E
1D67E 4F
1D67F 50
1D680 51
1D681 52
1D682 53
1D683 54
1D684 55
1D685 56
1D686 57
1D687 58
1D688 59
1D689 5A
1D68A 61
1D68B 62
1D68C 63
1D68D 64
1D68E 65
1D68F 66
1D690 67
1D691 68
1D692 69
1D693 6A
1D694 6B
1D695 6C
1D696 6D
1D697 6E
1D698 6F
1D699 70
1D69A 71
1D69B 72
1D69C 73
1D69D 74
1D69E 75
1D69F 76
1D6A0 77
1D6A1 78
1D6A2 79
1D6A3 7A
1D6A4 69
1D6A5 6A
1D6A8 41
1D6A9 42
1D6AC 45
1D6AD 5A
1D6AE 48
1D6B0 49
1D6B1 4B
1D6B3 4D
1D6B4 4E
1D6B6 4F
1D6B8 50
1D6BA 45
1D6BB 54
1D6BC 59
1D6BD 4F
1D6BE 58
1D6BF 57
1D6C2 61
1D6C3 42
1D6C4 79
1D6C5 64
1D6C6 65
1D6C7 7A
1D6C8 6E
1D6C9 6F
1D6CA 69
1D6CB 6B
1D6CD 75
1D6CE 76
1D6CF 65
1D6D0 6F
1D6D1 6E
1D6D2 70
1D6D3 63
1D6D4 6F
1D6D5 74
1D6D6 75
1D6D8 78
1D6D9 77
1D6DA 77
1D6DC 65
1D6DD 6F
1D6DE 6B
1D6E0 70
1D6E1 77
1D6E2 41
1D6E3 42
1D6E6 45
1D6E7 5A
1D6E8 48
1D6EA 49
1D6EB 4B
1D6ED 4D
1D6EE 4E
1D6F0 4F
1D6F2 50
1D6F4 45
1D6F5 54
1D6F6 59
1D6F7 4F
1D6F8 58
1D6F9 57
1D6FC 61
1D6FD 42
1D6FE 79
1D6FF 64
1D700 65
1D701 7A
1D702 6E
1D703 6F
1D704 69
1D705 6B
1D707 75
1D708 76
1D709 65
1D70A 6F
1D70B 6E
1D70C 70
1D70D 63
1D70E 6F
1D70F 74
1D710 75
1D712 78
1D713 77
1D714 77
1D716 65
1D717 6F
1D718 6B
1D71A 70
1D71B 77
1D71C 41
1D71D 42
1D720 45
1D721 5A
1D722 48
1D724 49
1D725 4B
1D727 4D
1D728 4E
1D72A 4F
1D72C 50
1D72E 45
1D72F 54
1D730 59
1D731 4F
1D732 58
1D733 57
1D736 61
1D737 42
1D738 79
1D739 64
1D73A 65
1D73B 7A
1D73C 6E
1D73D 6F
1D73E 69
1D73F 6B
1D741 75
1D742 76
1D743 65
1D744 6F
1D745 6E
1D746 70
1D747 63
1D748 6F
1D749 74
1D74A 75
1D74C 78
1D74D 77
1D74E 77
1D750 65
1D751 6F
1D752 6B
1D754 70
1D755 77
1D756 41
1D757 42
1D75A 45
1D75B 5A
1D75C 48
1D75E 49
1D75F 4B
1D761 4D
1D762 4E
1D764 4F
1D766 50
1D768 45
1D769 54
1D76A 59
1D76B 4F
1D76C 58
1D76D 57
1D770 61
1D771 42
1D772 79
1D773 64
1D774 65
1D775 7A
1D776 6E
1D777 6F
1D778 69
1D779 6B
1D77B 75
1D77C 76
1D77D 65
1D77E 6F
1D77F 6E
1D780 70
1D781 63
1D782 6F
1D783 74
1D784 75
1D786 78
1D787 77
1D788 77
1D78A 65
1D78B 6F
1D78C 6B
1D78E 70
1D78F 77
1D790 41
1D791 42
1D794 45
1D795 5A
1D796 48
1D798 49
1D799 4B
1D79B 4D
1D79C 4E
1D79E 4F
1D7A0 50
1D7A2 45
1D7A3 54
1D7A4 59
1D7A5 4F
1D7A6 58
1D7A7 57
1D7AA 61
1D7AB 42
1D7AC 79
1D7AD 64
1D7AE 65
1D7AF 7A
1D7B0 6E
1D7B1 6F
1D7B2 69
1D7B3 6B
1D7B5 75
1D7B6 76
1D7B7 65
1D7B8 6F
1D7B9 6E
1D7BA 70
1D7BB 63
1D7BC 6F
1D7BD 74
1D7BE 75
1D7C0 78
1D7C1 77
1D7C2 77
1D7C4 65
1D7C5 6F
1D7C6 6B
1D7C8 70
1D7C9 77
1D7CA 46
1D7CB 66
1D7CE 30
1D7CF 31
1D7D0 32
1D7D1 33
1D7D2 34
1D7D3 35
1D7D4 36
1D7D5 37
1D7D6 38
1D7D7 39
1D7D8 30
1D7D9 31
1D7DA 32
1D7DB 33
1D7DC 34
1D7DD 35
1D7DE 36
1D7DF 37
1D7E0 38
1D7E1 39
1D7E2 30
1D7E3 31
1D7E4 32
1D7E5 33
1D7E6 34
1D7E7 35
1D7E8 36
1D7E9 37
1D7EA 38
1D7EB 39
1D7EC 30
1D7ED 31
1D7EE 32
1D7EF 33
1D7F0 34
1D7F1 35
1D7F2 36
1D7F3 37
1D7F4 38
1D7F5 39
1D7F6 30
1D7F7 31
1D7F8 32
1D7F9 33
1D7FA 34
1D7FB 35
1D7FC 36
1D7FD 37
1D7FE 38
1D7FF 39
1DF09 74
1DF11 6C
1DF13 6C
1DF16 72
1DF1A 69
1DF1B 6F
1DF1D 63
1DF1E 73
1DF25 64
1DF26 6C
1DF27 6E
1DF28 72
1DF29 73
1DF2A 74
1E030 61
1E031 36
1E033 72
1E035 65
1E039 6B
1E03C 6F
1E03D 6E
1E03E 70
1E03F 63
1E040 74
1E041 79
1E043 78
1E044 75
1E04C 69
1E04D 6A
1E04F 79
1E050 69
1E051 61
1E052 36
1E054 72
1E056 65
1E05A 6B
1E05C 6F
1E05D 6E
1E05E 63
1E05F 79
1E061 78
1E062 75
1E068 69
1E069 73
1F100 30+2E
1F101 30+2C
1F102 31+2C
1F103 32+2C
1F104 33+2C
1F105 34+2C
1F106 35+2C
1F107 36+2C
1F108 37+2C
1F109 38+2C
1F10A 39+2C
1F110 28+41+29
1F111 28+42+29
1F112 28+43+29
1F113 28+44+29
1F114 28+45+29
1F115 28+46+29
1F116 28+47+29
1F117 28+48+29
1F118 28+49+29
1F119 28+4A+29
1F11A 28+4B+29
1F11B 28+4C+29
1F11C 28+4D+29
1F11D 28+4E+29
1F11E 28+4F+29
1F11F 28+50+29
1F120 28+51+29
1F121 28+52+29
1F122 28+53+29
1F123 28+54+29
1F124 28+55+29
1F125 28+56+29
1F126 28+57+29
1F127 28+58+29
1F128 28+59+29
1F129 28+5A+29
1F12A 5B+53+5D
1F12B 28+43+29
1F12C 28+52+29
1F12D 28+43+44+29
1F12E 28+57+5A+29
1F130 41
1F131 42
1F132 43
1F133 44
1F134 45
1F135 46
1F136 47
1F137 48
1F138 49
1F139 4A
1F13A 4B
1F13B 4C
1F13C 4D
1F13D 4E
1F13E 4F
1F13F 50
1F140 51
1F141 52
1F142 53
1F143 54
1F144 55
1F145 56
1F146 57
1F147 58
1F148 59
1F149 5A
1F14A 5B+48+56+5D
1F14B 5B+4D+56+5D
1F14C 5B+53+44+5D
1F14D 5B+53+53+5D
1F14E 5B+50+50+56+5D
1F14F 5B+57+43+5D
1F150 28+41+29
1F151 28+42+29
1F152 28+43+29
1F153 28+44+29
1F154 28+45+29
1F155 28+46+29
1F156 28+47+29
1F157 28+48+29
1F158 28+49+29
1F159 28+4A+29
1F15A 28+4B+29
1F15B 28+4C+29
1F15C 28+4D+29
1F15D 28+4E+29
1F15E 28+4F+29
1F15F 28+50+29
1F160 28+51+29
1F161 28+52+29
1F162 28+53+29
1F163 28+54+29
1F164 28+55+29
1F165 28+56+29
1F166 28+57+29
1F167 28+58+29
1F168 28+59+29
1F169 28+5A+29
1F16A 4D+43
1F16B 4D+44
1F16C 4D+52
1F16D 28+43+43+29
1F170 41
1F171 42
1F172 43
1F173 44
1F174 45
1F175 46
1F176 47
1F177 48
1F178 49
1F179 4A
1F17A 4B
1F17B 4C
1F17C 4D
1F17D 4E
1F17E 4F
1F17F 50
1F180 51
1F181 52
1F182 53
1F183 54
1F184 55
1F185 56
1F186 57
1F187 58
1F188 59
1F189 5A
1F18A 5B+50+5D
1F18B 5B+49+43+5D
1F18C 5B+50+41+5D
1F18D 5B+53+41+5D
1F18E 5B+41+42+5D
1F18F 5B+57+43+5D
1F190 44+4A
1F191 5B+43+4C+5D
1F192 5B+43+4F+4F+4C+5D
1F193 5B+46+52+45+45+5D
1F194 5B+49+44+5D
1F195 5B+4E+45+57+5D
1F196 5B+4E+47+5D
1F197 5B+4F+4B+5D
1F198 5B+53+4F+53+5D
1F199 5B+55+50+21+5D
1F19A 5B+56+53+5D
1F19B 5B+33+44+5D
1F19D 5B+32+4B+5D
1F19E 5B+34+4B+5D
1F19F 5B+38+4B+5D
1F1A0 5B+35+2E+31+5D
1F1A1 5B+37+2E+31+5D
1F1A2 5B+32+32+2E+31+5D
1F1A3 5B+36+30+70+5D
1F1A4 5B+31+32+30+70+5D
1F1A5 64
1F1A6 5B+48+43+5D
1F1A7 5B+48+44+52+5D
1F1AA 5B+53+48+56+5D
1F1AB 5B+55+48+44+5D
1F1AC 5B+56+4F+44+5D
1F1E6 41
1F1E7 42
1F1E8 43
1F1E9 44
1F1EA 45
1F1EB 46
1F1EC 47
1F1ED 48
1F1EE 49
1F1EF 4A
1F1F0 4B
1F1F1 4C
1F1F2 4D
1F1F3 4E
1F1F4 4F
1F1F5 50
1F1F6 51
1F1F7 52
1F1F8 53
1F1F9 54
1F1FA 55
1F1FB 56
1F1FC 57
1F1FD 58
1F1FE 59
1F1FF 5A
1F672 26
1F673 26
1F674 26
1F675 26
1F676 22
1F677 22
1F6C8 28+69+29
1F784 2E
1F785 4F
1F786 4F
1F787 4F
1F788 4F
1F789 4F
1F7A8 58
1F7A9 58
1F7AA 58
1F7AB 58
1F7AC 58
1F7AD 58
1F7AE 58
1F7AF 2A
1F7B0 2A
1F7B1 2A
1F7B2 2A
1F7B3 2A
1F7B4 2A
1F7B5 2A
1F7B6 2A
1F7B7 2A
1F7B8 2A
1F7B9 2A
1F7BA 2A
1F7BB 2A
1F7BC 2A
1F7BD 2A
1F7BE 2A
1F7BF 2A
1FBC4 5B+3F+5D
1FBF0 30
1FBF1 31
1FBF2 32
1FBF3 33
1FBF4 34
1FBF5 35
1FBF6 36
1FBF7 37
1FBF8 38
1FBF9 39
