package Convert::Dayi;

use utf8;
use warnings;
use strict;

=encoding utf8

=head1 NAME

Convert::Dayi - Dayi Convention Module

=head1 VERSION

Version 0.04

=cut

our $VERSION = '0.04';
use Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(as_ascii as_dayi from_ascii);

=head1 SYNOPSIS

    use utf8;  # you will need this
    use Convert::Dayi qw(as_ascii as_dayi from_ascii);

    print as_ascii("大易中文")    # v db/ o1 kx

    print as_dayi("大易中文")     # 禾 日馬竹 口言 立水

for from_ascii function , you need to provide index (start from 1)

    print from_ascii('v1 db/1')   # 大易

    print from_ascii('v db/')   # 大易  (if the word is not defined , select index 1 by default)

to translate chinese from STDIN:

    #!/usr/bin/env perl
    use utf8;
    use Convert::Dayi qw(as_ascii);
    use Encode;
    while( <STDIN> ) {
        my $word = Encode::decode_utf8( $_ );
        print as_ascii( $word );
    }

=head1 DESCRIPTION

L<Convert::Dayi> module provides a way to convert chinese words to ascii code by Input Method Keys.

You can also convert the keycode to chinese back.

=head1 EXPORT

=head1 FUNCTIONS

=cut

our %KEYS = ();
our %KEYS_REV = ();
our %WORDS = ();
our %WORDS_REV = ();

INIT {
    init(); 
}

sub init  {
    init_keys();
    init_words();
}

sub init_keys {
    while( <DATA> ) {
        if( /^KEYPROMPT\((.)\):\s+(.)/ ) {
            $KEYS{ $1 } = $2;
            $KEYS_REV{ $2 } = $1;
        }
        elsif( /BEGINDICTIONARY/ ) {
            last;
        }
    }

}

sub init_words {
    while( <DATA> ) {
        if( $_ !~ m/^#/ ) {
            my ($key,$word) = split /\s/;
            $WORDS{ $word } = $key;
            my $key_index = 0;
            while( defined( $WORDS_REV{ $key . ++$key_index } ) ) { }
            $WORDS_REV{ $key . $key_index } = $word;
        }
    }
}

# must provide index
=head2 from_ascii 

=cut

sub from_ascii {
    my $string = shift;
    my $out = '';
    for( split /\s/,$string ) {
        if( defined $WORDS_REV{$_} ) {
            $out .= $WORDS_REV{$_};
        }
        elsif( defined $WORDS_REV{$_} ) {
            $out .= $WORDS_REV{$_ . '1'};  # try index 1
        }
        else {
            $out .= $_;
        }
    }
    return $out;
}

=head2 as_dayi

=cut

sub as_dayi {
    my $string = shift;
    my $out = '';
    while( $string =~ /(.)/g ) {
        my $word = $1;

        if( defined $WORDS{$word} ) {
            my $keys = $WORDS{ $word } ;
            for ( split //,$keys ) {
                $out .= $KEYS{ $_ };
            }
            $out .= ' ';
        }
        else {
            $out .= $word;
        }
    }
    return $out;
}

=head2 as_ascii

=cut

sub as_ascii {
    my $string = shift;
    my $out = '';
    while( $string =~ /(.)/g ) {
        my $word = $1;
        if( defined $WORDS{$word} ) {
            $out .= $WORDS{$word} . " ";
        }
        else {
            $out .= $word;
        }
    }
    return $out;
}


=head1 AUTHOR

Cornelius, C<< <cornelius.howl_at_gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-convert-dayi at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Convert-Dayi>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Convert::Dayi


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Convert-Dayi>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Convert-Dayi>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Convert-Dayi>

=item * Search CPAN

L<http://search.cpan.org/dist/Convert-Dayi/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 COPYRIGHT & LICENSE

Copyright 2009 Cornelius, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.


=cut

1; # End of Convert::Dayi
__DATA__
# $Id: DAYI.tit,v 1.0 1996/06/15 22:45:00 ygz Exp $
# HANZI input table for cxterm
# To be used by cxterm, convert me to .cit format first
# .cit version 2
ENCODE: BIG5
PROMPT: 漢字輸入::大易::\040
AUTOSELECT:     NEVER
#
COMMENT 大易輸入法
#
# input key definitions
VALIDINPUTKEY:  `',-./;=[]0123456789abcdefghijklmnopqrstuvwxyz
WILDCARDKEY:    *
WILDCHARKEY:    ?
ASSOCIATIONKEY: _
# choice list keys
SELECTKEY:      1\040
SELECTKEY:      2
SELECTKEY:      3
SELECTKEY:      4
SELECTKEY:      5
SELECTKEY:      6
SELECTKEY:      7
SELECTKEY:      8
SELECTKEY:      9
SELECTKEY:      0
MOVERIGHT:      >
MOVELEFT:       <
# pre-editing keys
BACKSPACE:      \010\177                # ctrl-H,  DEL
DELETEALL:      \015\025                # RETURN,  ctrl-U
CURSOR-BEGIN:   \001                    # ctrl-A
CURSOR-END:     \005                    # ctrl-E
CURSOR-FORW:    \006                    # ctrl-F
CURSOR-BACK:    \002                    # ctrl-B
CURSOR-ERASE:   \004                    # ctrl-D
REPEATKEY:      \020\022                # ctrl-P,  ctrl-R
# labels for input keys
KEYPROMPT(`):   ‘
KEYPROMPT('):   ’
KEYPROMPT(,):   力
KEYPROMPT(-):   －
KEYPROMPT(.):   點
KEYPROMPT(/):   竹
KEYPROMPT(;):   虫
KEYPROMPT(=):   ＝
KEYPROMPT([):   （
KEYPROMPT(]):   ）
KEYPROMPT(0):   金
KEYPROMPT(1):   言
KEYPROMPT(2):   牛
KEYPROMPT(3):   目
KEYPROMPT(4):   四
KEYPROMPT(5):   王
KEYPROMPT(6):   車
KEYPROMPT(7):   田
KEYPROMPT(8):   八
KEYPROMPT(9):   足
KEYPROMPT(a):   人
KEYPROMPT(b):   馬
KEYPROMPT(c):   鹿
KEYPROMPT(d):   日
KEYPROMPT(e):   一
KEYPROMPT(f):   土
KEYPROMPT(g):   手
KEYPROMPT(h):   鳥
KEYPROMPT(i):   木
KEYPROMPT(j):   月
KEYPROMPT(k):   立
KEYPROMPT(l):   女
KEYPROMPT(m):   雨
KEYPROMPT(n):   魚
KEYPROMPT(o):   口
KEYPROMPT(p):   耳
KEYPROMPT(q):   石
KEYPROMPT(r):   工
KEYPROMPT(s):   革
KEYPROMPT(t):   糸
KEYPROMPT(u):   廾
KEYPROMPT(v):   禾
KEYPROMPT(w):   山
KEYPROMPT(x):   水
KEYPROMPT(y):   火
KEYPROMPT(z):   心
# the following line must not be removed
BEGINDICTIONARY
#
, 力
,,, 劦
,,,j 脅
,,,l 姭
,,,z 勰
,.rk 狼
,/90 狿
,/c 犰
,/d1 猈
,/dx 獥
,1 狺
,1rx 獀
,1v. 獄
,1vh 鸑
,2c 犵
,2e 狌
,2mn 猘
,33x 玃
,38 狽
,4b; 獨
,4ek 獧
,4tf 玀
,5 狂
,5 狅
,50 逛
,5h 鵟
,6g. 猼
,6h1 獑
,7ek 猥
,88 猶
,8x 獙
,9h5 獩
,;8x 狻
,;q 狁
,a4d 獪
,ab. 狑
,ai 狳
,aoa 獫
,aro 獊
,arv 猴
,aso 猞
,b,i 猱
,b,o 獝
,bc 犯
,bf 狃
,bfw 峱
,bgt 猻
,bgu 猛
,bo 狗
,ccf 狴
,cq 猇
,cs8 獷
,d1 狎
,d2e 猩
,d4x 獌
,dbl 猲
,dcc 猑
,dd 猖
,de 狚
,df 狸
,di 猓
,e8x 獮
,ede 狟
,eet 狋
,es 犴
,es5 狨
,fc 狫
,ffq 獟
,fj 猜
,fnv 猰
,fo 狤
,fok 猿
,gh1 狾
,h/t 獂
,h3x 獿
,h9e 狉
,h;0 狐
,hc 狔
,hex 猳
,hj 猏
,hjx 獶
,hmu 猵
,hr1 獅
,hua 獗
,io8 獺
,j2w 猺
,je 狙
,jrj 猙
,k8x 狡
,k9 犿
,kas 猝
,kdf 獞
,kdq 獍
,kds 獐
,kok 獽
,kq 犺
,l75 獵
,m8 狖
,meo 狪
,meu 狦
,mg. 狩
,mh1 獳
,mj 猾
,mo 猧
,mzj 獰
,n, 猭
,nn2 獬
,o 加
,o0 迦
,o1 狆
,o38 賀
,ob 駕
,oej 哿
,oh 鴐
,oi 架
,oj 狷
,okk 袈
,oox 玁
,ovc 毠
,q; 猦
,qy 狣
,r5 狘
,rb1 狒
,rbx 獼
,rj/ 犽
,rk 狠
,se8 獚
,so 狜
,soj 猢
,uax 獲
,udl 獦
,uo1 玂
,uof 獾
,uuv 獛
,v4 獯
,v8t 獠
,vaa 狹
,vao 獢
,vej 猗
,vn 猁
,vv 獉
,wby 獡
,wq 猊
,xs1 狶
,xt/ 猀
,xwx 猣
,y 狄
,y0 逖
,yz 悐
. ．
. 方
. 舟
.. 舫
./bp 鄬
./by 為
./d 舶
./d5 艎
./qy 艞
.0 凎
.1.1 州
.1rx 艘
.22e 旌
.22h 鷟
.22v 族
.23 舴
.24t 旚
.2a. 旍
.2ad 旝
.2b0 遊
.2b9 旋
.2bc 施
.2bg 斿
.2h1 旂
.2hc 旎
.2j7 旛
.2k1 旆
.2kq 旒
.2mk 旃
.2p8 旗
.2q 冼
.2q 斻
.2qy 旐
.2rj 膂
.2rk 旅
.2tj 旓
.2u0 旞
.2vc 旄
.2vj 旖
.2w8 旟
.4f 凐
.50 艇
.7s 舳
.7so 凅
.;j 朗
.;jf 塱
.;n 剆
.;na 欴
.;o 冶
.;p 郎
.a. 於
.ab. 冷
.ab. 舲
.ai 艅
.akf 准
.akx 艭
.ao 冾
.aro 艙
.aro 凔
.b 馮
.bgu 艋
.bz 憑
.c29 凝
.c7, 艣
.c7u 艫
.d1 舺
.d3 艒
.dol 艛
.e7d 艚
.ej. 瓬
.en 冱
.enn 冽
.eoj 舸
.erl 凄
.f8x 凌
.fj 凊
.h 鵃
.hmu 艑
.hvc 艉
.hx 舨
.id 凍
.jbg 艀
.jrj 凈
.k7v 凜
.k8. 艕
.k8p 艩
.kdf 艟
.kq 航
.kt 舷
.mc 舵
.mfo 凋
.n 舠
.na 次
.na. 瓷
.na8 資
.na8 粢
.nai 栥
.nai 楶
.nak 餈
.nal 姿
.nao 咨
.nav 秶
.naz 恣
.no. 艬
.o1 舯
.p 邡
.p81 凘
.qo 船
.qx 般
.qx1 幋
.qxg 搫
.qxi 槃
.qxk 褩
.qxl 媻
.qxq 磐
.qxs 鞶
.qxt 縏
.qxu 盤
.r 舡
.r1c 舥
.r1u 艦
.r1y 凞
.rk 良
.s 斗
.s8 舼
.sli 艓
.sm1 艜
.sx. 冹
.tj 艄
.tm7 艡
.u/3 艏
.u8u 艗
.uf5 艤
.ufr 艖
.um, 艨
.veb 舿
.w 舢
.waf 凗
.x 永
.x 冰
.x 放
.x2 舽
.xd 昶
.xf 艂
.xwx 艐
/ 竹
/, 竻
/,o 笳
/.2c 箷
/.2v 簇
/.a. 箊
/.rk 筤
/.ru 簋
//30 籩
//7c 篦
//90 筵
//b/ 笫
//bx 笈
//d1 箄
//d5 篁
/0e5 籛
/0eu 簦
/0nx 籙
/0vf 籦
/23 笮
/2e 笙
/2o 筶
/2o0 簉
/2q 筅
/2q8 籫
/3 自
/3, 臮
/37. 齀
/37c 鼽
/37c 齆
/37e 齇
/37k 齈
/37k 齉
/37l 齃
/37n 劓
/37o 齁
/37s 鼾
/37u 鼻
/37x 齂
/3i 臬
/3i. 甈
/3ic 臲
/3ih 鷍
/3m0 邊
/3p 郋
/3q 筧
/3t/ 篎
/3u 算
/3v. 臭
/3v; 篡
/3vc 篹
/3vt 纂
/3z 息
/3zp 鄎
/4, 眾
/4ds 簟
/4et 篻
/4f. 籈
/4fs 睾
/4h5 篾
/4i 篥
/4tf 籮
/50 筳
/5u 筭
/6/1 簰
/6bc 範
/6d 簡
/6i4 籣
/6kx 簢
/6mu 簠
/6na 簐
/7/0 囪
/7e1 箅
/7e5 簂
/7jb 粵
/7n0 邍
/7s 笛
/7so 箇
/7v 箘
/7x 囟
/7x 笝
/90 延
/90; 蜑
/90p 郔
/9o 笘
/9xo 簬
/; 么
/; 鬼
/;.s 魁
/;4t 魒
/;;/ 篸
/;af 魋
/;cu 魖
/;e5 魊
/;ea 魎
/;ei 魅
/;h1 鬿
/;k; 魑
/;ml 魍
/;o 笞
/;p 節
/;p8 魌
/;r5 魆
/;s. 魃
/;sx 鬾
/;t5 魕
/;tj 魈
/;u 笲
/a1i 篠
/a1x 筱
/a23 筰
/aa5 籤
/ab 笒
/ab. 笭
/ae5 筏
/aex 箯
/af 筌
/afj 篟
/ag. 符
/ai 筡
/ao 答
/ao8 籲
/aoa 簽
/aon 劄
/aou 籥
/arv 篌
/b 篤
/b. 笉
/b.f 筩
/b/ 笏
/bb 乃
/bb5 戤
/bb6 鼐
/bbf 籊
/bbg 孕
/bbt 尕
/bbu 盈
/bbx 夃
/bd 筍
/beo 笥
/bg0 箛
/bo 笱
/brc 笣
/bww 篘
/bx 及
/bxy 篜
/c 九
/c 簏
/c. 丸
/c.l 奿
/c1 訄
/c7u 籚
/c; 厹
/cc 笓
/ccq 虓
/cd 旭
/cd 旮
/ce0 籧
/ch 鳩
/ch8 頄
/cq 箎
/cu3 馗
/cut 簾
/d 白
/d.0 邀
/d.1 譥
/d.h 鸄
/d/; 魄
/d/d 皛
/d/e 皞
/d/i 皪
/d/x 皦
/d0 迫
/d1 笚
/d2o 皓
/d5 皇
/d50 遑
/d5q 皝
/d9c 皉
/da. 皊
/db. 的
/dc 皂
/dc 癿
/dch 鵖
/dcy 皫
/de 笪
/des 筸
/dhx 皈
/dih 鷎
/dj. 皭
/dj7 皤
/dko 皜
/dkx 皎
/dm1 帛
/dmq 皖
/dol 簍
/dox 籔
/drq 兜
/ds 皁
/dti 樂
/duf 篳
/duu 皏
/dv1 卑
/dv5 皒
/dvb 翱
/dvh 鵯
/dvp 郫
/dvs 皋
/dwu 皚
/dx 泉
/dxx 灥
/e,0 篴
/e2p 篽
/e55 箋
/e99 簁
/e;n 箌
/ed. 篿
/ee 竺
/eeq 笎
/eg 竽
/eoj 笴
/er9 箑
/es 竿
/eu 笄
/f38 簀
/f;u 篕
/fao 簭
/fb. 筠
/fb. 籌
/fd 箸
/ff 筀
/ffr 篲
/fg. 等
/fj 箐
/fof 籉
/g4s 籜
/go 筘
/gr1 箍
/gr7 籀
/gsd 簎
/gse 箝
/gsf 簼
/h10 笊
/h38 須
/h3l 嬃
/hcq 篪
/hmu 篇
/ho5 箴
/hr1 篩
/i3 箱
/iaa 箂
/ie 笨
/ii 箖
/im 策
/io8 籟
/ioa 簌
/j, 筋
/jhx 箙
/jrj 箏
/jux 籐
/k 乏
/k 笠
/k8. 篣
/k8x 筊
/kas 箤
/kdk 簑
/kh 鴔
/kh 鷑
/kjx 籠
/kl 笀
/kl. 籯
/ko, 籇
/koo 篙
/kop 篰
/kq 笐
/kx; 篱
/kxe 簅
/kxf 籬
/lo 筎
/lx 笯
/m8r 箜
/maf 篧
/maf 籗
/meo 筒
/meq 筦
/mex 簆
/mf 甪
/mhr 管
/mjv 奧
/mjw 嶴
/mo 向
/mo0 簻
/mu 血
/mu5 衊
/mu9 衃
/muf 衄
/mup 卹
/mup 衈
/n, 篆
/nh1 簷
/no 笤
/o/o 筥
/o18 簣
/o38 篔
/oos 簞
/p8 箕
/p8f 簊
/p8x 簸
/pai 篨
/qy 筄
/r 笁
/r16 簫
/r1c 笆
/r1u 籃
/r33 篚
/r38 篢
/r5 筐
/ra0 遾
/raa 筮
/rb. 篛
/rb/ 第
/rb1 笰
/rc8 簨
/rf 筆
/rj5 笢
/ro 笸
/rp 筇
/rp 笻
/rq. 筑
/rqd 簪
/rqg 篫
/rqi 築
/rva 篋
/s6. 簙
/sds 簳
/se8 簧
/smf 篝
/t 系
/t/3 箵
/tb. 箹
/tj 筲
/tjn 箾
/tm7 簹
/tt1 籪
/ujn 箭
/uso 箬
/v1o 筈
/v8t 簝
/va 笑
/vaa 筴
/vao 簥
/vid 籍
/vn 筣
/vnn 簃
/vuf 箠
/we8 籅
/x/d 箔
/x6. 簿
/x;o 箈
/xbc 笵
/xd/ 簜
/xdf 篞
/xf0 篷
/xj7 籓
/xs8 篊
/xx 笅
/yy, 簩
/zra 筷
0 金
0. 鈁
0.11 銂
0.29 鏇
0.2v 鏃
0.;p 鎯
0.rk 鋃
0.s 鈄
0.x 銢
0/ 釤
0/3i 鎳
0/3z 鎴
0/7c 鎞
0/90 鋋
0/bb 釢
0/bx 鈒
0/c 釚
0/d 鉑
0/d1 錦
0/d1 錍
0/d5 鍠
0/di 鑠
0/im 鏼
000 鑫
00eu 鐙
00rx 鏺
0111 釧
01rx 鎪
027. 鋂
02c 釳
02dx 鍑
02e 鉎
02i 銖
02m1 鉓
02o 鋯
02o2 鍏
02q 銑
02q8 鑽
03 鉬
033x 钁
038 鋇
038n 鍘
03q 鋧
04b; 鐲
04ds 鐔
04ek 鐶
04et 鏢
04fs 鐸
04tf 鑼
05. 鈺
050 鋌
05; 銩
05oo 鑩
06 鋪
06 鍆
060 鏈
06g. 鎛
06i4 鑭
06j. 鏘
07 鈿
07. 鉧
077f 鑸
07ek 鍡
07s 鈾
07so 錮
07t 鏍
07v 銦
07z 鍶
08 銤
088. 鐏
08; 鈆
08n 鈖
08n2 鏻
08o 鋊
08oq 銳
08r/ 銻
09 釙
0938 鍞
097y 鏀
09e9 鉲
09h5 鐬
09j 錹
09o 鉆
09x 鈙
0;2 鉾
0;8x 鋑
0;;/ 鏒
0;a 鉯
0;o 鈶
0;q 鈗
0a1/ 鎀
0a5 銋
0aa5 鑯
0aaf 銼
0ab 鈐
0ab. 鈴
0abo 鋡
0af 銓
0ai 銝
0akf 錐
0aki 鏶
0aks 鎨
0akw 鐫
0aky 鐎
0amu 錀
0ao 鉿
0aoa 鐱
0aou 鑰
0aro 鎗
0arv 鍭
0b 鎷
0b,i 鍒
0b,o 鐍
0b. 釣
0b. 鈞
0b.z 憌
0b1 鍧
0b2w 鋾
0bb/ 鏐
0beo 鉰
0bf 鈕
0bg 釨
0bgu 錳
0bj 釕
0bo 鉤
0brc 鉋
0c 釔
0c 釓
0c 鏕
0c7, 鐪
0c7u 鑪
0c7z 鑢
0c8 鎎
0cc 鈚
0ccd 鍇
0ce, 鐻
0ce. 钀
0cjg 鏬
0crf 鏞
0cro 鎕
0crx 鏮
0csx 鍍
0csy 鏣
0cut 鐮
0cx 鈹
0cy 鑣
0d 鈤
0d1 鉀
0d4x 鏝
0db/ 錫
0dbb 鎉
0dbl 鍻
0dcc 錕
0dd 錩
0de 鉭
0de/ 鍚
0de9 鍉
0des 銲
0df 鋰
0di 錁
0dol 鏤
0dsx 鑤
0dtq 鎤
0duf 鏎
0e1e 錏
0e2o 鋙
0e5 釴
0e55 錢
0e8x 鑈
0e9 鉦
0e9b 鈣
0e;f 銍
0ea9 鏦
0ed. 鏄
0edz 鏸
0eeb 釫
0eet 祭
0eg 釪
0ej 釘
0elr 鋞
0eo0 邆
0eo1 鎘
0eoj 鈳
0eop 鄧
0eoq 凳
0eou 登
0eou 豋
0eou 鋀
0er5 鉽
0es 釬
0eu 鈃
0eu1 銔
0eun 鉶
0ev 癸
0ev5 戣
0evp 鄈
0ewc 鈍
0ex. 銶
0eyx 鋄
0ez5 鋱
0f 釷
0f8f 錴
0f8x 錂
0f; 鉣
0f;u 鎑
0fb. 鑄
0fc 銠
0fd 鍺
0feb 銬
0ff 銈
0ffq 鐃
0ffr 鏏
0fj 錆
0fk 錶
0fnv 鍥
0fo 銡
0fo5 鐵
0fok 鎱
0fu0 鐽
0fz 鋕
0g/ 釮
0h10 釽
0h1; 钃
0h38 顉
0h5 銊
0h95 鏚
0h;0 鈲
0hb5 鋮
0hbo 鋦
0hc 鈮
0hd 銆
0he. 鎒
0he1 釿
0he8 鑕
0hek 鋠
0heo 銗
0hex 鍜
0hey 鎢
0ho5 鍼
0hr0 鎚
0hs3 鍎
0hso 鋸
0hx 鈑
0hxx 鍛
0i. 鉥
0i4 鍊
0iaa 錸
0io 鋉
0j 鈅
0je 鉏
0je, 鋤
0jex 鍰
0jg. 鋝
0ji7 鐇
0jj 錋
0jmx 鑀
0jrj 錚
0k8. 鎊
0k81 鍗
0k8o 鏑
0k8p 鑇
0k8x 鉸
0k;j 錥
0k;q 銃
0kdf 鐘
0kdq 鏡
0kdz 鐿
0kf 鉒
0kjx 鑨
0kk 銥
0km1 鈰
0kog 錞
0kok 鑲
0koo 鎬
0kox 鐓
0kq 鈧
0ks 鋅
0kt 鉉
0kxe 鏟
0l 釹
0l7 錙
0l75 鑞
0lo 銣
0m6 鍕
0m7 鐳
0m8o 鎔
0ma 鈉
0mc 鉈
0me, 鎵
0me8 鏔
0me8 鑌
0me9 錠
0mee 鍹
0meo 銅
0meq 鋎
0met 錝
0mfo 錭
0mh1 鑐
0mhr 錧
0ml 銨
0mnc 鋺
0mo 鍋
0mo0 鐹
0mq 鈂
0ms8 錪
0ms9 鑳
0ms; 鋐
0muw 鋼
0mv 鉠
0mzj 鑏
0n 釗
0n 釸
0n4, 鐌
0na 欽
0nn 鉹
0no 銘
0no 鉊
0no. 鑱
0nrc 銫
0nw 錎
0nx 錄
0o 釦
0o/o 鋁
0o18 鐀
0oj 鋗
0olv 鋘
0oob 鍔
0ooi 鐰
0p 鉺
0p 銵
0p8 錤
0pd/ 鐊
0pej 錒
0pk 鋹
0ppp 鑷
0pql 鍖
0px 鋷
0q 鉐
0q. 釩
0qo 鉛
0qx 癹
0qy 銚
0r 釭
0r.7 鎦
0r16 鏽
0r1c 鈀
0r1f 鏗
0r1p 鉚
0r1u 鑑
0r5 鉞
0ra 鈌
0rb1 鈏
0rb8 鐨
0rb; 鏹
0rbx 發
0rc8 鐉
0red 錉
0rf0 鍵
0rj5 鈱
0rk 銀
0rm1 鉔
0rmx 鋟
0roo 鏂
0rqd 鐕
0rr 鉅
0s 針
0s1g 銌
0s38 鎮
0s; 鈜
0sd 錯
0se 鉗
0sj 銪
0sjx 鏾
0sli 鍱
0sm1 鈽
0smg 鋍
0smt 鎍
0so 鈷
0su8 鐼
0sx. 鈸
0t/ 鈔
0t38 鎖
0teq 銧
0tfo 鐑
0tj 銷
0tm4 钂
0tm7 鐺
0tmf 鏜
0tmi 鏿
0tmo 鋿
0tmq 鎲
0tt5 鐖
0u,0 鐩
0u7 錨
0u8u 鎰
0uac 錵
0uao 鎝
0uax 鑊
0udv 鏌
0uef 鏵
0ufq 錓
0ufr 鎈
0ufu 鑉
0ufv 鎂
0uh5 鑶
0ukl 鋩
0uno 鍣
0uof 鑵
0urt 鎌
0us 鉡
0usi 鐷
0utt 鎡
0uu 鉼
0uu. 鑆
0uud 鐠
0uuv 鏷
0uvc 錈
0uwj 鎙
0ux. 鑮
0ux0 鑝
0v 鈇
0v 釱
0v 鉌
0v. 鈦
0v/b 銹
0v1o 銛
0v1w 鍤
0v1z 懖
0v8t 鐐
0vaa 鋏
0vbx 鑗
0vdf 鍾
0veb 銙
0vej 錡
0vet 錼
0vff 鍷
0vg5 鋨
0vi 銇
0vn 鋓
0vrb 銕
0vsu 錛
0vuf 錘
0vy 鍬
0waf 鏙
0wao 鑴
0weu 鎧
0whs 錌
0x. 釵
0x. 鉖
0xd 錔
0xd/ 鐋
0xf 鋒
0xhk 鎃
0xix 鎩
0xo 鉻
0xwx 鍐
0xxi 鎟
0xxx 錣
0y 鈥
0yy 錟
0yy, 鐒
0yyi 鑅
0z 鈊
0z/ 鉍
1 言
1,5 誑
1. 訪
1.11 詶
1.no 諮
1.rk 誏
1.x 詠
1/7z 謥
1/90 誕
1/ao 譗
1/d1 諀
1/d5 諻
10 這
10eu 證
111 川
1111 訓
1118 順
11rx 謏
123 詐
127. 誨
12c 訖
12i 誅
12o 誥
12o2 諱
12q 詵
12q8 讚
12s 許
12sy 譕
12v 詄
133 誹
14ds 譚
14et 謤
14f 諲
14fs 譯
16 誧
160 謰
16i4 讕
16j 譋
1777 讄
178x 謖
17j 謂
17uk 譨
17z 諰
180 謎
188. 譐
18; 訟
18eu 諡
18oq 說
19 訃
19c 訿
19ex 諔
19o 詀
1;2v 誒
1;o 詒
1a/ 診
1aa5 讖
1ab. 詅
1abz 諗
1ac 訛
1af 詮
1ajn 諭
1akf 誰
1akf 讎
1akx 謢
1aky 譙
1amu 論
1aoa 譣
1aro 謒
1b,o 譎
1b.f 誦
1b1c 訑
1b9j 諝
1bb 詡
1bb/ 謬
1bbd 謵
1bd 詢
1beo 詞
1bww 謅
1c/. 訊
1c29 譺
1ccd 諧
1cd 詣
1ce. 讞
1cje 謯
1cjg 謼
1cq 諕
1cr 謔
1csy 謶
1cut 譧
1cx 詖
1d4x 謾
1dbl 謁
1de9 諟
1di 課
1dol 謱
1e,. 諑
1e2o 語
1e55 諓
1e9 証
1edz 譓
1eg 訏
1ej 訂
1elr 誙
1eo7 諨
1eoj 訶
1er5 試
1es 訐
1eu1 評
1eu5 誡
1ewc 訰
1f38 謮
1f48 讀
1fb. 譸
1fd 諸
1ff 詿
1ffq 譊
1ffz 譿
1fg. 詩
1fj 請
1fk 諘
1fo 詰
1foo 譆
1fz 誌
1g. 討
1h/t 謜
1hb5 誠
1hbb 謆
1hcq 謕
1he1 訢
1he9 訴
1hek 誫
1heo 詬
1hks 謘
1hmu 諞
1ho5 諴
1hv9 讈
1i. 訹
1i4 諫
1iaa 誺
1ii 諃
1j2w 謠
1je 詛
1jex 諼
1ji7 譒
1jrj 諍
1jtv 謑
1k7e 譠
1k8. 謗
1k81 諦
1k8o 謫
1k8o 謪
1k8x 詨
1kas 誶
1kd 諳
1kd5 識
1kf 註
1kla 該
1ko, 譹
1kog 諄
1kok 讓
1koo 謞
1kot 諒
1kox 譈
1kx/ 諺
1kx; 謧
1m6 諢
1ma 訥
1mc 詑
1meb 謣
1mee 諠
1meo 詷
1mer 譝
1mfo 調
1mj7 讅
1mje 誼
1mo 詗
1mvc 詫
1n. 訒
1n.9 譅
1n.z 認
1n2i 謋
1nh1 譫
1nhc 詭
1nmx 讂
1nn 誃
1no 詔
1no 詺
1no. 讒
1nw 諂
1o10 譴
1olv 誤
1oob 諤
1ooi 譟
1oos 譂
1op 諿
1p8 諆
1pg. 謝
1pjz 讔
1ppp 讘
1pql 諶
1px 諏
1q; 諷
1qx 設
1qy 誂
1r 訌
1r5 誆
1ra 訣
1raa 誣
1rbt 誸
1rc 記
1rc8 譔
1rcz 誋
1red 諙
1ree 詆
1rj/ 訝
1roo 謳
1rpx 譀
1rqd 譖
1rr 詎
1rrp 鄋
1rrx 叟
1s 計
1s38 謓
1sc. 訧
1se 詌
1sei 謀
1sj 詴
1sl 詍
1sli 諜
1smf 講
1smg 誖
1sms 諵
1so 詁
1sof 謹
1soy 讌
1sx. 詙
1sy 詼
1t, 詏
1t/ 訬
1tj 誚
1tm4 讜
1tt0 鑾
1tt3 矕
1tt5 譏
1tt; 蠻
1tta 臠
1ttb 彎
1ttd 曫
1ttg 孿
1ttg 攣
1tth 鸞
1tti 欒
1ttl 孌
1ttv 奱
1ttw 巒
1ttx 變
1ttz 戀
1u8u 謚
1uax 護
1udl 譪
1udv 謨
1uef 譁
1uf 詳
1uf5 議
1ujb 譾
1ukq 謊
1uof 讙
1urt 謙
1us 詊
1uso 諾
1uu 誁
1uud 譜
1v/b 誘
1v1o 話
1va 訞
1vao 譑
1vc 託
1veb 誇
1vfd 譇
1vi 誄
1vl 諉
1vnn 謻
1vuf 諈
1w 訕
1wa 諛
1wfx 譭
1wh1 諯
1wq 誽
1ww 詘
1xd 誻
1xo 詻
1yy 談
1yyx 燮
1z/u 謐
2 牛
2, 牞
2/b/ 牬
22 年
22 牪
22e 牲
22o 牿
23 乍
230 迮
238 犋
23v. 犑
23z 怎
24;c 犤
27. 每
27. 牳
27.q 毓
27.t 繁
27.x 敏
27.x 瀪
27s 牰
2af 牷
2b/ 物
2b10 迤
2b1c 牠
2bww 犓
2c 乞
2c 牝
2c0 迄
2cn 刉
2cy 犥
2dsx 犦
2dx 复
2e 生
2e2e 甡
2e2o 牾
2e3 眚
2e7, 甥
2e9p 卸
2elr 牼
2f 牡
2f48 犢
2fg. 特
2hex 犌
2hu; 犡
2i 朱
2ih 鴸
2ip 邾
2kdf 犝
2kog 犉
2koo 犒
2l75 犣
2lm2 舝
2mbg 牸
2mfo 犗
2mn 制
2mng 掣
2mnk 製
2muw 犅
2n 觕
2n. 牣
2o 告
2o, 勂
2o0 造
2o2 韋
2o2. 韌
2o2. 韍
2o20 違
2o26 韗
2o28 韥
2o2; 韣
2o2f 韝
2o2f 韡
2o2g 韕
2o2i 韎
2o2i 韖
2o2i 韘
2o2k 韔
2o2o 韐
2o2p 郼
2o2s 韟
2o2u 韞
2o2w 韜
2o33 靠
2oh 鵠
2op 郜
2q 先
2q28 贊
2q2n 劗
2q2p 酇
2q2q 兟
2ree 牴
2rf0 犍
2s 午
2s/f 犕
2s0 迕
2s3 犆
2sc/ 牻
2seo 啎
2sii 橆
2sn2 舞
2so 牯
2sw 缶
2swa 缺
2swe 缽
2swf 罐
2swg 罅
2swo 缿
2swr 缸
2sws 罈
2swu 缾
2swu 罏
2swy 缹
2sy 無
2sy. 甒
2syh 鷡
2syp 鄦
2u/x 犪
2uf5 犧
2uvc 犈
2v 失
2v 矢
2v, 劮
2v.s 犐
2v0 迭
2v4c 矲
2v8d 矰
2vaf 雉
2vaf 矬
2vao 犞
2vej 犄
2veu 短
2vh 鴩
2vh 鴙
2vo 知
2vod 智
2vr1 矧
2vrr 矩
2vux 矱
2vva 矨
2vvl 矮
2vvo 矯
2x 牧
3 目
3/3. 矏
3/d1 睥
3/k 眨
30eu 瞪
30ev 睽
31wx 瞍
32c 盵
32sy 瞴
32v 眣
33 非
336 輩
33; 蜚
33ac 氍
33af 瞿
33ak 餥
33ax 矍
33bb 翡
33e 韭
33i 棐
33kk 裴
33kx 斐
33l 婓
33n 剕
33q 睍
33v 奜
33v. 瞁
33z 悲
34ds 瞫
34et 瞟
36 鼎
36d 瞷
36rx 矙
37i 睏
38 貝
38 具
38 眯
38/k 貶
380 瞇
382e 貹
383. 甖
3831 譻
3838 賏
3838 贔
383h 鸚
383l 嬰
383w 罌
384s 贉
386. 賻
388d 贈
389o 貼
38;/ 贂
38;b 瞈
38;o 貽
38at 賒
38bc 貤
38d/ 賜
38d3 賵
38e. 賕
38e5 賊
38e5 賦
38e5 賤
38e; 蠈
38eb 盻
38f5 賳
38f8 贖
38fd 賭
38g/ 財
38h 鵙
38h5 贓
38hk 賑
38hx 販
38ka 賅
38ko 賠
38kq 貥
38ks 賥
38m6 賱
38m8 賟
38mi 賝
38mj 貯
38mo 賙
38n 則
38n 盼
38n1 贍
38n2 瞵
38o1 賗
38oq 貺
38p 郥
38pk 賬
38r/ 睇
38re 貾
38ru 贐
38sf 購
38sj 賄
38ut 賺
38uu 賹
38v. 贆
38vd 賰
38x 敗
38xo 賂
38yy 賧
3;2 眸
3;o 眙
3a/ 眕
3a4d 瞺
3ajn 睮
3akf 睢
3aky 瞧
3amu 睔
3aoa 瞼
3b,o 瞲
3b. 盷
3bd 眴
3cs8 矌
3d1 眒
3de9 睼
3des 睅
3dol 瞜
3e1b 眄
3e9 眐
3ea9 瞛
3ee; 眃
3eg 盱
3ei 眛
3ej 盯
3er9 睫
3es 盰
3ewc 盹
3f8f 睦
3f8x 睖
3f;u 瞌
3fd 睹
3ff 眭
3fj 睛
3h 瞗
3h1; 矚
3he1 盺
3hff 睚
3hrk 眽
3hs3 瞃
3hv. 睙
3hx 眅
3iaa 睞
3ji 睬
3jm2 瞬
3jrj 睜
3kas 睟
3kdf 瞳
3kds 瞕
3kjx 矓
3kl 盳
3kt 眩
3kx; 瞝
3lp 郻
3maf 矐
3md8 瞑
3me8 瞚
3me8 矉
3mej 眝
3meq 睆
3mfo 瞎
3mfo 睭
3mnc 睕
3mq 眈
3nh1 瞻
3nmx 矎
3nn 眵
3no 眧
3no 眳
3noq 睌
3nx 睩
3o18 瞶
3oj 睊
3ooi 矂
3ooz 瞣
3p 眲
3q 見
3q3q 覞
3qex 覹
3qy 眺
3r5 眶
3r5 眓
3red 睧
3rj5 眠
3rk 眼
3rpx 瞰
3s38 瞋
3sma 瞞
3t, 眑
3t/ 眇
3t/1 矊
3t/t 縣
3t/z 懸
3tj 睄
3tm4 矘
3tmf 瞠
3u7 瞄
3udv 瞙
3uef 瞱
3uf 眻
3um, 矇
3uof 矔
3uuv 瞨
3uv 眹
3uvc 睠
3v. 狊
3v.h 鶪
3v.p 郹
3v3q 瞡
3v4 矄
3v8t 瞭
3vc 眊
3vg5 睋
3vrb 眱
3vuf 睡
3vy 瞅
3wq 睨
3xs1 睎
3yy 睒
4 四
4 西
4 黑
4/dx 覈
4/k 覂
40 迺
41 詈
41n 罰
41tt 羉
433 罪
438 買
438 賈
438h 鷶
444v 奰
44f 黫
47z 罳
48 粟
484d 罾
49ds 罩
49o 點
4;;/ 黲
4;jc 罷
4;jy 羆
4ab 黔
4aou 黭
4b, 罞
4b; 蜀
4b;a 歜
4b;h 鸀
4b;x 斀
4ds 覃
4dsh 鷣
4duf 罼
4e2x 覆
4e5 黓
4e9 罡
4ee/ 彯
4ee0 瓢
4ee8 顠
4ee; 飄
4eeb 翲
4een 剽
4eet 票
4eo0 還
4eo5 罭
4eob 翾
4f 墨
4f 垔
4f48 黷
4fd 署
4fe. 甄
4ff 罣
4ff9 罫
4fna 歅
4fo 黠
4fok 瞏
4fp 鄄
4fuh 鸅
4fus 睪
4fux 斁
4h9 罘
4h;0 罛
4he. 罻
4hyn 罽
4i 栗
4ih 鷅
4ii 罧
4jbg 罦
4je 罝
4kd 黯
4kdf 罿
4kf 罜
4kot 黥
4l 要
4lb/ 覅
4ldi 罺
4mnc 黦
4mq 黕
4nh1 黵
4nn 黟
4oj 罥
4pql 黮
4r.7 罶
4rj5 罠
4rq 黖
4s3 置
4s38 黰
4sb 羈
4se 黚
4so 罟
4svj 羇
4t, 黝
4ta0 邏
4taf 羅
4ud1 羃
4v. 默
4vb0 遷
4vdc 罨
4vdc 黤
4ww 黜
4x0 遝
4xxx 罬
4zaf 罹
5 壬
5 王
5,o 珈
5. 玉
5. 玊
5. 珘
5.15 班
5.29 璇
5.;p 瑯
5.rk 琅
5/7z 璁
5/; 瑰
5/d 珀
5/di 瓅
5/dq 碧
5/dx 瑔
50 廷
50 迋
50eu 璒
50h8 頲
5111 玔
52b9 琁
52i 珠
52o2 瑋
52q 珗
52q8 瓚
52s 玝
533 琲
538 珼
538l 瓔
53q 現
54ek 環
54i 瑮
55 玨
550 珽
555/ 璱
55ab 琴
55cc 琵
55rc 琶
55z/ 瑟
56.x 璷
560 璉
5777 瓃
57v 珚
584d 璔
58n 玢
58n2 璘
58r/ 珶
59 珿
59c 玼
59ex 琡
59m3 璿
59n8 璨
59o 玷
59xo 璐
5; 丟
5;m 瑀
5;o 珆
5a/ 珍
5a11 玠
5a4d 璯
5ab. 玲
5abo 琀
5ae5 玳
5ajn 瑜
5ak0 璡
5ao 珨
5aro 瑲
5b 瑪
5b,o 璚
5b. 玓
5bb 珝
5bb/ 璆
5bbs 璻
5bd 珣
5brc 玸
5cc 玭
5ccd 瑎
5ce, 璩
5ce. 瓛
5cq 琥
5cro 瑭
5cx 玻
5d1 珅
5d1 玾
5d2e 瑆
5d3 瑁
5dcc 琨
5dd 琩
5de/ 瑒
5df 理
5dkt 璟
5dkx 琝
5e,. 琢
5e2o 珸
5e55 琖
5ea9 瑽
5ed. 瑼
5eej 珩
5eeq 玩
5eg 玗
5ej 玎
5eoj 珂
5es 玕
5eu1 玶
5ex. 球
5f 玤
5f.x 璈
5fby 瓙
5ff 珪
5h 鳿
5h13 瑂
5h38 頊
5hex 瑕
5ho5 瑊
5hso 琚
5ii 琳
5j 玥
5j2w 瑤
5jbg 琈
5je 珇
5jex 瑗
5ji7 璠
5jmx 璦
5jrj 琤
5k80 瓋
5k8p 璾
5k8x 珓
5k;q 琉
5k;q 珫
5kds 璋
5kjx 瓏
5ko 琣
5kok 瓖
5kt 玹
5kx 玟
5kx5 斑
5kx; 璃
5l/x 瑙
5ldi 璅
5m6 琿
5m60 璭
5m8/ 瑏
5m8i 琛
5m8o 瑢
5me8 璸
5mee 瑄
5met 琮
5mf 珃
5mfo 琱
5mh1 瓀
5mhr 琯
5mk 玬
5mme 珊
5mnc 琬
5ms8 琠
5n, 瑑
5n0 玖
5nmv 瑍
5nmx 瓊
5no 玿
5nx 琭
5o5 珵
5oj 琄
5ooi 璪
5ooo 噩
5p 珥
5p8 琪
5qe1 珮
5qy 珧
5r 玒
5r/o 珺
5r1p 珋
5r1u 璶
5ra 玦
5rf 珒
5rj/ 玡
5rjp 琊
5rr. 璕
5s1g 珔
5s38 瑱
5s78 璜
5s8 珙
5se 玵
5sj 珛
5sl 玴
5sma 璊
5sof 瑾
5soj 瑚
5t38 瑣
5teq 珖
5tm7 璫
5tt5 璣
5u 弄
5u,0 璲
5uai 瑹
5uax 瓁
5ubx 璥
5uf 珜
5ufr 瑳
5ufu 瓂
5ujn 瑐
5umv 瑛
5uof 瓘
5uuv 璞
5v/b 琇
5vej 琦
5vf 琫
5vg5 珴
5vn 琍
5vv 瑧
5wab 琌
5waf 璀
5wao 瓗
5we8 璵
5wh1 瑞
5x 玫
5x.; 瑵
5xf; 琺
5xo 珞
5xs1 琋
5yy 琰
5z/ 珌
6 片
6 甫
6 車
6 門
6 鬥
6 爿
6 丱
6. 閍
6. 輈
6.a. 閼
6.rk 閬
6.x 敷
6/3i 闑
6/c 軌
6/d1 牌
6/di 轢
60 連
60 逋
60ev 闋
61 誾
62dx 輹
62o2 闈
62v 軼
63kx 闅
63v. 闃
64ek 轘
64ek 闤
64f 闉
65 閏
65 軠
650 閮
66 輔
660 轋
666 轟
6777 轠
67i 閫
67s 軸
688 輶
68n2 轔
68o 輍
68oq 閱
6; 閩
6;o 軩
6;u 閞
6a 閃
6a/ 軫
6ab 軡
6ab. 軨
6ae5 閥
6af 輇
6ag. 軵
6ajn 輸
6ajn 牏
6akf 閵
6amu 輪
6ao 閤
6aob 闟
6aro 牄
6b 闖
6b,i 輮
6b1 輷
6bb/ 轇
6bo 軥
6c 軋
6c 轆
6c7u 轤
6d 間
6d1 閘
6dbb 闒
6dbl 輵
6dcc 輥
6dd 閶
6di 輠
6dl 閺
6dmu 轀
6e 閂
6e5 戕
6e55 牋
6e55 輚
6e;f 輊
6ed. 轉
6eee 閆
6el0 鑋
6elr 輕
6ema 輛
6eo5 閾
6eo7 輻
6eoj 軻
6eoj 牁
6eoj 閜
6eq 軏
6er5 軾
6es 軒
6es 閈
6eu 開
6eu1 軯
6eu1 閛
6ewc 軘
6f 壯
6f48 牘
6f8x 輘
6f;u 闔
6fao 牆
6fao 轖
6fd 闍
6ff 閨
6fj 輤
6fkk 裝
6fok 轅
6fu0 闥
6fv 奘
6fy 焋
6g. 尃
6g/ 閉
6h 鵏
6h6 牖
6h;0 軱
6hbc 軛
6hbg 轏
6hc 軶
6he0 鏨
6he1 斬
6he1 斨
6he9 蹔
6hed 暫
6hef 塹
6hei 槧
6heq 磛
6hm1 輀
6hos 闢
6hoz 轗
6hs3 輴
6hsk 輾
6hv9 轣
6hx 版
6i 閑
6i4 闌
6ih 鷴
6j 閒
6jg. 將
6jg8 醬
6jg; 螿
6jgf 墏
6jgi 槳
6jgv 獎
6jgx 漿
6jh 鷳
6ji7 轓
6jj 輣
6jmx 閿
6k8. 牓
6k8x 較
6k;x 轍
6kd 闇
6kf 軴
6kla 閡
6kla 輆
6km1 鬧
6kot 輬
6kq 閌
6kx 閔
6l 妝
6l7 輜
6ldi 轈
6ma 軜
6meq 輐
6mfo 轄
6mfo 輖
6mul 輞
6mv 軮
6n. 軔
6na 軟
6nmr 鬮
6no 牊
6no 軺
6noq 輓
6nw 閻
6o 問
6o/o 閭
6o18 闠
6o8 軹
6ooo 闆
6oos 闡
6op 輯
6oq 軦
6oqx 毄
6p 聞
6p 郙
6pc 輒
6q. 軓
6qx 軗
6r/o 輑
6r1u 轞
6re5 軝
6red 閽
6ree 軧
6rpx 闞
6rpx 鬫
6s38 闐
6s8 鬨
6s8 輁
6s8c 闀
6s; 閎
6sli 牒
6su8 轒
6sx. 軷
6tj 輎
6tm7 闣
6tmf 闛
6tt6 關
6tto 轡
6udl 轕
6uf 牂
6uf5 轙
6us 牉
6uu 軿
6uu. 轛
6uuv 轐
6uwa 闕
6v 軑
6v. 狀
6v3q 闚
6v8t 轑
6vao 轎
6vc 軞
6vc 閐
6vdc 閹
6vv 轃
6vva 輳
6w 閰
6weu 闓
6wh1 輲
6wq 輗
6wq 鬩
6wq6 轚
6wq; 蟿
6wqf 墼
6wqg 擊
6wqi 檕
6wqt 繫
6wqw 罊
6xo 閣
6xo 輅
6xvo 闊
6xxx 輟
6y 焛
6z 悶
6z/ 閟
7 田
7 囗
7, 男
7. 母
71 圁
7111 甽
71ti 圞
72o2 圍
74ek 圜
74fs 圛
76 圃
777 畾
777e 疊
777f 壘
777f 壨
777h 鸓
777i 櫐
777q 礨
777t 纍
777w 罍
78m 囧
78n2 疄
78x 畟
7;8x 畯
7a 囚
7a/ 畛
7a11 界
7ab. 囹
7ac 囮
7amu 圇
7b. 畇
7b/ 囫
7bg 囝
7cc 毗
7cc 毘
7e, 圂
7e,z 慁
7e11 畀
7e2o 圄
7ed. 團
7ej 町
7ek 畏
7eo5 國
7ewc 囤
7f;u 圔
7fb. 疇
7ff 畦
7fg. 畤
7fj 圊
7fok 園
7fus 圉
7hx 畈
7i 困
7j 胃
7kl 甿
7kq 囥
7l 囡
7mnc 畹
7o 回
7o0 迴
7o18 圚
7o38 圓
7oso 圖
7s 毋
7s 由
7s 毌
7s0 迪
7s38 貫
7s8 異
7seb 甹
7sj 冑
7sj 囿
7sj 胄
7so 固
7t 累
7u 曲
7ueu 豊
7uhk 農
7us 畔
7uvc 圈
7v 因
7v 囷
7v. 畎
7vdf 畽
7vej 畸
7vna 欭
7vz 恩
7wh1 圌
7x 畋
7xo 略
7xxx 畷
7z 思
8 八
8 米
8 酉
8.11 酬
8.s 料
8/3. 糗
8/; 醜
8/d 粕
8/d1 粺
80 迷
80rx 醱
81 丫
81rx 醙
823 酢
82c 籺
82o 酷
82o0 糙
83q 粯
84 粞
84d 曾
84d. 甑
84dp 鄫
84ds 醰
84et 醥
84fd 糬
84fs 醳
86 酺
86g. 糐
87e. 糰
87s8 糞
87uk 醲
87uu 醴
88 酋
880 遒
880 醚
88;; 蠤
88g. 尊
88g0 遵
88gh 鷷
88n 粉
88n 酚
88v 奠
88v. 猷
88vp 鄭
89o 粘
89o 酟
8; 公
8;8x 酸
8;;/ 糝
8;bb 翁
8;bh 鶲
8;e. 瓮
8;h8 頌
8;qc 毤
8ai 酴
8akf 醀
8aky 醮
8b,i 糅
8b. 酌
8b/ 粅
8b1c 酏
8b2w 醄
8b9j 糈
8b9j 醑
8bb/ 醪
8bf 粈
8bg 籽
8c 麊
8c/. 籸
8cd 酯
8ce, 醵
8cf 粧
8ci8 醾
8cro 糖
8cro 醣
8crx 糠
8d2e 醒
8de9 醍
8def 糧
8df 粴
8di 粿
8dmu 醞
8e55 醆
8e7d 糟
8eb 兮
8edx 粳
8ei 粖
8ej 酊
8emc 釃
8eoj 酠
8es 酐
8fbg 酵
8fd 醏
8fj 精
8g. 籿
8g. 酎
8hd 粨
8hu; 糲
8hx 粄
8ii 醂
8j4. 釂
8je 粗
8jg. 酹
8k 粒
8k;u 醯
8kas 粹
8kas 醉
8kdz 醷
8ko 醅
8kog 醇
8kok 釀
8kx; 醨
8l 籹
8m83 醡
8mc 酡
8meo 粡
8meo 酮
8met 粽
8mh1 糯
8mh1 醹
8mme 粣
8moo 醽
8mq 酖
8mqu 醓
8mvu 醠
8n 分
8n2. 甐
8n20 遴
8n2b 翷
8n2c 亃
8n2l 粼
8n2p 鄰
8n38 貧
8nf 坌
8ngg 掰
8nh 鳻
8nh8 頒
8nmu 盆
8no 酩
8np 邠
8nu 弅
8nw 岔
8nx 攽
8nx 醁
8ny 炃
8nz 忿
8o 谷
8o5 酲
8oao 谽
8ob/ 豂
8ocf 谻
8oh 鵒
8okx 豅
8omr 谾
8on; 螸
8ona 欲
8onz 慾
8oox 釅
8op 卻
8op 郤
8oq 兌
8oqx 敓
8orx 豃
8os8 谼
8os; 谹
8pk 粻
8qx 酘
8r1c 粑
8r1u 糮
8rb. 糑
8rb/ 弟
8rb1 弚
8rb1 粌
8rb; 糨
8rbh 鵜
8rbn 剃
8rc 配
8roo 醧
8rr 粔
8s 籵
8s/f 糒
8s0 粀
8sd 醋
8se 酣
8so 酤
8soj 糊
8soj 醐
8sou 醢
8soy 醼
8th8 纇
8tj 酳
8u64 糷
8uaf 醛
8udv 糢
8ufr 醝
8ufy 糕
8ujn 糋
8uuv 醭
8v 酥
8v.8 類
8v1 粁
8v4 醺
8vc 粍
8vc 酕
8vdc 醃
8x 父
8x 敉
8x 敝
8x.; 糔
8x3 瞥
8x9 蹩
8x9d 糌
8x; 蟞
8xec 斃
8xeu 釜
8xh 鷩
8xh1 斧
8xl 嫳
8xm1 幣
8xn 鱉
8xnn 爹
8xo 酪
8xoc 鄨
8xpp 爺
8xrb 彆
8xrc 爸
8xu 弊
8xv. 獘
8xw 酗
8xxx 醊
8xz 憋
9 卜
9 止
9 足
9,o 跏
9.rk 踉
9/bx 趿
9/di 躒
9/dx 躈
90eu 蹬
90rx 蹳
91yx 躞
927. 踇
92c 趷
92i 跦
92q 跣
92q8 躦
92s2 躌
92v 跌
92vo 踟
933f 躣
933x 躩
938 貞
9380 遉
94 跴
94b; 躅
94v0 躚
960 蹥
96i4 躝
96j. 蹡
97r 卣
97r0 逌
97x5 鹹
97xa 鹼
97xr 鹺
97xy 鹵
98 趴
984d 蹭
988. 蹲
988p 躑
98n2 蹸
99 趾
99 踀
99c 跐
99ds 踔
99ex 踧
99o 跕
99t 踄
9;8x 踆
9;m 踽
9;o 跆
9a/ 跈
9abz 踗
9af 跧
9ag. 跗
9ajn 踰
9akf 踓
9amu 踚
9ao 跲
9aro 蹌
9ayj 蹐
9b,i 蹂
9b. 趵
9b., 踴
9b.f 踊
9b8 踘
9bbf 躍
9brc 跑
9c 此
9c 蹗
9c0 鈭
9c1 訾
9c3 眥
9c38 貲
9c9o 踮
9caf 雌
9cak 飺
9cd; 歶
9cdf 躔
9ce, 躆
9cee 些
9cet 祡
9cf. 躕
9ch8 蹞
9ci 柴
9cma 胔
9cn 觜
9cn 鮆
9co 呰
9cq 砦
9csx 踱
9csy 蹠
9ct 紫
9cx 跛
9d5 跩
9db/ 踢
9dbb 蹋
9de/ 踼
9de9 踶
9di 桌
9di 踝
9ds 卓
9ds0 逴
9dsh 鵫
9duf 蹕
9e 上
9e55 踐
9e7d 蹧
9e8x 躎
9e9 卡
9e99 蹝
9e;f 跮
9ea9 蹤
9ec 北
9ec8 冀
9ecj 背
9ecp 邶
9ecp 鄁
9eg 趶
9eo7 踾
9er9 踕
9et0 錖
9et3 督
9etk 裻
9etx 叔
9etx 敊
9etz 惄
9eu 趼
9ez 忐
9f38 蹟
9f8f 踛
9f8x 踜
9f9 跿
9fb. 躊
9fbg 踍
9ff 跬
9ffq 蹺
9fg. 歭
9fg. 跱
9fu0 躂
9h 鳪
9hbo 跼
9hc 跜
9he8 躓
9he9 跅
9hmu 蹁
9hsk 蹍
9hso 踞
9ht5 歲
9ht8 顪
9htb 翽
9htn 劌
9hua 蹶
9ide 蹅
9j 肯
9je 跙
9ji 踩
9ji7 蹯
9jtv 蹊
9jw 蹈
9k81 蹄
9k8o 蹢
9k8p 躋
9k8x 跤
9kas 踤
9kf 跓
9kjx 躘
9kl 踥
9ko 踣
9ko. 蹴
9kok 躟
9kt 跡
9l75 躐
9m85 躥
9mad 蹜
9mc 跎
9me3 睿
9mef 壑
9mex 叡
9mjx 敹
9mme 跚
9mnc 踠
9mua 欳
9nhc 跪
9nn 跢
9nx8 粲
9nxe 韰
9nxk 餐
9o 占
9o0 迠
9o18 蹪
9o3q 覘
9o9x 敁
9oc 乩
9ooi 躁
9p8 踑
9pc 踂
9ppp 躡
9pql 踸
9px 踙
9q 跖
9qi 跺
9qy 跳
9r.7 蹓
9r1c 跁
9ra 趹
9rcz 跽
9rk 跟
9rr 距
9s38 蹎
9sd 踖
9sl 跇
9sli 蹀
9sm1 蹛
9sma 蹣
9so 跍
9sx 歧
9sx 跂
9sx. 跋
9t 步
9th1 顰
9th8 頻
9tj 踃
9tmf 蹚
9tte 躖
9u/x 躨
9u6f 躪
9ufd 躇
9ufr 蹉
9us 跘
9uu 踫
9uu 跰
9uuv 蹼
9uvc 踡
9uvd 躤
9v 趺
9vao 蹻
9vd 踳
9vdf 踵
9veb 跨
9vej 踦
9vl 踒
9vrb 跠
9vw 蹖
9vy 踿
9w 齒
9w2c 齕
9w7i 齫
9w9 齪
9w9c 齜
9w;m 齲
9w;o 齝
9wa. 齡
9wa1 齘
9wbc 齙
9wbo 齣
9wbw 齺
9wc 齔
9wd5 齥
9wd; 齵
9weo 齬
9wh1 踹
9wh1 齗
9whf 齷
9wje 齟
9wjj 蹦
9wk/ 齴
9wno 齠
9wo8 齞
9wob 齶
9wpx 齱
9wr/ 齖
9wrk 齦
9ws8 齻
9wsd 齰
9wsl 齛
9wuu 齸
9wvj 齮
9wwq 齯
9xd 踏
9xo 路
9xoh 鷺
; 虫
;./y 蟡
;.2g 蝣
;.;p 螂
;.n8 蠀
;.rk 蜋
;.s 蚪
;/3. 螑
;/3z 螅
;/7z 蟌
;/90 蜒
;/; 螝
;/d1 蜱
;/d5 蝗
;1fd 蠩
;2 牟
;23 蚱
;2b9 蜁
;2c 虼
;2dx 蝮
;2h 鴾
;2i 蛛
;2v 矣
;2v 蛈
;2va 欸
;2vo 蜘
;33f 蠷
;33x 蠼
;38l 蠳
;3q 蜆
;4 蟔
;4b; 蠋
;4ds 蟫
;4ek 蠉
;4et 螵
;4fs 蠌
;50 蜓
;6 蜅
;6 蜵
;6h1 螹
;6hx 蝂
;777 蠝
;7e5 蟈
;7j 蝟
;7o 蛔
;7s 蚰
;7t 螺
;7u 蛐
;7v 蜠
;88 蝤
;8; 蚣
;8;b 螉
;8n 蚡
;8oq 蛻
;8x 蚥
;8x0 逡
;8xx 皴
;9o 蛅
;;2 蛑
;;; 蟲
;;;/ 參
;;;c 毿
;;;n 剼
;;;t 絫
;;;u 蠱
;;m 蝺
;;p 蝍
;a 以
;a11 蚧
;ab 蚙
;ab. 蛉
;ae8 蟘
;ag. 蚹
;ai 蜍
;ajn 蝓
;akf 蜼
;aky 蟭
;amu 蜦
;ao 蛤
;ar/ 蛜
;b 螞
;b,i 蝚
;b. 虳
;b. 蚐
;b.f 蛹
;b2w 蜪
;b9j 蝑
;bb/ 蟉
;bbf 蠗
;bgu 蜢
;bo 蚼
;bxw 蜬
;c 螰
;c.v 螏
;c7u 蠦
;cc 蚍
;ccd 蝔
;ccf 蜌
;cro 螗
;csy 蟅
;cut 蠊
;cuw 蝷
;cx 蚾
;d 蚎
;d3 蝐
;d4x 蟃
;db/ 蜴
;dba 蠍
;dbl 蝎
;de/ 蝪
;de9 蝭
;di 蜾
;dmu 蝹
;dol 螻
;e7d 螬
;e;f 蛭
;edz 蟪
;eeq 蚖
;ej 虰
;elr 蛵
;ema 蛃
;enn 蛚
;eo5 蜮
;eo7 蝠
;eoj 蚵
;er9 蜨
;es 虷
;ex. 蛷
;f 蚌
;f7. 蝳
;fd 蝫
;ff 蛙
;ffq 蟯
;fj 蜻
;fo 蛣
;fo/ 蟛
;foo 蟢
;h/t 螈
;h0 蚇
;h13 蝞
;h1; 蠾
;h38 蝢
;h7p 蝒
;h9e 蚽
;h;0 蛌
;hbc 蚅
;hcq 螔
;hd 蛨
;he1 蚚
;he9 蚸
;hek 蜄
;hex 蝦
;hey 螐
;hl5 蝛
;hmu 蝙
;hmv 蝡
;hr 蚯
;hr1 螄
;hri 蟂
;hso 蜛
;hu; 蠣
;hv. 蜧
;i 蚞
;i8; 蜙
;id 蝀
;ih1 蜥
;jbg 蜉
;jc; 螚
;jcc 能
;jcy 熊
;jcz 態
;je 蛆
;jex 蝯
;jg. 蛶
;ji7 蟠
;jiz 蟋
;jtv 螇
;k7e 蟺
;k8. 螃
;k8o 螪
;k8p 蠐
;k8x 蛟
;kd5 蟙
;kds 蟑
;kf 蛀
;kjx 蠬
;kl 虻
;ko, 蠔
;kog 蜳
;koj 蝏
;kok 蠰
;kq 蚢
;kt 蚿
;kts 蟀
;kx 蚊
;kx; 螭
;l1 虯
;l75 蠟
;m 禹
;m8f 螲
;ma 蚋
;mc 蛇
;md8 螟
;me8 螾
;me8 蠙
;mee 蝖
;mer 蠅
;mf 蚺
;mfo 蜩
;mfo 螛
;mh1 蠕
;mj 螖
;mnc 蜿
;mo 蝸
;mp 鄅
;n 虭
;n, 蝝
;n38 蝜
;n4, 蟓
;nh1 蟾
;nhc 蛫
;no 蛁
;nw 蜭
;o 台
;o0 迨
;oe. 瓵
;oi 枲
;oj 蜎
;olv 蜈
;oos 蟬
;op 邰
;oy 炱
;oz 怠
;p 即
;p8 蜞
;pf 堲
;po5 蟶
;q 允
;q 虮
;r 虹
;r1c 蚆
;r2b 蠮
;ra 蚗
;rb1 蚓
;rc8 蟤
;rdl 蝘
;re5 蚔
;ree 蚳
;rj/ 蚜
;rk 蛝
;rk0 螁
;rq 既
;rqe 暨
;rqf 塈
;rqn 鱀
;rr 蚷
;rr. 蟳
;sc. 蚘
;sc/ 蛖
;sd 蜡
;se 蚶
;se8 蟥
;sli 蝶
;sm1 螮
;sms 蝻
;so 蛄
;sof 螼
;soj 蝴
;su8 蟦
;sx 蚑
;sx. 蛂
;t, 蚴
;tj 蛸
;tm7 蟷
;tmf 螳
;tt5 蟣
;u 弁
;u45 蠛
;uax 蠖
;udv 蟆
;uf 蛘
;uf 蝆
;uf5 蟻
;ufq 蜣
;uh 鴘
;um, 蠓
;umv 蝧
;uof 蠸
;ur6 蠨
;usi 蠂
;uu 蛢
;uvc 蜷
;uvu 蟒
;v 蚨
;v.s 蝌
;v1o 蛞
;v38 貵
;v6 軬
;v7 畚
;v8t 蟟
;vaa 蛺
;vao 蟜
;vc 虴
;vc 蚝
;vdf 蝩
;vg5 蛾
;vl 蜲
;vn 蜊
;vrb 蛦
;vv 螓
;wao 蠵
;weu 螘
;wq 蜺
;xf 蜂
;xri 蟝
;xwx 蝬
;xxx 蝃
;yy, 蟧
;yyi 蠑
= ｜
= –
= ︱
= —
= ︳
= ╴
= ∥
= ∣
= ／
= ＼
= ∕
= ﹨
= ▔
= ▕
= ╱
= ╲
= 碁
=' ”
=' ′
=, ，
=, 《
=, ︽
=, 〈
=, ‘
=, ’
=, “
=, ”
=, 〝
=, 〞
=, 〃
=, ＜
=, ≦
=, ∩
=, ∪
=, ∫
=, ∮
=- ¯
=- ￣
=- ＿
=- ˍ
=- ﹉
=- ﹊
=- ﹍
=- ﹎
=- ┴
=- ┬
=- ┤
=- ├
=- ─
=- │
=- ┌
=- ┐
=- └
=- ┘
=. 、
=. 。
=. ．
=. ‧
=. ：
=. …
=. ‥
=. ﹕
=. 》
=. ︾
=. 〉
=. ＞
=. ≧
=. ⊥
=. ∠
=. ∟
=. ⊿
=. ∵
=. ∴
=/ ？
=/ ／
=/ Σ
=/ Φ
=/ Ψ
=/ Ω
=/ α
=/ β
=/ γ
=/ δ
=/ ε
=/ η
=/ θ
=/ λ
=/ μ
=/ π
=/ ρ
=/ φ
=/ ψ
=/ ω
=0 ）
=0 ○
=0 ●
=0 ◎
=0 ㊣
=0 ♀
=0 ♂
=0 ⊕
=0 ⊙
=0 ０
=0 Ⅹ
=0 十
=0 
=0 
=0 
=1 ！
=1 ▁
=1 ▂
=1 ▃
=1 ▄
=1 ▅
=1 ▆
=1 ▇
=1 █
=1 １
=1 Ⅰ
=1 〡
=1 
=1 
=1 
=2 ＠
=2 兙
=2 兛
=2 兞
=2 兝
=2 兡
=2 兣
=2 嗧
=2 瓩
=2 糎
=2 ２
=2 Ⅱ
=2 〢
=2 卄
=2 
=2 
=2 
=3 ＃
=3 △
=3 ▲
=3 ▽
=3 ▼
=3 ◢
=3 ◣
=3 ◥
=3 ◤
=3 ３
=3 Ⅲ
=3 〣
=3 卅
=3 
=3 
=3 
=4 ＄
=4 ￥
=4 〒
=4 ￠
=4 ￡
=4 ４
=4 Ⅳ
=4 〤
=4 
=4 
=4 
=5 ℅
=5 √
=5 ∞
=5 ≡
=5 ㏒
=5 ㏑
=5 ％
=5 ℃
=5 ℉
=5 °
=5 ５
=5 Ⅴ
=5 〥
=5 
=5 
=5 
=6 ︿
=6 ﹀
=6 ↑
=6 ↓
=6 ←
=6 →
=6 ↖
=6 ↗
=6 ↙
=6 ↘
=6 ６
=6 Ⅵ
=6 〦
=6 
=6 
=6 
=7 ︴
=7 ﹏
=7 ＆
=7 ﹋
=7 ﹌
=7 ═
=7 ╞
=7 ╪
=7 ╡
=7 ７
=7 Ⅶ
=7 〧
=7 
=7 
=7 
=8 ＊
=8 ※
=8 §
=8 ☆
=8 ★
=8 ◇
=8 ◆
=8 □
=8 ■
=8 ﹡
=8 ８
=8 Ⅷ
=8 〨
=8 
=8 
=8 
=9 （
=9 ㏕
=9 ㎜
=9 ㎝
=9 ㎞
=9 ㏎
=9 ㎡
=9 ㎎
=9 ㎏
=9 ㏄
=9 ９
=9 Ⅸ
=9 〩
=9 
=9 
=9 
=; ；
=; ：
== ＋
== －
== ×
== ÷
== ±
== ＝
== ≠
== ≒
== ┼
== ╳
=[ （
=[ ︵
=[ ｛
=[ ︷
=[ 〔
=[ ︹
=[ 【
=[ ︻
=[ 「
=[ ﹂
=[ 『
=[ ﹄
=[ ╭
=[ ╰
=] ）
=] ︶
=] ｝
=] ︸
=] 〕
=] ︺
=] 】
=] ︼
=] 」
=] ﹁
=] 』
=] ﹃
=] ╮
=] ╯
=` ‵
=` ∞
=` ～
=` ▏
=` ▎
=` ▍
=` ▌
=` ▋
=` ▊
=` ▉
=a Ａ
=a ａ
=b Ｂ
=b ｂ
=c Ｃ
=c ｃ
=d Ｄ
=d ｄ
=e Ｅ
=e ｅ
=f Ｆ
=f ｆ
=g Ｇ
=g ｇ
=h Ｈ
=h ｈ
=i Ｉ
=i ｉ
=j Ｊ
=j ｊ
=k Ｋ
=k ｋ
=l Ｌ
=l ｌ
=m Ｍ
=m ｍ
=n Ｎ
=n ｎ
=o Ｏ
=o ｏ
=p Ｐ
=p ｐ
=q Ｑ
=q ｑ
=r Ｒ
=r ｒ
=s Ｓ
=s ｓ
=t Ｔ
=t ｔ
=u Ｕ
=u ｕ
=v Ｖ
=v ｖ
=w Ｗ
=w ｗ
=x Ｘ
=x ｘ
=y Ｙ
=y ｙ
=z Ｚ
=z ｚ
a 人
a 入
a, 仂
a,5 俇
a,o 伽
a. 仿
a. 侜
a./y 偽
a.na 佽
a.rk 俍
a.x 倣
a/7x 傻
a/7z 傯
a/; 傀
a/bb 仍
a/bx 伋
a/c 仇
a/d 伯
a/d1 俾
a/d5 偟
a/dx 儌
a/mu 侐
a/t 係
a0et 傺
a1 信
a11 介
a1fd 儲
a1rv 候
a1x 攸
a1x. 倏
a1x/ 修
a1x0 鎥
a1x4 儵
a1x9 跾
a1xb 翛
a1xi 條
a1xj 脩
a1xn 鯈
a1xs 鞗
a1xt 絛
a1xy 焂
a1xz 悠
a2 件
a23 作
a27. 侮
a2c 仡
a2d/ 傷
a2i 侏
a2o2 偉
a2q 侁
a2q8 儹
a2s 仵
a2v 佚
a33 俳
a38 俱
a38/ 儩
a38n 側
a3q 俔
a438 價
a48 僳
a4d 會
a4dn 劊
a4dp 鄶
a4ek 儇
a4et 僄
a4l 偠
a4tf 儸
a4vc 僊
a5 任
a50 侹
a538 賃
a5h 鵀
a5i 栠
a5z 恁
a6 們
a60 僆
a6g. 傅
a6j 僩
a7 佃
a777 儡
a77t 儽
a7ek 偎
a7o 佪
a7sb 俜
a7so 個
a7uk 儂
a7z 偲
a8 仈
a84d 僧
a88 偤
a88. 僔
a8; 伀
a8bf 糴
a8j 佾
a8n 份
a8n2 僯
a8o 俗
a8oq 侻
a9 仆
a9 企
a9 促
a938 偵
a9c 佌
a9ci 偨
a9ds 倬
a9e 仩
a9e9 佧
a9ej 偝
a9ex 俶
a9o 佔
a;/5 餭
a;/; 餽
a;/o 餉
a;1x 餿
a;2 侔
a;2, 飭
a;21 飾
a;28 饡
a;2v 俟
a;5 飪
a;6 餔
a;6. 餺
a;7k 餵
a;8a 餩
a;8x 俊
a;95 饖
a;; 蝕
a;;o 飴
a;;x 餕
a;a 似
a;ai 餘
a;av 餱
a;bc 飽
a;bo 飼
a;c8 餼
a;d/ 餳
a;dc 餛
a;dl 餲
a;dx 饅
a;e5 餞
a;ec 飩
a;ej 飣
a;ej 餰
a;eu 餖
a;fo 饎
a;fq 饒
a;fu 饁
a;fu 饐
a;hv 餪
a;hx 飯
a;io 餗
a;jl 餒
a;ke 饘
a;kk 饟
a;kx 餃
a;m 偊
a;m6 餫
a;mo 餇
a;mr 館
a;n, 餯
a;n. 饞
a;na 飲
a;nw 餡
a;o 佁
a;o8 饋
a;p 餌
a;q 飢
a;r7 餾
a;r8 饌
a;ro 饇
a;rp 飹
a;s8 饙
a;sf 饉
a;sg 餑
a;sj 餬
a;t5 饑
a;ti 饓
a;u, 饛
a;uf 饈
a;uu 餅
a;uv 饃
a;ux 餀
a;v5 餓
a;va 飫
a;vc 飥
a;vl 餧
a;vo 餂
a;xj 餚
a;xx 餟
a;yy 餤
a;z/ 飶
aa11 价
aa4d 儈
aaaf 侳
aaas 傘
aab 仱
aab. 伶
aab8 僋
aae5 韱
aaf 坐
aaf 佺
aafn 剉
aafx 夎
aaje 俎
aajn 偷
aakf 倠
aaky 僬
aamu 倫
aao 佮
aaoa 儉
aaro 傖
aaw 佡
ab 今
ab 傌
ab,o 僪
ab. 令
ab.. 瓴
ab.8 領
ab.b 翎
ab.f 俑
ab.h 鴒
ab1c 他
ab1z 怹
ab38 貪
ab8 酓
ab8d 韽
ab8u 盦
abaf 雂
abb/ 僇
abd 侚
abeo 伺
abg 仔
abh 鳹
abkk 衾
abo 佝
abo 含
abo8 頷
abz 念
abzx 敜
ac 化
ac29 儗
ac38 貨
ac7z 儢
aca. 俯
acc 仳
accd 偕
ach8 傾
acrf 傭
acvc 侂
acy 儦
ad1 伸
ad4x 僈
adbb 傝
adbl 偈
adcc 倱
add 倡
ade 但
ade9 偍
adf 俚
adm; 偶
adol 僂
adsx 儤
adty 儑
ae2 伍
ae2o 俉
ae5 代
ae5 伐
ae52 牮
ae54 黛
ae55 俴
ae58 貸
ae5i 柋
ae5k 袋
ae5w 岱
ae7d 傮
ae7e 僵
ae9y 傿
ae;f 侄
ae;n 倒
aea9 傱
aed. 傳
aedx 便
aee 仁
aee5 倵
aee; 伝
aeee 仨
aef. 偫
aej 仃
aej. 佤
aelr 俓
aema 倆
aemc 儷
aenn 例
aeo7 偪
aeoj 何
aeox 使
aer9 倢
aet 佘
aeu1 伻
aeun 侀
aewc 伅
aex. 俅
aex; 禽
af 仕
af 全
af.x 傲
af38 債
af48 儥
af8x 倰
af; 佉
afb. 儔
afc 佬
aff 佳
affq 僥
afg. 侍
afj 倩
afk 俵
afnv 偰
afo 佶
afof 儓
afoo 僖
aft 傃
afus 倖
ag. 付
ag.l 姇
ag.z 怤
agor 倳
ah0 伬
ah7p 偭
ah9 伓
ah95 傶
ah9e 伾
ah9o 俖
ahaf 僱
ahbg 僝
ahbo 侷
ahcq 傂
ahd 佰
ahe1 伒
ahef 偓
ahek 侲
ahex 假
ahjx 優
ahm1 侕
ahmu 偏
ahos 僻
ahso 倨
ahuu 偋
ahv9 儮
ahww 倔
ai 休
ai 余
ai.s 斜
ai0 途
ai7 畬
ai8; 倯
aiaf 雓
aie 体
aih 鵂
aii9 儊
aiit 僸
aiw 峹
aiww 嵞
aix 敘
aiy 烋
aiz 悆
aj 仴
aj2w 傜
ajbg 俘
aji7 僠
ajiz 僁
ajj 倗
ajl 俀
ajmf 偁
ajmx 僾
ajn 俞
ajn0 逾
ajna 歈
ajnc 毹
ajnp 鄃
ajnq 覦
ajnz 愈
ajrc 俷
ajtv 傒
ak 位
ak7e 儃
ak8. 傍
ak8p 儕
ak8x 傚
ak8x 佼
akas 倅
akd 偣
akdf 僮
akds 傽
akdz 億
akf 住
akf 隹
akf0 進
akf2 犨
akff 雔
akff 雥
akfh 鷦
akfi 集
akfn 劁
akfo 售
akfs 隼
akfw 雋
akfx 隻
akfx 雙
akfy 焦
akiq 儭
akjx 儱
akk 依
akl 佞
akla 侅
ako 倍
ako. 僦
akoj 停
akok 偯
akok 儴
akot 倞
akq 伉
akt 伭
al75 儠
alo 侞
am8o 傛
am8r 倥
amaa 肏
amaf 傕
amc 佗
ame, 傢
ame8 儐
amej 佇
ameo 侗
amer 僶
amet 倧
amf 佣
amfo 倜
amh1 儒
amhr 倌
aml 侒
amnc 倇
amq 伔
ams8 倎
amsf 僿
amu 侖
amv 佒
amvc 侘
amzj 儜
an. 仞
an2i 傑
an38 偩
an4, 像
anh1 儋
anhc 佹
ann 侈
ano. 儳
anoq 俛
ant 你
antz 您
ao 合
ao/o 侶
ao1 仲
ao18 僓
ao7 畣
ao8 伿
ao9x 敆
aoba 歙
aobb 翕
aog 拿
aogg 搿
aoh 鴿
aoh8 頜
aoi 保
aoif 堡
aoij 樖
aoiy 煲
aokx 龕
aolv 俁
aomu 盒
aona 欱
aoo8 顩
aooa 僉
aooa 歛
aood 龤
aooh 鸙
aoon 劍
aoos 僤
aoou 龠
aoov 龢
aoox 斂
aoox 儼
aop 命
aop 郃
aop 偮
aoq 侃
aorc 俋
aou 弇
ap 佴
ap8 倛
ap8a 僛
apk 倀
apql 偡
aq 仉
aq; 偑
aqe1 佩
aqy 佻
ar 仜
ar 仝
ar/ 伊
ar/5 戧
ar/h 鶬
ar/n 創
ar/o 倉
ar1u 儘
ar2p 鄇
ar2v 侯
arb1 佛
arb1 伄
arc8 僎
ardl 偃
aree 低
arf0 健
arj/ 伢
ark 食
ark 佷
armx 侵
aroo 傴
arp 仰
arqd 僭
arr 佢
as 什
as 升
as/f 備
as/z 憊
as0 仗
as3 值
as38 傎
as8 供
asc. 优
asd 借
asj 侑
asl 伳
asli 偞
asm 伂
asm1 佈
asmf 傋
aso 估
aso 佑
aso 舍
aso6 舖
asof 僅
asof 儺
asoh 鵨
asoj 舒
asox 做
asr 佐
asu8 僨
asx 伎
at 尒
at/3 偗
atj 俏
atm4 儻
atm7 儅
atm8 償
atmo 倘
au.y 儰
au4n 儚
aubx 儆
auf 佯
auf5 儀
aufr 傞
aumv 偀
aup 傇
aurt 傔
aus 伴
auso 偌
auu 併
auuv 僕
auvc 倦
av 伕
av. 伏
av1 仟
av1o 佸
av1w 偛
av8t 僚
av; 俬
avaa 俠
avao 僑
avd 偆
avdc 俺
avdf 偅
aveb 侉
avej 倚
avet 倷
avf 俸
avg5 俄
avl 倭
avn 俐
avrb 侇
avuf 倕
avvd 僣
avx 傣
avxx 傸
avy 偢
avyz 僽
aw 仙
aw 仚
aw 屳
awab 侺
awaf 催
awh1 偳
awjj 傰
awq 倪
ax 汆
ax. 佟
axad 偺
axo 佫
axs1 俙
ay 伙
ayj 脊
ayjh 鶺
ayy 倓
ayy, 僗
az 伈
az/ 佖
b 馬
b, 矛
b,;; 蟊
b,ab 矜
b,i 柔
b,ih 鶔
b,m0 遹
b,mc 氄
b,mh 鷸
b,mn 劀
b,mo 矞
b,sd 矠
b,x, 務
b,x0 鍪
b,x3 瞀
b,x; 蝥
b,xb 騖
b,xf 堥
b,xh 鶩
b,xi 楘
b,xl 婺
b,xs 鞪
b. 勺
b. 勻
b.m, 勇
b.m0 通
b.mf 甬
b.mz 恿
b.rk 駺
b/ 勿
b/. 匆
b/7z 驄
b/; 騩
b/bx 馺
b/c 勼
b/d 曶
b/d5 騜
b/mf 甮
b/n 刎
b/z 忽
b0ev 騤
b1 訇
b111 馴
b1c 也
b1c0 迆
b1rx 騪
b2q 駪
b2sw 匋
b33 騑
b4 駟
b4ds 驔
b4et 驃
b4fs 驛
b4l 騕
b6 匍
b7 甸
b7s 駎
b7sb 騁
b7t 騾
b7v 駰
b8 匊
b84d 驓
b8h 鵴
b8n2 驎
b8oq 駾
b9 疋
b9; 蛋
b9e8 驥
b9j 胥
b9kq 疏
b; 勾
b;2v 騃
b;8x 駿
b;;/ 驂
b;o 駘
b;q 馻
ba/ 駗
bab. 駖
baf 駩
bag. 駙
bai 駼
bakf 騅
baoa 驗
baso 騇
bb 羽
bb,i 騥
bb,o 驈
bb. 馰
bb/; 飁
bb/d 習
bb/q 翫
bb1c 馳
bb2v 翐
bb2w 騊
bb60 翴
bb78 翼
bb8n 翂
bba/ 翏
bba5 戮
bba5 戳
bba8 顟
bbaf 翟
bbaf 雡
bbah 鷚
bbah 鸐
bbap 鄝
bbav 翭
bbbd 騽
bbbo 翑
bbcx 翍
bbd9 翨
bbh 翵
bbh8 頨
bbk 翌
bbk 翋
bbkl 翣
bbko 翯
bbks 翠
bbkt 翞
bbm6 翬
bbo 駒
bbo1 翀
bbr 羾
bbsx 翇
bbu 羿
bbva 翜
bbvc 毣
bbww 騶
bbxx 翪
bc 乜
bc7u 驢
bcc1 騛
bcu 驉
bd 旬
bd 馹
bd0 迿
bdbl 騔
bdcc 騉
bde9 騠
bdes 駻
bdi 騍
bdml 騴
bdp 郇
bduf 驆
be 刁
be;f 駤
bemc 驪
beo 司
beo7 匐
beoq 覗
bes 馯
bes5 駥
beu1 匉
beu1 駍
beu5 駴
bf 丑
bffq 驍
bfo5 驖
bfp 那
bfq 騞
bg 子
bg 孑
bg/0 遜
bg/t 孫
bg/z 愻
bgbc 孢
bgbg 孖
bgc 孔
bgee 孲
bgh0 孤
bgka 孩
bgm1 孺
bgmt 孮
bgmu 孟
bgru 孻
bgx 孜
bh/t 騵
bh9e 駓
bhbb 騸
bhbg 驏
bhbo 駶
bhex 騢
bhmu 騙
biaa 騋
bio 駷
bj 了
bj 予
bj0 孓
bje 駔
bjh8 預
bjn, 豫
bjtv 騱
bjxk 承
bk7e 驙
bk8x 駮
bkds 騿
bkf 駐
bkla 駭
bkok 驤
bkox 驐
bks 騂
bmc 駝
bme8 驞
bmeo 駧
bmfo 騆
bmi3 驦
bmo 駉
bmo 騧
bno 駋
bnx 騄
bo 句
bo, 劬
bo1 馽
boaf 雊
boh 鴝
boh1 斪
boj 駽
bona 欨
boos 驒
box 駛
bp 駬
bp8 騏
bprb 匑
bpx, 驟
bq; 颿
bqy 駣
br.7 騮
br16 驌
bra 駃
brc 包
brcn 刨
brcy 炰
brf0 騝
brmx 駸
broo 驅
brr 駏
bsb8 驧
bsc/ 駹
bsof 騹
bsoy 驠
bu 馵
buao 匒
buds 騲
buef 驊
bujn 騚
buof 驩
buu 駢
bv 馱
bv9c 騬
bvao 驕
bvc 馲
bvdc 匎
bvej 騎
bwao 驨
bwbf 雛
bwbh 鶵
bwbp 鄒
bwbw 芻
bwbx 皺
bx 馭
bx 氶
bx 駇
bx.; 騷
bxe 丞
bxec 巹
bxej 脀
bxey 烝
bxo 駱
bxw 匈
bxw 函
bxw8 顄
bxwx 騣
bxx 駁
bz/ 駜
c 乙
c 七
c 匕
c 鹿
c 气
c 虍
c.,a 瘸
c.,o 痂
c... 瘀
c..v 瘯
c..x 瘢
c./1 痺
c./; 瘣
c./k 疺
c./p 癤
c./u 癟
c./z 瘜
c.0t 瘵
c.0x 癈
c.1a 癵
c.1x 瘦
c.2. 痗
c.23 痄
c.2c 疙
c.2n 痸
c.2o 痴
c.2v 疾
c.33 痱
c.3l 癭
c.4t 瘭
c.4x 瘝
c.6 痡
c.6j 癇
c.71 痹
c.7o 痐
c.7o 痼
c.7t 瘰
c.9c 疵
c.9o 痁
c.; 痋
c.;p 廊
c.;x 痠
c.a/ 疹
c.a1 疥
c.ad 癐
c.af 痊
c.af 痤
c.af 痽
c.aj 瘠
c.an 瘉
c.ao 瘡
c.av 瘊
c.az 癒
c.b/ 瘳
c.bf 痛
c.bo 痀
c.c 疕
c.c9 癡
c.cc 疪
c.cr 瘧
c.cx 疲
c.d/ 瘍
c.de 疸
c.du 瘟
c.e. 瘃
c.e9 症
c.e9 瘲
c.ea 病
c.ej 疔
c.eo 痦
c.er 痙
c.eu 痘
c.ex 癥
c.ex 癓
c.f. 痔
c.fd 瘏
c.fg 痚
c.fv 瘈
c.fz 痣
c.fz 瘛
c.hm 瘺
c.ho 痞
c.hs 癖
c.hx 瘕
c.hx 癜
c.i8 癩
c.ii 痲
c.ii 痳
c.in 瘌
c.je 疽
c.jj 痭
c.ka 痎
c.kd 瘖
c.kf 疰
c.kp 癠
c.ks 瘁
c.ks 瘴
c.lf 癰
c.m7 癗
c.m8 痶
c.mo 痌
c.mo 瘑
c.mr 痯
c.n0 疚
c.n1 癚
c.nf 癬
c.nn 痑
c.nv 瘓
c.o8 疻
c.os 癉
c.ow 癌
c.pe 癃
c.pj 痾
c.pz 癮
c.q; 瘋
c.qx 疫
c.r 疘
c.r1 疿
c.r5 疧
c.r7 瘤
c.rc 疤
c.rd 痻
c.rk 痕
c.s. 疣
c.s/ 痝
c.s8 癲
c.s8 瘨
c.se 疳
c.sf 癱
c.sf 瘽
c.sj 痏
c.sl 疶
c.tj 痟
c.u; 癘
c.ua 瘚
c.uf 痒
c.uk 癢
c.uo 瘩
c.uo 瘔
c.ur 瘥
c.uv 瘼
c.v8 癪
c.vb 痍
c.vc 痷
c.vf 瘞
c.vg 痵
c.vl 痿
c.vn 痢
c.vt 療
c.vz 瘱
c.w 疝
c.w5 癙
c.wa 瘐
c.x. 疼
c.x/ 痧
c.x; 瘙
c.y 疢
c.y, 癆
c.yy 痰
c/ 氕
c/.0 迅
c/.; 蝨
c/; 廆
c/bb 氖
c/d1 庳
c/d; 螷
c/d; 蠯
c0 鏖
c0na 廞
c0rx 廢
c11 氘
c111 氚
c1rx 廋
c2sy 廡
c2v9 疑
c2vf 肄
c2vn 觺
c38l 廮
c38n 廁
c4 氥
c4i 麜
c50 庭
c6 庫
c7, 虜
c7j 膚
c7m8 顱
c7mh 鸕
c7mu 盧
c7s8 廙
c7v 氤
c7z 慮
c7z, 勴
c8 氣
c8 麋
c8 庮
c8n 氛
c8n2 麟
c9c 庛
c9o 店
c; 虱
c;qx 廄
ca 庂
ca4d 廥
caaf 座
caah 鷹
caaj 膺
caaz 應
cab 庈
cag. 府
caga 腐
cai 庥
cakb 虧
cbb/ 廖
cbj 序
cbrc 庖
cc 比
cc 庀
cc 麀
cc 廘
cc/, 勓
cc/d 皆
cc/h 鶛
cc7u 廬
cc8 粊
cca1 飛
ccc 庇
ccc 麤
ccf 坒
cch8 廎
ccvc 毞
ccz/ 毖
cd 旨
cd1 氠
cdff 廛
cdmu 氳
cdol 廔
ce, 豦
ce,0 遽
ce,n 劇
ce1e 氬
ce7e 麠
ce;f 庢
celr 氫
cemc 廲
ceo. 獻
ceo. 甗
ceo1 鬳
ceo5 戲
ceow 齾
cewc 庉
cf 塵
cf 丮
cf 庄
cf.x 廒
cf8x 庱
cf;u 廅
cfao 廧
cfbg 庨
cfg. 庤
cfj 氰
cfnt 緳
cfo. 廚
cgsx 庪
ch38 頃
ch3x 廈
che1 庍
chek 麎
chex 麚
chfd 廜
cho5 麙
chos 廦
ci 床
ci3 廂
ciaa 庲
cii 麻
cii/ 虨
cii2 犘
cii3 靡
cii8 糜
cii8 黂
cii; 魔
ciic 麾
ciif 塺
ciig 摩
ciin 劘
ciiq 磨
ciit 麼
ciit 縻
ciiv 穈
ciix 黀
ciiy 爢
cje 麆
cjex 皻
cjg 虖
cjga 歑
ck7v 廩
ck8p 麡
ck;q 麍
ckf 麈
ckjx 龐
ckla 氦
ckop 廓
ckx 虔
clof 廱
cma 氝
cml 氨
cn 切
cnkk 袃
cnn2 廨
cnuf 廯
cnx 氯
colh 鸆
colv 虞
colv 麌
cp 鄜
cp5z 廳
cp8 麒
cp81 廝
cpa; 廕
cpg. 麝
cq 虎
cq 麂
cq/ 彪
cq3q 覤
cqa4 虪
cqc8 贙
cqcq 虤
cqe5 虥
cqse 甝
cqy 庣
cr 虐
cr.7 廇
cr1h 鶶
cr1o 唐
cr1x 慶
cr1y 廌
cra 庚
cra8 賡
crah 鶊
crb1 氟
crbp 麛
cree 底
crj/ 庌
crmf 庸
crmh 鷛
crmp 鄘
crx 康
cs7p 鄺
csb 駂
csc/ 庬
csd 庴
csdj 廟
cse8 廣
csh 鳵
csh 鴇
csh0 庹
csm1 席
csm1 廗
csof 廑
csoq 氪
csx 度
csx 庋
csxn 剫
csy 庶
csy0 遮
csyh 鷓
cth8 顈
ctmx 廠
cu 虛
cu3q 覷
cu8 虡
cuf 庠
cuf 氧
cuna 歔
curt 廉
cuu 庰
cv 麇
cvdc 庵
cvh8 穎
cw 氙
cwa 庾
cwq 麑
cx 皮
cx. 氡
cxao 麔
cxbc 皰
cxes 皯
cxh8 頗
cxh8 潁
cxq 處
cy 麃
cyh8 熲
cyy 氮
cz 忥
cz/ 虙
d 日
d 曰
d. 昉
d.2c 暆
d.x 昹
d/bx 昅
d/c 旯
d/dx 曒
d/mo 晌
d/mo 晑
d0et 暩
d0ev 暌
d1 甲
d1 申
d1d/ 暢
d1h 鴨
d1i4 朄
d23 昨
d27. 晦
d2e 星
d2e5 戥
d2o2 暐
d2v 昳
d3 冒
d3, 勖
d3q 晛
d3vc 毷
d4 晒
d4ds 曋
d4fd 曙
d4x 曼
d4xp 鄤
d5 曳
d5 旺
d6 晡
d8;b 暡
d8n 昐
d8n2 暽
d8r/ 晜
d;8x 晙
dab 昑
dab. 昤
dak0 暹
das 昇
db. 昀
db. 旳
db/ 易
db/ 昒
db/n 剔
dba0 遏
dbaa 歇
dbac 毼
dbah 鶡
dbal 曷
dbb0 遢
dbbc 毾
dbbf 曜
dbo 昫
dboy 煦
dcc 昆
dcs8 曠
dd 昌
dd 昍
ddb/ 晹
ddbl 暍
ddd 晶
dde/ 暘
ddkt 暻
ddsx 曝
de 旦
de2o 晤
de9 是
de9/ 尟
de92 韙
de98 題
de9c 匙
de9h 鶗
de9x 晸
de;f 晊
dea9 暰
deb/ 昜
deb0 逿
ded. 暷
dede 晅
dedf 量
deg 旴
deh 鴠
dei 昧
dekf 暀
dema 昺
demc 曬
des 旱
des 旰
dev 昊
dewc 旽
df 里
dfbf 墅
dfbj 野
dfd 暑
dffq 曉
dfg. 時
dfj 晴
dfmu 曀
dh 昈
dh38 暊
dha 昃
dhb5 晟
dhc 昵
dhe1 昕
dheh 鷐
dhek 晨
dhex 暇
dhu; 曞
dhx 昄
di 果
di 杲
di4 暕
di9x 敤
dih1 晰
dih8 顆
dik 褁
dinn 夥
dj 明
dje, 勗
djex 暖
djmr 曌
djmu 盟
djmx 曖
djv. 猒
dk 昱
dkas 晬
dkd 暗
dkdf 曈
dkds 暲
dkjx 曨
dko/ 影
dko8 顥
dkok 曩
dkoo 暠
dkot 景
dkot 晾
dkox 暾
dkt 昡
dkx 旼
dkx 旻
dm6 暉
dm6 暈
dm; 禺
dm;0 遇
dm;8 顒
dm;z 愚
dmaf 曤
dmd8 暝
dme; 曇
dmee 暄
dmeq 晥
dmh1 曘
dml 晏
dmlh 鷃
dmnc 晼
dms8 晪
dmv 映
dna 欥
dno 昭
dnoq 冕
dnoq 晚
dnoy 照
doeg 冔
dol 婁
dolc 氀
dolh 鷜
dolx 數
doos 暺
doox 曮
dpx 最
dqy 晁
dr, 昮
dr1p 昴
drb1 昲
drb8 曊
drp 昂
drry 煚
druo 暱
ds 早
ds81 謈
ds8x 暴
dsma 暪
dsms 暔
dsov 暵
dsoy 曣
dt.p 曏
dteq 晃
dtm4 曭
dtt8 顯
dtta 歞
dudv 暯
duef 曄
duf 畢
duf5 曦
dufh 鷝
dum, 曚
dv 旲
dv11 昦
dv4 曛
dv8t 暸
dvd 暙
dvdc 晻
dveb 晇
dw 旵
dweu 暟
dwq 晲
dww 昢
dxao 晷
dxs1 晞
dy 炅
dy 炚
dyy 晱
e 一
e 彳
e, 豕
e,. 豖
e,.n 剢
e,0 逐
e,du 豱
e,e, 豩
e,e9 豵
e,eu 豜
e,ew 豳
e,ey 燹
e,fd 豬
e,fi 櫫
e,fu 豷
e,hc 豟
e,ht 豲
e,hx 豭
e,jv 豯
e,ka 豥
e,rc 豝
e,rk 豤
e,s8 豶
e,x1 豨
e. 彷
e/bx 彶
e/d5 徨
e/dx 徼
e0ej 銜
e11 丌
e1lb 丏
e1r; 蝁
e1re 亞
e1rf 堊
e1rz 惡
e2 五
e2dx 復
e2ep 御
e2et 禦
e2o 吾
e2o0 逜
e2o1 讆
e2o2 徫
e2o9 躗
e2oj 衛
e2op 郚
e2ox 敔
e33 徘
e33j 衢
e4 徆
e5 弋
e5 戈
e538 貣
e5af 隿
e5e5 戔
e5eu 盞
e5h 鳶
e5n 划
e7o 徊
e7u0 遭
e7ud 曹
e8; 彸
e8m. 璽
e8m0 邇
e8mh 鸍
e8mx 爾
e8x 徶
e9 下
e9 正
e99 徙
e9b 丐
e9ep 鄢
e9ey 焉
e9x 政
e9z 忑
e;;5 戩
e;;d 晉
e;;p 鄑
e;f 至
e;f5 臹
e;ff 臸
e;fn 到
e;fp 郅
e;fv 臻
e;fx 致
ea4d 徻
eaa9 從
eaap 聳
eaay 熧
eaaz 慫
eab. 彾
eai 徐
eb. 彴
ebd 徇
ebe 与
eboe 亟
ebx 攷
ecx 彼
ed;. 專
ed;h 鷒
ed;n 剸
ed;p 鄟
ed;z 惠
ede. 得
ede9 徥
edwa 忁
edx 更
edxe 甦
edxp 郠
ee 二
ee11 亓
ee2j 衙
ee9 征
ee95 武
ee9h 鵡
ee9q 虣
ee; 云
ee;; 魂
eee 三
eee5 貳
eeej 行
eej 亍
eelr 徑
eeq 元
eeq8 頑
eeqn 刓
eeqp 邧
eeqr 黿
eesj 衎
eet 示
ef9 徒
effj 街
efg. 待
eg 于
eg0 迂
egmu 盂
egp 邘
eh12 徲
ehex 徦
ehs3 循
ei 未
ei 末
ei.j 術
eiaa 徠
eix2 犛
eixa 斄
eixc 氂
eixf 釐
eixg 孷
eixl 嫠
eixn 剺
eixx 漦
ej 丁
ej2w 徭
ejc. 瓦
eje 徂
ejh8 頂
ejtv 徯
ek8. 徬
ek;x 徹
ekf 往
ekjx 徿
ekok 忀
ektj 衒
el11 卍
elr 巠
elr, 勁
elr0 逕
elr8 頸
elrh 鵛
elrn 剄
elrq 巰
em.0 邐
em.c 麗
em.h 鸝
em.p 酈
em1a 兩
ema 丙
emap 邴
emej 衕
emet 徖
emf 再
emfo 徟
emw 襾
en 互
en 歹
en. 亙
en/. 殠
en/u 殈
en2/ 殤
en2i 殊
en7j 衡
en;o 殆
ena/ 殄
ena5 殲
enaa 殮
enak 餮
enb/ 歾
enbd 殉
enc 死
endu 殟
ene. 殏
ene5 殘
enee 殛
enee 殭
ener 殌
enf8 殰
enfu 殪
enh 殦
enje 殂
enjg 殍
enk. 殧
enko 殕
enm8 殯
enm8 殥
enmv 殃
enn 列
enn0 迾
ennh 鴷
ennk 裂
ennl 姴
ennn 鮤
enny 烈
eno8 殞
enos 殫
enqx 歿
enrd 殙
enrf 肂
enrx 殔
ens1 殢
ens3 殖
ensf 殣
ensq 殑
enva 殀
enva 殎
envc 殗
eo1a 歌
eo1j 哥
eo70 逼
eo7h 鶝
eo7n 副
eoe/ 彧
eoe5 或
eoez 惑
eoj 可
eom/ 鬺
eom1 鬲
eom; 融
eomb 翮
eomh 鷊
eoms 鞷
eomx 鬷
eorj 事
eou 豆
eou0 逗
eou1 豍
eou8 頭
eouc 豌
eoup 郖
eour 豇
eout 豏
eoux 豉
eox 吏
eq 兀
eq 兀
eq; 虺
eqe, 豗
eqnc 卼
eqx 役
eqxf 垼
er5 式
er9 疌
erb1 彿
eree 彽
erf 律
erk 很
erl 妻
erlh 鶈
erlp 郪
es 干
es4z 德
es5 戎
es8j 衖
eses 幵
esh 鳱
esh8 頇
esoj 衚
esp 邗
etmo 徜
etx 後
eu 井
eu. 丼
eu/ 形
eu1 平
eu5 戒
euaf 雃
euf 徉
euh 鳽
eun 刑
eun. 刱
eunf 型
eup 邢
ev 天
evdj 衝
evej 徛
ew1x 徾
ewc 屯
ewc0 迍
ewc8 頓
ewex 微
ewex 徵
ewex 徽
ewex 黴
ewex 鰴
ewez 懲
ex. 求
ex.0 逑
ex.k 裘
ex.u 盚
ex.x 救
ex.z 慦
exej 衍
exez 愆
ez5 忒
f 土
f 士
f 丰
f. 坊
f.rk 埌
f.x 敖
f.x. 獒
f.x0 遨
f.x0 鏊
f.x1 謷
f.x8 贅
f.x; 螯
f.xb 驁
f.xg 摮
f.xn 鰲
f.xp 聱
f.xr 鼇
f.xy 熬
f/90 埏
f/; 塊
f/bx 圾
f/d1 埤
f/d5 堭
f/dx 墽
f/mv 墺
f0eu 墱
f111 圳
f2c 圪
f38 責
f38 埧
f38, 勣
f431 讟
f438 賣
f43q 覿
f43x 皾
f4e9 堽
f4et 墂
f4f 堙
f4fs 墿
f4i 塛
f6 埔
f65 載
f6nw 壛
f7. 毒
f7. 坶
f7.t 纛
f7s 毐
f7s5 戴
f7so 堌
f84d 增
f85 酨
f88. 墫
f88. 壿
f8;b 塕
f8f 坴
f8f, 勢
f8f. 埶
f8f0 逵
f8fg 摰
f8fh 鵱
f8fi 槷
f8fy 熱
f8fy 燅
f8mr 鼀
f8n 坋
f8x 夌
f9 址
f9 走
f9, 赲
f9.a 趑
f9// 趀
f90 趛
f928 趲
f92i 趎
f988 趥
f99 赴
f99s 趠
f9a/ 趁
f9af 趖
f9af 趡
f9ay 趭
f9b. 赹
f9b8 趜
f9bb 趐
f9bf 趯
f9bw 趨
f9d9 趧
f9ds 趕
f9en 趔
f9es 赶
f9ex 埱
f9fo 趌
f9fq 趬
f9je 趄
f9l1 赳
f9nn 趍
f9no 超
f9o 坫
f9oi 趮
f9px 趣
f9qi 趓
f9qy 趒
f9r5 越
f9ra 赽
f9rc 起
f9s8 趪
f9t/ 赻
f9tj 趙
f9to 趟
f9vo 趏
f9vo 趫
f9w 赸
f9ww 趉
f; 去
f;, 劫
f;2v 埃
f;5 蛓
f;;/ 墋
f;dl 朅
f;mr 鼁
f;mu 盍
faao 嗇
fab 坅
fab. 坽
fajn 堬
fak5 截
fakf 堆
fao 垥
farv 堠
fav. 垘
fayj 塉
fb. 均
fb. 圴
fb.0 鋆
fb.f 埇
fb1c 地
fbg 孝
fbgx 教
fbo 耇
fbr. 壽
fbrb 翿
fbry 燾
fc 老
fc 圠
fc 塶
fc7u 壚
fcbj 垿
fcd 耆
fce, 壉
fcef 耋
fcrf 墉
fcro 塘
fcs8 壙
fcu 墟
fcvc 耄
fcx 坡
fd 者
fd1 坤
fd4x 墁
fdb/ 埸
fdbb 塌
fdbb 翥
fdbl 堨
fde 坦
fde/ 場
fde9 堤
fdf 埋
fdf. 塒
fdi 堁
fdm; 堣
fdmu 塭
fdol 塿
fdp 都
fdy 煮
fe1e 埡
fe;5 臷
fe;f 垤
feb 考
fed. 塼
fede 垣
fedx 埂
feea 款
feeb 圬
feg 圩
fej 圢
feo1 塥
feo5 域
feo7 堛
feoj 坷
feu1 坪
fewc 坉
ff 圭
ff8x 堎
ff9 卦
ffao 墻
ffb. 壔
ffd 堵
fff 垚
fff0 遶
fff8 顤
fffb 翹
fffq 堯
fffq 墝
ffg. 封
ffg1 幫
ffg2 犎
ffgs 鞤
ffj 埥
fflv 奊
ffn 刲
ffp 邽
ffp 垹
ffr 彗
ffrz 慧
ffwc 艷
ffwp 酆
ffwu 豔
ffz 恚
fg. 寺
fg.p 邿
fh10 鏧
fh11 謦
fh12 墀
fh13 堳
fh1d 馨
fh1d 韾
fh1o 毊
fh1p 聲
fh1q 磬
fh1w 罄
fh1x 漀
fh9 坏
fh9e 坯
fhb5 城
fhc 坭
fhe1 圻
fhe9 坼
fheo 垢
fhey 塢
fhff 堐
fhmv 堧
fho5 堿
fhr 坵
fhrs 埠
fhv9 壢
fhww 堀
fi4 堜
fi5 栽
fid 埬
fj 青
fj3q 靚
fjbg 垺
fjg 垀
fjg. 埒
fjh 鶄
fji 埰
fji7 墦
fjj 堋
fjjj 靜
fjm9 靛
fjnc 靘
fjp 郬
fjrj 埩
fk 垃
fk 表
fk4k 壞
fk7e 壇
fk7v 壈
fk8. 塝
fk8o 墑
fk;j 堉
fkas 埣
fkd 堷
fkdf 墥
fkdq 境
fkds 墇
fkk5 裁
fkla 垓
fko 培
fko, 壕
fkog 埻
fkok 壤
fkoo 塙
fkop 墎
fkox 墩
fkq 坑
fks 垶
fm1e 壺
fm8i 堔
fm8o 塎
fma5 胾
fmc 坨
fmd8 塓
fme. 塚
fme; 螜
fmec 亄
fmee 壼
fmeo 垌
fmeq 垸
fmeu 壹
fmex 殼
fmex 穀
fmex 轂
fmex 嗀
fmex 彀
fmex 嗀
fmex 瞉
fmex 縠
fmex 觳
fmex 豰
fmex 鷇
fmez 懿
fmez 愨
fmh1 壖
fmk 坍
fml 垵
fmo 堝
fmo 坰
fmsj 壩
fmuw 堈
fmv 坱
fmvc 垞
fmvx 榖
fn 埆
fn9w 齧
fn; 蛪
fna 坎
fng 挈
fnhc 垝
fni 栔
fnn 觢
fno. 堍
fnt 絜
fnv 契
fnw 埳
fnx 洯
fnz 恝
fo 吉
fo, 劼
fo0 迼
fo10 壝
fo38 塤
fo5 哉
fo5 埕
foh 鴶
foh8 頡
fok 袁
fok0 遠
fomf 臺
fona 欯
foob 堮
fook 喪
foos 墠
foox 壧
fou 壴
fou. 鼕
fou. 尌
fou. 甏
fou/ 彭
fou1 鼙
fou3 瞽
fou6 鼘
fou; 鼜
foua 歖
fouf 鼞
fouk 鼚
fouo 喜
fouo 嘉
fouo 嚭
fouo 鼛
fouo 囍
four 壾
foux 鼓
foux 敼
fouy 熹
fp 邦
fpq, 墈
fpql 堪
fpt6 壣
fq 砉
fq; 堸
fqi 垛
fqy 垗
fr.7 塯
fr1u 壏
frb1 坲
frc 圯
frc 圮
frdl 堰
fre5 坁
free 坻
frf 垏
frk 垠
frm1 埽
frmx 埐
frx 埭
fs3 埴
fs38 填
fs8z 塨
fsdc 墘
fse 坩
fsli 堞
fsm1 墆
fsof 墐
fsrj 堶
fsu8 墳
ft 赤
ft 素
ft, 坳
ft98 赬
ft; 赨
ftco 赯
fteq 垙
ftfd 赭
ftft 赫
fthx 赧
fthx 赮
ftnc 赩
ftp 郝
ftx 赦
ftx; 螫
fuao 塔
fudv 塻
fuf 垟
fuf0 達
fufo 墡
fufu 壒
fus 幸
fus 坢
fus. 執
fus0 瓡
fus1 謺
fus8 贄
fus; 蟄
fusb 騺
fusf 墊
fusg 摯
fush 鷙
fust 縶
fusu 盩
fusx 報
fusz 慹
fuuv 墣
fuvc 埢
fv4 壎
fvdf 堹
fveb 垮
fvej 埼
fvf 埲
fvuf 埵
fvxx 塽
fwab 埁
fwaf 墔
fweu 塏
fwjj 塴
fwq 堄
fyy 埮
fz 志
fz3q 覟
fze. 堩
g 手
g, 扐
g,,, 拹
g,,j 搚
g. 寸
g./y 撝
g.2c 揓
g.n 刌
g.qx 搬
g.s 抖
g/ 才
g/36 鼒
g/3t 攥
g/3u 擤
g/7z 摠
g/90 挻
g/;p 擳
g/ao 撘
g/bb 扔
g/bx 扱
g/d 拍
g/d1 捭
g/d5 揘
g/di 擽
g/dx 撽
g/k 抸
g/mv 擙
g0eu 撜
g0ev 揆
g0na 撳
g0rx 撥
g1rx 搜
g27. 挴
g2bc 拖
g2c 扢
g2d/ 摥
g2o 捁
g2q8 攢
g2sy 撫
g2v 抶
g33 排
g33x 攫
g38l 攖
g3q 挸
g4;c 擺
g4b; 擉
g4ds 撢
g4ek 擐
g4et 摽
g4fs 擇
g4i 搮
g50 挺
g6 捕
g6 捫
g6 捙
g60 摙
g65 撋
g6es 搟
g6g. 搏
g6h1 摲
g6i4 攔
g6j 撊
g6xo 擱
g7. 拇
g74k 攌
g7e5 摑
g7ek 揋
g7i 捆
g7s 抽
g7s8 摜
g7t 摞
g7uk 擃
g7vz 摁
g7xo 撂
g7z 揌
g8 扒
g88 揂
g88. 撙
g88p 擲
g8n 扮
g8oq 挩
g8x 撇
g9 扑
g9 扯
g9 捉
g9ds 掉
g9ej 揹
g9j 掯
g9o 拈
g9t 捗
g;2v 挨
g;8x 捘
g;;/ 摻
g;o 抬
g;p 揤
g;q 抁
g;rq 摡
g;u 拚
ga/ 抮
ga11 扴
ga5 拰
gaa5 攕
gaaf 挫
gab 扲
gab. 拎
gabz 捻
gae; 擒
gaed 摿
gaf 拴
gag. 拊
gai 捈
gajn 揄
gakf 推
gamu 掄
gao 拾
gaoa 撿
gaou 揜
garo 搶
gaso 捨
gb,i 揉
gb. 扚
gb.f 捅
gb1 揈
gb1c 扡
gb2w 掏
gb8 掬
gb9j 揟
gbb/ 摎
gbbd 摺
gbbf 擢
gbf 扭
gbfp 挪
gbgt 搎
gbj 抒
gbo 拘
gbrc 抱
gbww 搊
gbxe 拯
gc 扎
gc 摝
gc29 擬
gc7, 擄
gc7u 攎
gc7z 攄
gc8 攗
gc9o 掂
gcc 批
gccd 揩
gcd 指
gce, 據
gci3 攠
gcro 搪
gcs8 擴
gcsy 摭
gcx 披
gd 抇
gd1 押
gd1 抻
gd5 拽
gdbb 搨
gdbl 揭
gdcc 掍
gde/ 揚
gde9 提
gdes 捍
gdf 捚
gdi 捰
gdlv 捑
gdmu 搵
gdol 摟
gdox 擻
gdpx 撮
ge1e 掗
ge2o 捂
ge5 找
ge;d 搢
ge;f 挃
ge;n 捯
gea9 摐
ged. 摶
gedx 挭
gee; 抎
geeq 抏
geg 扜
gei 抹
gej 打
gelr 挳
gemc 攦
geo1 搹
geo5 掝
geo7 揊
geq 扤
ger5 拭
ger9 捷
ges 扞
geu1 抨
gewc 扽
gex. 捄
gf; 抾
gf;u 搕
gfb. 擣
gfcd 搘
gfeb 拷
gff 挂
gff9 掛
gffq 撓
gfg. 持
gfj 掅
gfnv 揳
gfo 拮
gfo8 擷
gft 捇
gfu0 撻
ggg 掱
gh10 抓
gh12 摨
gh9 抔
gh95 摵
ghbb 搧
ghbc 扼
ghbo 挶
ghc 抳
ghca 擨
ghcq 搋
ghd. 擫
ghe0 逝
ghe0 銴
ghe1 折
ghe1 誓
ghe9 拆
ghe9 踅
ghe; 蜇
ghea 掀
ghed 晢
ghef 握
ghek 振
ghek 裚
gheo 哲
gheq 硩
ghey 焎
ghey 摀
ghez 悊
ghff 捱
ghj 掮
ghjx 擾
ghmu 揙
gho5 撼
ghos 擗
ghr0 搥
ghrk 挀
ghrw 搗
ghs3 揗
ghsk 搌
ghso 据
ghua 撅
ghuu 摒
ghv. 捩
ghww 掘
ghx 扳
ghy5 搣
gi4 揀
gim 拺
gj 抈
gj2w 搖
gjbg 捊
gje 抯
gjex 援
gjg. 捋
gji 採
gji7 播
gjj 掤
gjl 挼
gjmx 授
gjrj 掙
gjvy 撚
gjw 搯
gk 拉
gk7e 擅
gk8. 搒
gk80 擿
gk81 揥
gk8o 摘
gk8p 擠
gk8x 挍
gk9 抃
gk;x 撤
gka0 掖
gkas 捽
gkdf 撞
gkdq 摬
gkf 拄
gkjx 攏
gkk 挔
gkl 接
gkl. 攍
gko 掊
gkoj 揨
gkok 攘
gkoo 搞
gkot 掠
gkq 抗
gkt7 搐
gktf 擁
gkts 摔
gkx 抆
gkx; 摛
gkxf 攡
gl75 擸
gldi 摷
gln 拶
gm0t 擦
gm6 揮
gm7 擂
gm8 抭
gm83 搾
gm85 攛
gm8c 挖
gm8i 探
gm8r 控
gmad 摍
gmaf 搉
gmaf 攉
gme8 擯
gme9 掟
gmeb 摴
gmee 揎
gmeo 挏
gmeq 捖
gmf 抩
gmfo 搳
gmfz 攇
gmh1 擩
gmj 搰
gml 按
gmnc 捥
gmo0 撾
gmq 抌
gms8 捵
gms9 攓
gmsk 攐
gmv 抰
gmvc 挓
gmzj 擰
gn 捔
gn, 掾
gn,; 攭
gna 扻
gnd 擼
gnh1 擔
gnmv 換
gnn 拸
gno 招
gno. 攙
gnoq 挽
gnw 掐
go 扣
go,n 捌
go18 撌
go38 損
goj 捐
gon 拐
gooi 操
goos 撣
gop 揖
gor 事
gorc 挹
gorn 剚
gp 挕
gp8 掑
gp81 撕
gpk 掁
gpp 揶
gppp 攝
gpql 揕
gpsj 撱
gpx 掫
gq 拓
gqx 投
gqy 挑
gr 扛
gr/o 捃
gr11 挋
gr1c 把
gr1q 攬
gr38 摃
gra 抉
grb. 搦
grb1 拂
grc8 撰
grdl 揠
gre5 扺
gree 抵
grf0 揵
grj5 抿
grk 拫
grm1 掃
groo 摳
grp 抑
grpx 撖
grr 拒
grr. 撏
grx 捸
gs0 扙
gs1g 拵
gs38 搷
gs8 拱
gsc, 拋
gsc. 扰
gsd 措
gsd5 撠
gse 拑
gse8 撗
gsjx 撒
gsl 抴
gsli 揲
gsm1 抪
gsmf 搆
gsmg 挬
gsms 揇
gsof 攤
gsok 攮
gsx 技
gsx. 拔
gsy 拻
gt, 拗
gt/ 抄
gtj 捎
gtm4 攩
gtm7 擋
gtmg 撐
gu0t 攃
gu7 描
gu8u 搤
guai 搽
guao 搭
guax 擭
gubx 擏
gudl 擖
gudv 摸
gufk 攁
gufr 搓
gujn 揃
gurt 搛
gus 拌
gusi 擛
guu 拼
guu 掽
guuv 撲
guvc 捲
guwj 搠
gv 扶
gv1 扦
gv1o 括
gv1w 插
gv3q 摫
gv8t 撩
gva 扷
gvaa 挾
gvao 撟
gvaz 掭
gvc 托
gvc 扥
gvcc 撬
gvdc 掩
gvdf 揰
gve0 摦
gveb 挎
gvej 掎
gvet 捺
gvf 捧
gvfd 撦
gvl 捼
gvuf 捶
gvv6 攆
gvva 揍
gvy 揪
gwaf 摧
gwao 攜
gwh1 揣
gwmq 攪
gwq 掜
gwr 捏
gww 拙
gx. 扠
gx.; 搔
gxad 揝
gxf 捀
gxf0 摓
gxix 摋
gxo 挌
gxxi 搡
gxxx 掇
gyy 掞
gyy, 撈
gze. 揯
h 尸
h 戶
h 鳥
h 厂
h. 房
h.5 戍
h.s 戽
h/c 尻
h/d8 願
h/dt 原
h/dz 愿
h0 尺
h0 迉
h0o8 咫
h10 爪
h10c 爬
h13 眉
h13p 郿
h1y0 遲
h1y1 斸
h1y2 犀
h1y; 屬
h23 厏
h2sw 厒
h33 屝
h33 扉
h33 厞
h38 頁
h388 屭
h39x 夒
h3e5 戛
h3v1 奡
h3x 夏
h5 戊
h5 戌
h6 厙
h7 屇
h7p 面
h7p; 蠠
h7pq 靦
h8 屎
h84d 層
h9 不
h9e 丕
h9e5 戚
h9e5 慼
h9e8 顣
h9e9 歪
h9e9 蹙
h9ep 邳
h9h 鴀
h9lg 孬
h9mf 甭
h9mu 盃
h9o 否
h9o 扂
h9v 奀
h;0 瓜
h;0. 瓝
h;0; 瓥
h;0c 瓟
h;0v 瓞
h;8x 鵔
ha 仄
haaf 雁
hah 鴈
hah8 贗
hai 鵌
hak8 顧
hakf 雇
hakf 鵻
hb 万
hb 鷌
hb5 成
hb5p 郕
hb5u 盛
hbb 扇
hbc 厄
hbgg 孱
hbo 局
hc 尼
hc 鳦
hc0 迡
hcc 屁
hcq 虒
hcq0 遞
hcqa 歋
hd 百
hdas 鵿
hdbg 厚
hdf 厘
hdhd 皕
hdj. 厭
hdj1 厴
hdj4 黶
hdj; 魘
hdjf 壓
hdjk 饜
hdjl 嬮
hdjp 靨
hdol 屢
hdsx 鸔
hdxo 厬
he1 斤
he10 近
he18 質
he18 頎
he1; 斶
he1a 欣
he2x 履
he9 斥
he99 屣
he;f 屋
he;f 厔
he;n 剭
heba 歍
hebc 卮
hebp 鄔
heby 烏
hedl 屨
hee. 辱
hee. 尉
hee0 遐
hee2 犚
hee; 蜃
heej 脣
heek 辰
heek 褽
heeo 唇
heep 鄏
heey 熨
heez 慰
henc 屍
heo 后
heo0 逅
heof 垕
heop 郈
hesi 屧
hesl 屜
hesx 屐
hevo 屩
hfd 屠
hfdh 鷵
hfw 屆
hh 鳲
hhcq 鷈
hhe1 所
hhh 驫
hhr1 鶳
hj 肩
hjh 鵳
hjh8 顅
hjmp 鄾
hjmx 憂
hkk 扆
hkox 鷻
hks 厗
hks 屖
hl5 威
hlh5 臧
hm1. 耐
hm1/ 耏
hm11 而
hm1h 鴯
hm1l 耍
hm1n 刷
hm1v 耎
hm1z 恧
hm8 屄
hm8 鴥
hmc 鴕
hmo 扃
hmu 扁
hmu. 甂
hmu0 遍
hmub 翩
hmuh 鶣
hmvt 屪
ho5 咸
ho58 顑
ho5n 觱
hok. 璧
hok. 甓
hok0 避
hok1 譬
hok1 幦
hok8 糪
hok9 躄
hokf 壁
hokg 擘
hokh 鷿
hoki 檗
hokj 臂
hokk 襞
hokl 嬖
hokn 劈
hokr 鼊
hoks 辟
hokt 繴
hokz 憵
hom1 屌
horc 扈
hox 啟
hoz5 感
hp8 鶀
hpej 屙
hq 鳧
hr 丘
hr. 乓
hr/ 乒
hr0 追
hr8 兵
hr91 歸
hr; 蛗
hra 鴃
hrbx 殷
hrbz 慇
hrc 戺
hre1 師
hree 厎
hrei 梟
hrek 裊
hrep 鄡
hrew 島
hrh 鴭
hrh8 頧
hrhc 屔
hrj/ 厊
hrkq 覛
hrm1 帥
hrp 邱
hrs 阜
hrs8 顊
hrw 岳
hrw 峊
hs3 盾
hs3. 瞂
hs30 遁
hs38 厧
hs3h 鶞
hs8f 壂
hs8j 臀
hs8x 殿
hsc/ 厖
hsd 厝
hsk 展
hso 居
hsoh 鶋
htj 屑
hud, 勵
hud; 厲
huf5 鸃
huff 羼
huu 屏
huw; 蟨
huwa 厥
huwh 鷢
huwn 劂
hv. 戾
hvc 尾
hvcg 屘
hvuf 厜
hvv 厤
hvv9 歷
hvvd 曆
hvvq 磿
hww 屈
hwwh 鶌
hx 反
hx 尿
hx0 返
hxi 棨
hxqx 段
hxrf 肇
hxt 綮
hyy 扊
i 木
i, 朸
i,o 枷
i,y 梑
i. 朮
i. 枋
i.0 述
i.0h 鶐
i.11 栦
i.2c 椸
i.2j 檹
i.;p 榔
i.a. 棜
i.na 栨
i.rk 桹
i.s 枓
i.z 怷
i/ 杉
i/3. 櫋
i/90 梴
i/; 槐
i/;p 櫛
i/;s 櫆
i/bd 橁
i/bu 楹
i/bx 极
i/c 朹
i/d 柏
i/d1 棉
i/d1 椑
i/d5 楻
i/di 櫟
i/ds 槔
i/dx 檄
i/fd 櫡
i/k 柉
i/kf 樦
i0eu 橙
i0ev 楑
i0rk 檭
i1fd 櫧
i23 柞
i27. 梅
i2bc 柂
i2c 杚
i2i 株
i2mk 栴
i2o 梏
i2o2 椲
i2q8 欑
i2s 杵
i2s/ 鬱
i2v 柣
i2vo 椥
i3 相
i33 棑
i33f 欋
i38 梖
i38 椇
i38l 櫻
i3z 想
i4 柬
i4 柶
i4 栖
i4. 楞
i433 檌
i438 檟
i4ds 橝
i4en 檦
i4et 標
i4fs 檡
i4tf 欏
i4v0 櫏
i5 枉
i50 梃
i5oo 櫮
i5u 梇
i6 棩
i60 槤
i65 橍
i6f 梉
i6g. 榑
i6i4 欄
i6oo 櫚
i777 櫑
i77t 欙
i7e5 槶
i7ek 椳
i7i 梱
i7s 柚
i7sb 梬
i7so 棝
i7t 樏
i7v 棞
i7z 楒
i8 朳
i84d 橧
i88 楢
i88. 樽
i8; 松
i8;z 棇
i8eb 枍
i8n 枌
i8n2 橉
i8ni 椕
i8oq 梲
i8r/ 梯
i8y 槱
i9 朴
i938 楨
i9ds 棹
i9ex 椒
i9h5 檅
i9o 枮
i;8x 梭
i;;/ 槮
i;m 楀
i;p 楖
i;rq 概
ia1j 樇
ia4d 檜
iaa 來
iaa. 猌
iaa5 櫼
iaa8 賚
iaaf 桽
iaah 鶆
iaap 郲
iaaz 憖
iab 枔
iab. 柃
iabo 梒
iabz 棯
iac 杹
iae; 檎
iaex 楩
iaf 栓
iag. 柎
iai 梌
iajn 榆
iakf 椎
iaks 榫
iakw 檇
iaky 樵
iamu 棆
iaoa 檢
iaro 槍
iax 桼
ib 榪
ib,i 楙
ib,i 楺
ib,o 橘
ib,z 懋
ib. 杓
ib. 枃
ib.f 桶
ib1c 杝
ib8 椈
ib9j 楈
ib; 构
ibb 栩
ibb/ 樛
ibbd 槢
ibbf 櫂
ibd 栒
ibeo 柌
ibf 杻
ibfp 梛
ibg 李
ibg 杍
ibgt 槂
ibj 杼
ibo 枸
ibrc 枹
ic 札
ic 朼
ic.v 槉
ic0x 櫠
ic50 榳
ic7u 櫨
ic; 蚻
icc 枇
iccd 楷
iccf 梐
icd 栺
icf. 櫥
icje 樝
ickx 榩
icrf 槦
icro 榶
ict 紮
icx 柀
icx8 櫇
id 東
id 杳
id1 柙
id1 柛
id4x 槾
idbb 榻
idbl 楬
idcc 棍
ide 查
ide 柦
ide/ 楊
ideh 鸉
ides 桿
idf 梩
idf. 榯
idh 鶇
idi 棵
idol 樓
idtq 榥
ie 本
ie,. 椓
ie1e 椏
ie2o 梧
ie5 杙
ie55 棧
ie7d 槽
ie7e 橿
ie8x 檷
ie;d 榗
ie;f 桎
iea9 樅
ieb 朽
iebb 翉
iebe 極
ied. 槫
iede 桓
iedx 梗
iedz 橞
iee5 樲
iee; 枟
ieeb 杇
ieej 桁
ieeq 杬
ieet 柰
ieex 隸
ieg 杅
iei 枺
iej 朾
ielr 桱
iema 柄
iemc 欐
ien 枑
ienn 栵
ieo5 棫
ieo7 楅
ieoj 柯
ieou 梪
ieq 杌
ier5 栻
ierl 棲
ies 杆
ieu 枅
ieu1 枰
ieu5 械
iewc 杶
iex. 梂
iexj 椼
if 杜
if38 樍
if48 櫝
if8. 槸
if8x 棱
if95 樾
if;u 榼
ifao 檣
ifb. 檮
ifc 栳
ifcd 榰
ifd 楮
ifeb 栲
iff 桂
iffq 橈
iffr 槥
ifj 棈
ifnv 楔
ifo 桔
ifo. 樹
ifof 檯
ifok 榬
ifp 梆
ift 榡
ig 杽
ig. 村
ig/ 材
ih/t 榞
ih12 樨
ih13 楣
ih1; 欘
ih3x 榎
ih9 杯
ih95 槭
ih9e 柸
ih9o 桮
ih;0 柧
ihbc 枙
ihbg 樼
ihbo 梮
ihc 柅
ihcq 榹
ihe1 析
ihe8 櫍
ihe9 柝
ihe; 蜤
ihec 梔
ihed 皙
ihei 梊
ihek 桭
ihex 椵
ihez 惁
ihjx 櫌
ihm1 栭
ihmu 楄
iho5 椷
ihoc 槴
ihr0 槌
ihs3 楯
ihsk 榐
ihso 椐
ihtj 榍
ihua 橛
ihv9 櫪
ihx 板
ihxx 椴
ii 林
ii/ 彬
ii4 楝
ii7 榃
ii8n 棼
iiaa 棶
iiab 棽
iib9 楚
iic 麓
iid 棟
iide 楂
iiet 禁
iif 埜
iii 森
iii9 檚
iil 婪
iim 栜
iio 梀
iioa 樕
iion 楋
iip 郴
iiq. 梵
iiq; 檒
iiy 焚
ij2w 榣
ijbg 桴
ije 柤
ijex 楥
ijhx 棴
iji 棌
iji7 橎
ijj 棚
ijl 桵
ijrj 棦
ijtv 榽
ijvy 橪
ijw 槄
ik 柆
ik4k 櫰
ik7e 檀
ik7t 檁
ik8. 榜
ik81 楴
ik8o 樀
ik8p 櫅
ik8x 校
ik;q 梳
ik;u 橀
ikas 椊
ikd5 樴
ikdf 橦
ikdk 榱
ikdq 樈
ikds 樟
ikdz 檍
ikf 柱
ikiq 櫬
ikjx 櫳
ikl 椄
ikla 核
ikm1 柿
iknk 橠
iko 棓
iko, 檺
ikoj 楟
ikok 欀
ikoo 槁
ikop 槨
ikot 椋
ikox 橔
ikq 杭
iks 梓
ikx/ 楌
ikx; 樆
il1 朻
il7 椔
ildi 樔
im 朿
im6 楎
im7 檑
im8 柼
im83 榨
im8i 棎
im8o 榕
im8r 椌
ima 枘
imaf 榷
imc 柁
imd8 榠
ime8 檳
ime9 椗
imeb 樗
imee 楦
imeo 桐
imeq 梡
imet 棕
imf 柟
imfo 椆
imh1 檽
imhr 棺
imia 僰
imim 棘
imim 棗
imj 榾
iml 桉
imme 柵
imn 刺
imo0 檛
imoa 欞
imoo 櫺
imq 枕
imuw 棡
imv 柍
imxo 楁
imzj 檸
in 桷
in, 椽
in,; 欚
in.s 槲
in2i 榤
in4, 橡
ina 杴
ind 櫓
inh1 檐
inhc 桅
inn 栘
inn2 檞
ino. 欃
inv 櫯
io 束
io 杏
io, 柺
io,h 鶒
io/o 梠
io0 速
io18 樻
io5 桯
io8 枳
ioeb 枵
ioj 梋
ion 剌
ion 枴
ion8 賴
iona 欶
iong 揧
iooi 橾
ioos 樿
iop 楫
ioq 柷
iox 敕
iox0 遫
iox9 整
ioxh 鷘
ip 栮
ip8 棋
ipg. 榭
ipid 樄
ipk 棖
ipo5 檉
ipp 椰
ipql 椹
ipsj 橢
ipx 棷
iq 机
iq 柘
iq. 杋
iq; 楓
iqx 杸
iqy 桃
ir 杠
ir.7 榴
ir/o 桾
ir16 橚
ir1c 杷
ir1p 柳
ir1q 欖
ir1u 檻
ir33 榧
ir38 槓
ir5 框
irb1 柫
irc 杞
irdl 椻
ired 棔
iree 柢
irf0 楗
irj/ 枒
irk 根
irmx 梫
irn0 柩
iro8 櫃
iroo 樞
irp 桏
irpx 橄
irr 柜
irx 棣
is0 杖
is1g 栫
is3 植
is38 槙
is8 栱
isd 棤
isd5 橶
ise 柑
ise8 橫
isj 栯
isl 枻
isli 楪
isma 樠
ismf 構
ismg 桲
isms 楠
iso 枯
isof 槿
isoj 楜
isos 橭
isu8 橨
isx 枝
isx. 柭
it, 柪
it/ 杪
iteq 桄
itj 梢
itm/ 橕
itm4 欓
itm7 檔
itmf 樘
itmi 橖
itn, 櫞
itt5 機
iu,0 檖
iu/0 檤
iu64 欗
iuao 榙
iuax 檴
iubd 槆
iudv 模
iuef 樺
iuf5 檥
iufa 檨
iufo 橏
iufr 槎
iufx 樣
iufy 榚
ium, 檬
iuof 權
iup 榵
iur6 櫹
iuro 櫙
iurt 槏
ius 柈
iuso 楛
iuso 楉
iuu 栟
iuu 椪
iuuv 樸
iuux 欉
iuv 栚
iuvc 棬
iux. 欂
iuxf 樥
iv 杕
iv 枎
iv1o 栝
iv3q 槻
iv8t 橑
ivaa 梜
ivao 橋
ivc 枆
ivcc 橇
ivd 椿
ive0 槬
iveb 桍
ivej 椅
ivf 棒
ivff 楏
ivrb 桋
ivro 檶
ivuf 棰
ivv 榛
ivva 楱
ivw 樁
ivxx 樉
ivy 楸
iw8n 梤
iwa 楰
iwab 梣
iwao 欈
iwef 櫸
iweu 榿
iwfx 檓
iwh1 椯
iww 柮
ix 枚
ix. 杈
ix. 柊
ixf 桻
ixix 樧
ixni 樑
ixo 格
ixs1 桸
ixt/ 桫
ixx; 蠜
ixxg 攀
ixxq 礬
ixxv 樊
ixxx 棳
iy 杰
iyy 棪
iyy, 橯
iyyq 橩
iz 杺
iz/ 柲
iz/u 榓
j 月
j, 肋
j, 豸
j,/0 邈
j,/1 貏
j,/c 貔
j,/q 貌
j,3x 貜
j,ai 貅
j,an 貐
j,b. 豹
j,df 貍
j,dl 貗
j,ee 貆
j,es 豻
j,g/ 豺
j,hd 貊
j,hx 貑
j,jv 貕
j,m8 貁
j,ma 豽
j,n; 蠫
j,no 貂
j,os 貚
j,rf 墾
j,rf 貄
j,ro 貙
j,rz 懇
j,u7 貓
j,uv 貘
j,w1 貒
j,ww 貀
j,xo 貉
j. 肪
j/ 肜
j/7c 膍
j/b/ 胏
j/c. 肒
j/d 胉
j/d1 脾
j/dx 腺
j23 胙
j27. 脢
j2bc 胣
j2c 肐
j2dx 腹
j2e 胜
j2q8 臢
j2s0 遙
j2s; 颻
j2sh 鷂
j2st 繇
j2sy 膴
j2v 胅
j33 腓
j33f 臞
j3q 覓
j4;. 爵
j4b; 臅
j4et 膘
j4l 腰
j50 脡
j6 脯
j6g. 膊
j7e5 膕
j7ek 腲
j7uk 膿
j7v 胭
j7z 腮
j8n2 膦
j8op 腳
j8oq 脫
j8s 肸
j;8x 朘
j;8x 脧
j;o 胎
ja/ 胗
ja4d 膾
jaaf 脞
jab 肣
jabz 腍
jag. 胕
jajn 腧
jakf 脽
jakw 臇
jaky 膲
jaoa 臉
jayj 膌
jb.c 亂
jb.q 覶
jb.s 辭
jbb/ 膠
jbg 孚
jbgc 乳
jbgp 郛
jbo 朐
jbo 胊
jbrc 胞
jbxw 胸
jc 肊
jc7u 臚
jca. 腑
jcd 脂
jce, 臄
jcy 臕
jd1 胛
jd1 胂
jd2e 腥
jde/ 腸
jdm; 腢
jdmu 膃
jdol 膢
je 且
je, 助
je, 豚
je,0 遯
je;f 胵
jeaf 雎
jed. 膞
jee5 膩
jeej 胻
jeeq 朊
jelr 脛
jen 刞
jeo1 膈
jeo7 腷
jeou 脰
jes 肝
jesh 鶢
jesx 爰
jewc 肫
jex. 脙
jf 肚
jf; 胠
jffq 膮
jfo/ 膨
jfox 臌
jft 膆
jg 乎
jg. 肘
jg.q 虢
jh9e 胚
jh;0 胍
jhe1 肵
jhef 腛
jhek 脤
jhm1 胹
jhr0 膇
jhrk 脈
jhs3 腯
jhso 腒
jhx 服
jhxh 鵩
jhxx 腶
ji 釆
ji 采
ji/ 彩
ji4s 釋
ji7 番
ji7, 勫
ji7b 翻
ji7h 鷭
ji7p 鄱
ji7s 釉
jiax 膝
jiz 悉
jj 朋
jjbg 脬
jjg. 脟
jjh 鵬
jji7 膰
jk7e 膻
jk8. 膀
jk8p 臍
jka0 腋
jkas 脺
jkd 腤
jkd5 膱
jkdf 朣
jkdf 膧
jkdz 臆
jkjx 朧
jkla 胲
jkm1 肺
jkoj 脝
jkq 肮
jkt 胘
jktf 臃
jkts 膟
jl 妥
jl/x 腦
jl75 臘
jm8f 膣
jm8r 腔
jma 肭
jmaa 朒
jmaf 臛
jme8 臏
jmeo 胴
jmeq 脘
jmh1 臑
jml 胺
jmn2 舜
jmnc 腕
jmo 腡
jmo0 膼
jms8 腆
jmu. 臗
jmv 胦
jmx 受
jmzx 愛
jn 刖
jn, 腞
jn. 肕
jnh1 膽
jnhc 脆
jnoq 脕
jo5 脭
jo8 胑
jooi 臊
jpk 脹
jq 肌
jqx 股
jqy 朓
jr 肛
jr1; 蜰
jr1c 肥
jrb1 胇
jrb; 膙
jree 胝
jrf0 腱
jrj 爭
jrk0 腿
js; 肱
jsd 腊
jsei 腜
jsmg 脖
jsms 腩
jsoy 臙
jsu8 膹
jsx 肢
jsx. 胈
jt.p 膷
jtc 乿
jteq 胱
jtmf 膛
jtv 奚
jtvf 雞
jtvo 谿
ju8u 膉
juax 臒
judv 膜
jufo 膳
juh5 臟
juh8 頯
juks 膵
jum, 朦
jumv 朠
jus 胖
juu 胼
juv 朕
juv, 勝
juv1 謄
juv8 賸
juv; 螣
juvb 騰
juvc 腃
juvf 塍
juvl 媵
juvt 縢
juvx 滕
jv.y 然
jv4 臐
jv8t 膫
jvaa 脥
jvcc 膬
jvdc 腌
jvdf 腫
jveb 胯
jvl 腇
jvrb 胰
jvuf 腄
jvva 腠
jw 舀
jwa 腴
jwaf 膗
jww 朏
jww 胐
jxo 胳
jxwx 朡
jxxx 腏
jy 炙
k 之
k 立
k,qx 毅
k. 祊
k/ 衫
k/3u 襣
k/;k 褢
k/bb 礽
k/bx 衱
k/d 袙
k/d1 裨
k/mv 襖
k0rx 襏
k23 祚
k2bc 袘
k2dx 複
k2i 祩
k2i 袾
k2o 祰
k2o2 禕
k2o2 褘
k2q8 禶
k2rk 祣
k2v 祑
k2v 袟
k33 裶
k3q 視
k41k 褱
k4;c 襬
k4b; 襡
k4bb 禤
k4ds 禫
k4et 褾
k4f 禋
k4fs 襗
k4vc 褼
k5 衽
k6 補
k6 裫
k60 褳
k6i4 襴
k6j 襉
k7i 裍
k7n0 畝
k7o, 勯
k7o0 邅
k7o8 顫
k7oc 氈
k7oe 亶
k7oh 鸇
k7ov 稟
k7ox 皽
k7s 袖
k7uk 襛
k7uu 禮
k7v 裀
k7z 禗
k8 六
k818 齎
k81e 齏
k81n 劑
k81p 齊
k81t 齋
k81u 齍
k81y 齌
k8ek 袬
k8m. 旁
k8m. 甋
k8m0 適
k8m1 帝
k8mh 鶙
k8mo 商
k8mo 啻
k8mx 敵
k8n 衯
k8o 裕
k8ok 袞
k8oq 兗
k8oq 裞
k8v8 禷
k8x 交
k8x 襒
k8x8 頝
k8xh 鵁
k8xp 郊
k8xx 效
k9 卞
k9 祉
k938 禎
k9ej 褙
k9o 站
k;8x 竣
k;;/ 襂
k;j 育
k;jc 褦
k;q 充
k;si 棄
ka/ 袗
ka11 祄
ka4d 禬
ka4d 襘
kaa5 襳
kaaf 雜
kaas 卒
kab 衿
kag. 祔
kajn 褕
kan0 夜
kao 祫
kao 袷
kaoa 襝
kaoi 褓
kaok 褒
kaou 禴
kav. 袱
kb 禡
kb,k 袤
kb. 礿
kb. 袀
kb1c 祂
kb1c 衪
kb2w 祹
kbb 祤
kbb 翊
kbbd 褶
kbeo 祠
kbk 褭
kbo 竘
kbo 袧
kbrc 袍
kbrk 袌
kc6 褲
kcs1 褯
kcx 被
kd 音
kd 衵
kd/5 韹
kd1 神
kdb/ 裼
kdbb 禢
kdbb 褟
kdbl 竭
kdbl 褐
kdd 裮
kde 袒
kde/ 禓
kde5 戠
kde9 禔
kde9 褆
kdf 童
kdf 裡
kdfc 氃
kdfk 裏
kdi 裸
kdi 祼
kdik 裹
kdk 衰
kdmu 褞
kdna 歆
kdno 韶
kdo8 韻
kdol 褸
kdq 竟
kds 章
kds/ 彰
kds0 遧
kds8 贛
kdsp 鄣
kdsx 竷
kdsx 襮
kdsz 戇
kduv 韺
kdux 頀
kdxf 韸
kdz 意
kdzh 鷾
ke8x 禰
ke99 褷
keg 衧
kema 裲
kemc 襹
keo7 福
keo7 褔
keou 裋
keu5 祴
kev 祆
kf 主
kf 社
kf48 襩
kf75 襶
kf8k 褻
kf8x 裬
kf; 祛
kf; 袪
kfb. 禱
kfd 褚
kff9 褂
kffq 襓
kfj 靖
kfk 裱
kfnv 禊
kfnv 褉
kfo 袺
kfo8 襭
kfoo 禧
kfqx 殶
kh 鴗
khcq 褫
khcq 禠
khd 袹
khe. 褥
khe1 祈
khek 祳
khek 裖
khmu 褊
khso 裾
khu; 禲
khuu 竮
ki3q 親
kih1 新
kiit 襟
kimm 襋
kio 竦
kixg 襻
kj91 讋
kj98 龔
kj9; 蠪
kj9f 壟
kj9h 鸗
kj9k 襲
kj9p 聾
kj9q 礱
kj9t 龒
kj9v 龑
kj9x 龍
kj9x 龘
kje 祖
kje 袓
kjex 禐
kjex 褑
kjrj 竫
kk 衣
kk7e 襢
kk81 禘
kk81 褅
kk;q 裗
kkas 祽
kkiq 襯
kkjx 襱
kkmo 裔
kkok 禳
kkt 袨
kkx; 褵
kl 亡
kl 妾
kl/, 劾
kl/8 賌
kl/8 頦
kl/a 亥
kl/a 欬
kl/n 刻
kl/u 衁
kl3 盲
kli 杗
klj 肓
klj5 望
kll 妄
klo 袽
klo. 嬴
klo. 羸
klo. 贏
klo. 蠃
klo. 臝
klo. 鸁
klp 邙
klq 巟
klr5 氓
klz 忘
km1 市
km6 禈
km6 褌
km8 袕
km8o 褣
kma 衲
kmc 袉
kmf 袡
kmfo 裯
kmfo 禂
kmh1 襦
kmo 禍
kmq 衴
kms; 竤
kn 初
kn, 禒
kn, 褖
kn4, 襐
knh1 襜
knhc 祪
knnk 袲
kno 祒
kno 袑
knx 祿
ko1 祌
ko1 衶
ko1k 衷
ko5 裎
kob. 孰
kobf 塾
kobg 享
kobg 撉
kobh 鶉
kobj 亨
kobp 郭
kobs 嚲
kobx 敦
koby 烹
koby 熟
kobz 憝
koe. 瓿
koj 裐
kok 哀
kom, 豪
koma 歊
komc 亳
komc 毫
komj 亭
komj 膏
komo 高
komp 鄗
komq 亮
koms 髜
komx 敲
komx 毃
kon 剖
koo, 勷
koo0 瓤
kooi 襙
kook 襄
koos 禪
koos 襌
kop 部
koq 祝
koqq 競
kork 裛
kot 京
kot, 勍
kot. 就
koth 鶁
koth 鷲
kox 敨
kp 祁
kp8 祺
kq 亢
kq 祏
kq0 迒
kq; 颯
kqh8 頏
kqp 邟
kqx 祋
kqy 祧
kr/o 裙
kr1u 襤
kra 袂
krb; 襁
krc 祀
krdl 褗
kre5 祇
kre5 衹
kree 祗
kree 袛
krk 裉
krk0 褪
krmx 祲
krr. 襑
krvk 褎
ks 辛
ks,s 辦
ks.s 辨
ks1g 袸
ks1s 辯
ks38 禛
ks; 竑
ksei 禖
kshs 瓣
ksio 辣
ksli 褋
kso 祐
kso 祜
ksts 辮
ksx 衼
ksx. 祓
ksx. 袚
kt 玄
kt 亦
kt, 袎
kt2k 玈
kt7 畜
kt7p 鄐
kta. 甕
ktaf 雍
ktaf 壅
ktak 饔
ktaw 罋
ktkt 玆
ktm1 帟
ktm2 牽
ktm7 襠
ktt/ 玅
ktt5 禨
ktu 弈
ktv 奕
ktys 率
ku,0 禭
ku,0 襚
ku45 襪
kuao 褡
kuf 祥
kuf; 襼
kufy 禚
kum; 襺
kus 袢
kuuv 襆
kuvc 裷
kv 衭
kva 祅
kvaa 裌
kvdc 裺
kvhd 襫
kwh1 端
kwk 裒
kwme 亹
kx 文
kx. 衩
kx2 袶
kx64 斕
kx;a 斔
kx;a 螤
kxe5 斌
kxe8 贇
kxh 鳼
kxh/ 彥
kxh8 顏
kxhe 產
kxhn 剷
kxhu 斒
kxo 吝
kxo 袼
kxt 紊
kxw; 离
kxwe 斖
kxwf 離
kxxi 褬
kxxx 裰
kxz 忞
kyy 裧
kz/ 祕
l 女
l. 妨
l./y 媯
l.29 嫙
l.rk 娘
l/3z 媳
l/7c 媲
l/; 媿
l/b/ 姊
l/bb 奶
l/d1 婢
l/d1 婂
l/d5 媓
l/dx 嬓
l/mo 姠
l0 巡
l0eu 嬁
l1 娮
l1oo 嘂
l1rx 嫂
l1x 收
l22 姩
l2e 姓
l2i 姝
l2q 姺
l2sy 嫵
l3q 娊
l41v 嬽
l4ek 嬛
l4et 嫖
l5 妊
l50 娗
l6i 嫻
l6j 嫺
l7 甾
l7,, 嬲
l7. 姆
l7e9 疀
l7h 鶅
l7j 媦
l7s 妯
l7sb 娉
l7so 婟
l7t 嫘
l7v 姻
l7x0 邋
l8; 妐
l8n 妢
l8r/ 娣
l9 娖
l938 媜
l9ds 婥
l9ex 婌
l;2v 娭
l;a 姒
l;o 始
la11 妎
laa5 孅
lab 妗
lab. 姈
laf 姾
lajn 媮
lakx 孇
laky 嫶
lao 姶
laoa 嬐
laoi 媬
laou 媕
lb 媽
lb,i 媃
lb. 妁
lb1c 她
lb9j 婿
lbb/ 嫪
lbbf 嬥
lbf 妞
lbfp 娜
lbg 好
lbj 妤
lbo 姁
lbww 媰
lc.v 嫉
lcc 妣
lcit 嬤
lcjg 嫭
lcrf 嫞
lcrx 嫝
lcsy 嫬
lcut 嬚
ld1 妽
ld3 媢
ld4x 嫚
ldd 娼
lde 妲
lde/ 婸
lde9 媞
ldf 娌
ldi 巢
ldi 婐
ldi, 勦
ldin 剿
ldip 鄛
ldmu 媼
le1e 婭
le8x 嬭
le9 姃
le9y 嫣
le;f 姪
led. 嫥
lede 姮
lee; 妘
leeq 妧
lei 妹
lei 妺
lelr 娙
lemc 孋
ler9 婕
les 奸
les5 娀
leu 妍
lf 妦
lf8x 婈
lfao 嬙
lfb. 嬦
lfc 姥
lfd 媎
lff 娃
lffq 嬈
lfj 婧
lfk 婊
lfo 姞
lfof 嬯
lfok 媴
lfoo 嬉
lft 嫊
lfus 婞
lh 妒
lh/t 嫄
lh13 媚
lh1; 孎
lh7p 媔
lhc 妮
lhe. 媷
lhe1 妡
lhek 娠
lheo 姤
lhex 婽
lhff 娾
lhmu 媥
lhrk 嬝
lhvc 娓
lio 娕
lio8 嬾
liox 嫩
lj2w 媱
lj5 婬
lje 姐
ljex 媛
lji 婇
lji7 嬏
ljl 娞
ljmx 嬡
ljw 嫍
lk7e 嬗
lk8o 嫡
lk8x 姣
lkd5 嬂
lkds 嫜
lkf 妵
lko 婄
lkoj 婷
lkok 孃
lkot 婛
lkt 妶
lkx 妏
ll 奻
lll 姦
lm8o 嫆
lma 妠
lmd8 嫇
lme, 嫁
lme8 嬪
lme9 婝
lmeb 嫮
lmee 媗
lmeo 姛
lmet 婃
lmf 姌
lmfo 婤
lmh1 嬬
lmhr 婠
lmi3 孀
lmj7 嬸
lml 姲
lmme 姍
lmnc 婉
lmo 媧
lms8 婰
lmv 姎
lmzj 嬣
lnhc 姽
lnn 姼
lno 妱
lno 姳
lnoe 嬔
lnoq 娩
lnt 妳
lo 如
lo18 嬇
log 挐
loh 鴽
loj 娟
lolv 娛
lom1 帤
looi 嬠
loos 嬋
loox 孍
lorc 邕
lorf 雝
lot 絮
loz 恕
lp8 娸
lpej 婀
lpsj 嫷
lpx 娵
lqe1 姵
lqy 姚
lr 妅
lr.7 媹
lr.n 嬼
lr11 姬
lr1u 嬧
lr2z 嫕
lrb. 嫋
lrbt 娹
lrc 妃
lred 婚
lrfe 嫿
lrm1 婦
lroo 嫗
lrr 姖
lruo 嫟
lsc/ 娏
lse 姏
lse8 嫹
lsei 媒
lsj 姷
lsli 媟
lsmf 媾
lsms 婻
lso 姑
lsoj 媩
lsos 嫴
lsov 嫨
lsoy 嬿
lsx 妓
lt/ 妙
ltm1 嫦
lu7 媌
ludv 嫫
luef 嬅
lufo 嫸
lufv 媄
lujn 媊
lup 媶
lurt 嫌
lus 姅
luso 婼
lutz 嬨
luu 姘
luvc 婘
luvf 嬞
lv 姀
lv1 奷
lv1o 姡
lv8t 嫽
lva 妖
lvao 嬌
lvaz 婖
lvc 奼
lvd 媋
lveb 姱
lvej 婍
lvg5 娥
lvl 婑
lvn 娳
lvrb 姨
lvuf 娷
lvv 嫀
lw 奾
lw; 媸
lwao 孈
lwex 媺
lwh1 媏
lwhs 婩
lwq 婗
lx 奴
lx, 努
lxb 駑
lxbg 孥
lxg 拏
lxm1 帑
lxq 砮
lxrb 弩
lxz 怒
ly 災
lyy 婒
lz/ 妼
m 雨
m. 雱
m./y 寪
m/c 宄
m0et 察
m0eu 霯
m1 巾
m1 霅
m1/d 帕
m1/p 幯
m122 幃
m12v 帙
m12y 幠
m175 幗
m18q 帨
m198 幀
m19o 帖
m1;/ 幓
m1af 帷
m1ao 帢
m1c. 幮
m1cx 帔
m1d3 帽
m1dl 靄
m1dq 幌
m1dt 幜
m1dx 幔
m1e5 帴
m1e7 幅
m1ej 帄
m1f. 幬
m1f8 幘
m1fd 帾
m1h8 幁
m1hf 幄
m1j7 幡
m1jw 幍
m1k5 幟
m1kf 幢
m1ks 幛
m1m, 幏
m1m8 幎
m1mq 帎
m1mz 幰
m1n1 幨
m1oi 幧
m1os 幝
m1pk 帳
m1q. 帆
m1r8 幊
m1rc 帊
m1s. 帗
m1s8 幩
m1tj 帩
m1u, 幪
m1u5 幭
m1uu 帡
m1uv 幙
m2 牢
m27. 霉
m33 霏
m36 鼏
m4ek 寰
m50 霆
m528 寶
m6 軍
m6/i 寱
m60 運
m61 寣
m6cx 皸
m6ea 寎
m6ei 寐
m6eo 寤
m6h 鶤
m6h8 顐
m6p 鄆
m6rx 寢
m7 雷
m7s 宙
m7s8 實
m7s8 霬
m8 穴
m8 冞
m8/c 究
m8/k 窆
m8/x 窗
m8/x 竅
m823 窄
m82o 窖
m83 窅
m88s 窲
m8ai 窱
m8an 窬
m8bc 窇
m8c 穵
m8di 窠
m8dl 窶
m8e5 窢
m8ea 窉
m8ec 窀
m8ef 窒
m8eg 穻
m8eq 竀
m8es 罕
m8f8 竇
m8fa 窾
m8ff 窐
m8fg 窙
m8fv 窫
m8h 窵
m8h0 窊
m8h0 窳
m8hw 窟
m8j; 竊
m8jz 窸
m8kd 窨
m8ks 窣
m8kx 窔
m8m1 帘
m8mo 窩
m8n 穸
m8n 雰
m8nw 窞
m8o 容
m8o 冏
m8pb 窮
m8pe 窿
m8pp 窷
m8qy 窕
m8r 空
m8r/ 穿
m8rb 穹
m8ro 窘
m8rp 窌
m8s8 窴
m8t, 窈
m8u0 邃
m8uy 窯
m8v. 突
m8va 穾
m8vc 竁
m8vh 鶟
m8vq 窺
m8w5 竄
m8ww 窋
m8xf 窪
m8xg 窏
m938 寊
m9ex 寂
m9xo 露
m; 禸
ma 內
maa 肉
mab. 零
mahd 宿
makf 霍
makf 靃
makh 鶴
makw 寯
marx 寑
mb, 雺
mb,, 霧
mb,3 霿
mbb/ 寥
mbbd 霫
mbg 字
mbrc 雹
mc 它
mc29 寲
mdc 電
mde9 寔
mdk8 冥
mdkp 鄍
mdkq 覭
mdkr 鼆
mdl 宴
mdm; 寓
mdol 寠
me, 家
me11 斲
me1r 黽
me75 戭
me78 寅
me9 定
me98 顁
me;f 室
mede 宣
mee, 冢
mee. 冠
mee0 靆
mee8 賨
mee; 雲
meeb 雩
meeb 霒
meec 甩
meep 鄠
meep 鄳
meeq 完
meet 宗
meex 寇
meex 靉
meg 宇
mej 宁
meo 同
meo0 迵
meo5 戙
meo7 富
mer9 寁
merl 霋
met8 賓
mewx 霺
mf 冉
mf 用
mf4z 憲
mfo 周
mfo 害
mfo/ 彫
mfo0 週
mfob 翢
mfof 雕
mfoh 鵰
mfoh 鶷
mfon 割
mfoo 豁
mg. 守
mh3n 寡
mh;0 寙
mhb5 宬
mhe. 霨
mhek 宸
mhek 震
mhex 霞
mhm1 需
mhos 霹
mhr 官
mhr0 逭
mhrz 悹
mhv9 靂
mi 宋
mi3 霜
mii 霖
mii/ 霦
mj 骨
mj,f 霾
mj/. 骫
mj/1 髀
mj4; 髑
mj6. 髆
mj7u 體
mj9c 骴
mja1 骱
mjci 髍
mjcx 骳
mjdi 髁
mjdl 髏
mje 宜
mjes 骭
mjex 骾
mjfq 髐
mjg 雽
mjh 鶻
mjh8 顝
mjhk 霢
mji 寀
mji7 審
mjiq 覾
mjka 骸
mjko 髇
mjkq 骯
mjkx 骹
mjm. 髖
mjm8 髕
mjmo 髂
mjqx 骰
mjs0 髓
mjso 骷
mjur 髊
mjuu 髒
mjuu 骿
mjvb 骻
mjxo 骼
mk 丹
mk/ 彤
mk8p 霽
mkab 霠
mkh 鴅
mkjx 寵
mkjx 靇
mkl 霎
mkn 刐
mkop 霩
mks 宰
mkux 雘
mkx 雯
ml 安
mlh 鴳
mlh8 頞
mli 案
mme 冊
mme, 霥
mmen 刪
mmv 雵
mnbc 宛
mnbh 鵷
mnbn 剜
mnbz 惌
mno. 冤
mo 咼
mo/o 宮
mo0 迥
mo0 過
mo38 霣
mon 剮
mooa 靈
mooo 霝
moop 酃
mop5 霵
mq 冗
mq 冘
mq 宕
mqh 鴆
mqp 邥
mqx 沊
mqy 宨
mqy 雿
mr 雪
mr.7 霤
mr/o 宭
mr11 宦
mr11 宧
ms38 寘
ms8 典
ms8. 寒
ms81 謇
ms88 賽
ms89 蹇
ms8b 騫
ms8c 寋
ms8f 塞
ms8g 搴
ms8h 鶱
ms8i 寨
ms8k 褰
ms; 宏
mse 雸
msj 宥
msj 霸
msjx 霰
mtj 宵
mtj 霄
mu 皿
mu3. 寬
mud1 冪
mudv 寞
mukl 罔
mumv 霙
muw 岡
muwn 剛
mv 央
mv8t 寮
mva 宎
mvc 宅
mvc 宒
mvej 寄
mvh 鴦
mvmu 盎
mwby 寫
mwq 霓
mx9o 霑
mxh9 靋
mxi 霂
mxj5 霪
mxk1 霈
mxmo 霘
mxo 客
mxo8 額
mxpl 霮
mxr; 霐
mxrx 寖
mxs8 霟
mxx 网
mz/ 宓
mz/; 蜜
mz/w 密
mzmf 甯
mzmh 鸋
mzmj 寧
mzmu 寍
n 刀
n 夕
n 角
n 魚
n, 觔
n, 彖
n,;; 蠡
n,;n 劙
n,h 鶨
n. 刃
n. 魴
n.s 斛
n.z 忍
n/bx 觙
n/d5 鰉
n/di 觻
n/di 鱳
n/h5 鱵
n/t 鯀
n0 久
n0et 鰶
n0rx 鱍
n0y 灸
n2 舛
n23 鮓
n2d/ 觴
n2dx 鰒
n2i 桀
n2i 鮢
n33 鯡
n38 負
n4, 象
n4b; 觸
n4ek 鱞
n4et 鰾
n4x 鰥
n5oo 鱷
n6 鯆
n60 鰱
n7a 鮂
n7uu 鱧
n7z 鰓
n88. 鱒
n8n 魵
n8n2 鱗
n8oq 鮵
n8tu 彝
n9 外
n9et 鮛
n; 触
n;o 鮐
n;p 鯽
na 欠
na1i 鰷
na4d 鱠
na; 飧
nabz 鯰
nac 魤
nag. 鮒
nai 鮽
nao 鮯
narv 鯸
nb,i 鰇
nb,o 鱊
nb. 魡
nbbd 鰼
nbc 夗
nbc3 眢
nbcb 駌
nbch 鴛
nbcl 妴
nbcz 怨
nbo 鮈
nbrc 鮑
nc2c 彘
nc7u 鱸
ncd 鮨
nckx 鰬
ncu 鱋
nd 魯
nd1 魻
nd4x 鰻
ndbb 鰨
ndcc 鯤
ndd 鯧
nde 觛
nde9 鯷
ndf 鯉
ndf. 鰣
ndm; 鰅
ne,0 鱁
ne2o 鯃
ne7d 鰽
ned. 鱄
nedx 鯁
nei 鮇
nemc 鱺
nen 魱
neoj 魺
newc 魨
nex. 觩
nex. 鯄
nf38 鰿
nf8f 鯥
nf8x 鯪
nf; 魼
nfcd 鰭
nfd 觰
nff 鮭
nff 觟
nffq 鱙
nfj 鯖
nfo 鮚
nh 鳭
nh8. 甔
nh81 詹
nh9e 魾
nh;0 觚
nhb8 頠
nhbc 危
nhm1 鮞
nho5 鰔
nhr1 鰤
nhu; 鱱
nhua 鱖
nhx 魬
niaa 鯠
nio 觫
nj2w 鰩
nji7 鱕
nk7e 鱣
nk8p 鱭
nk8x 鮫
nkds 鱆
nkl 鯜
nkog 鯙
nkoo 鰝
nkot 鯨
nkq 魧
nkx 魰
nl1 觓
nl7 鯔
nm8o 鰫
nma 魶
nmav 奐
nmax 敻
nmc 鮀
nmdi 彙
nme8 夤
nmeo 鮦
nmeq 鯇
nmer 龜
nmer 鱦
nmfo 鯛
nmr 鱈
nn 多
nn 魛
nn0 迻
nn2 解
nn20 邂
nn2; 蟹
nn4, 鱌
nnbb 翗
nnbo 夠
nnhc 觤
nnhc 鮠
nnlc 郺
nnmx 觼
nnoq 鮸
no 召
no 名
no, 劭
no0 迢
noc. 毚
nocn 劖
nocp 酁
nooi 鱢
noos 觶
nop 邵
nop 卲
noq 免
noq, 勉
noq. 兔
noq0 逸
noqh 鵵
np 鯓
np8 鯕
npc 鮿
npx 鯫
nqy 鮡
nr 魟
nr.7 鰡
nr/o 鮶
nr16 鱐
nr1c 色
nr1f 鰹
nra 觖
nrdl 鰋
nree 觝
nrf0 鰎
nrm1 鯞
nrr. 鱘
nrx 觨
nrxy 煞
nrz 急
ns, 鰳
nsc. 魷
nse 魽
nsj 鮪
nsli 鰈
nso 鮕
nsoj 鰗
nsoy 觾
nt/ 魦
nteq 觥
ntj 鮹
ntmd 鱨
nu45 鱴
nuf 鮮
nuf2 觲
nufo 鱔
nuof 鱹
nurt 鰜
nv 穌
nvao 鱎
nvbg 鯚
nvc 魠
nvd 鰆
nvej 觭
nvy 鰍
nwao 觿
nwe8 鱮
nwna 欿
nwq 觬
nwq 鯢
nwyy 燄
nx 彔
nx0 逯
nxao 鯦
nxmu 盝
nxn 剝
nxo 觡
nxo 鮥
nz/ 鮅
o 口
o, 另
o, 叻
o,,j 嗋
o,n 別
o,o 咖
o./y 噅
o.2v 嗾
o.;p 啷
o.a. 唹
o/3 咱
o/3. 嗅
o/90 唌
o/bx 吸
o/d1 啤
o/d5 喤
o/ds 嗥
o/dx 噭
o/mv 噢
o/o 呂
o/op 郘
o/ra 噬
o1 中
o1 唁
o1e0 遣
o1e0 遺
o1e8 貴
o1mu 盅
o1rx 嗖
o1z 忠
o2 吽
o22 哖
o23 咋
o2c 吃
o2ep 啣
o2i 咮
o2o 哠
o2o2 喡
o2q8 囋
o2s 吘
o2sy 嘸
o33 啡
o38 員
o38 唄
o38, 勛
o38l 嚶
o38p 鄖
o4 哂
o4 嘿
o4 呬
o438 嘪
o4b; 噣
o4ds 嘾
o4ek 噮
o4et 嘌
o4f 嚜
o4l 喓
o4tf 囉
o4x0 嚃
o5 呈
o50 逞
o5p 郢
o5u 哢
o6 哺
o60 嗹
o6e. 囀
o6nw 嚪
o7. 呣
o7e5 嘓
o7ek 喂
o7j 喟
o7uk 噥
o7v 咽
o7vz 嗯
o8 只
o8 叭
o8 咪
o8 唒
o84d 噌
o88. 噂
o8;b 嗡
o8n 吩
o8xn 嗲
o9 哫
o9c 呲
o9ci 喍
o9cn 嘴
o9ds 啅
o9h5 噦
o9j 啃
o9o 呫
o9t8 嚬
o9w 嚙
o;2 哞
o;2v 唉
o;8x 唆
o;af 雖
o;o 咍
o;p 唧
o;q 吮
oa11 吤
oa4d 噲
oaaf 唑
oab 吟
oab. 呤
oab8 嗿
oabo 唅
oabz 唸
oac 吪
oae; 噙
oag. 咐
oai 咻
oai 唋
oajn 喻
oakf 唯
oakh 鷕
oao 哈
oaoa 噞
oaou 啽
oar/ 咿
oaro 嗆
oarv 喉
oas 呏
oaso 啥
ob 嗎
ob,o 噊
ob. 呁
ob/ 吻
ob2w 啕
obb/ 嘐
obbf 嚁
obbs 噿
obd 咰
obe 叼
obf 吜
obfp 哪
obg 吇
obgc 吼
obo 呴
oboy 喣
obrc 咆
obxw 哅
oc 叱
oc7u 嚧
occd 喈
oce, 噱
ocii 嘛
ocov 噳
ocq 唬
ocs8 嚝
ocsx 喥
ocu 噓
od1 呻
od1 呷
odbl 喝
odd 唱
ode 呾
odes 哻
odf 哩
odm; 喁
odmu 嗢
odol 嘍
odpx 嘬
odsx 嚗
oduf 嗶
oe,. 啄
oe1e 啞
oe1z 噁
oe2o 唔
oe7d 嘈
oe9y 嘕
oe;f 咥
oebh 鴞
oebk 饕
oebq 號
oede 咺
oedx 哽
oeg 吁
oei 味
oej 叮
oema 啢
oenn 咧
oeo1 嗝
oeoj 呵
oer9 啑
oeu1 呯
oew8 噸
oewc 吨
of 吐
of.8 嚽
of.x 嗷
of38 嘖
of9 唗
of; 呿
of;u 嗑
ofbg 哮
ofc 咾
ofcd 嗜
ofdp 嘟
off 哇
offq 嘵
offr 嘒
ofmu 噎
ofnv 喫
ofnw 囓
ofo 咭
ofoo 嘻
oft 嗉
oft 哧
oftt 嚇
ofu0 噠
ofus 啈
og. 吋
og/d 啪
ogej 咑
ogh1 哳
ogk 啦
oh 鳴
oh0 呎
oh1; 囑
oh35 嘎
oh3x 嗄
oh7p 喕
oh9 吥
oh95 嘁
oh9e 呸
oh;0 呱
ohbc 呃
ohc 呢
ohdf 喱
ohe. 嗕
ohe1 听
ohef 喔
ohey 嗚
ohff 啀
ohjx 嚘
ohmn 唰
oho5 喊
ohri 嘄
ohua 噘
ohv. 唳
ohv9 嚦
ohww 啒
oi 呆
oiaa 唻
oide 喳
oie 呠
oiit 噤
oioa 嗽
oion 喇
oj 肙
oj2w 嗂
oj4. 嚼
oje 咀
ojg 呼
ojg. 哷
ojh 鵑
oji 啋
ojl 哸
ojmx 噯
ok7p 鄙
ok8. 嗙
ok81 啼
ok8o 嘀
ok8p 嚌
ok8x 咬
ok;j 唷
okaf 囃
okas 啐
okd 喑
okda 噷
okdz 噫
okiq 嚫
okjx 嚨
okl 唼
okla 咳
oko, 嚎
oko. 噈
okog 啍
okoj 哼
okok 嚷
okoo 嗃
okoq 喨
okq 吭
oktf 噰
okx 呅
okx/ 喭
ol1 叫
oloc 嗈
olv 吳
olx 呶
om0t 嚓
om1 吊
om8d 噾
oma 吶
ome9 啶
omee 喧
omeo 哃
omf 呥
omfo 啁
omfo 嗐
omh1 嚅
omo 喎
omuo 嗣
omxo 喀
omzj 嚀
omzw 嘧
on 叨
on 唃
on, 喙
on.s 嘝
ona 吹
ond 嚕
onmv 喚
onn 哆
ono. 嚵
onw 啗
oo.1 喌
oo1 串
oo11 丳
oo18 嘳
oo1z 患
oo7. 獸
oo7r 鼉
oob 罵
ood5 戰
oodh 鷤
oodk 辴
oodk 囅
oodp 鄲
oods 單
ooe8 顎
ooeb 呺
ooeb 咢
ooeh 鶚
ooep 鄂
ooho 囂
oohx 嚴
ooms 斝
ooo 品
oooc 氉
oooi 噪
oooi 喿
ooon 劋
ooop 鄵
oooq 碞
ooos 嘽
ooow 喦
ooow 嵒
ooq 咒
oorc 唈
ooro 嚚
oou0 遻
oov. 哭
oovo 器
op 叩
op 咡
op 咠
op8 唭
op81 嘶
ope5 戢
opej 啊
oppp 囁
opql 啿
oq 兄
oqy 咷
or16 嘯
or1c 吧
or1c 邑
or1u 嚍
or1u 嚂
or38 嗊
ora 吷
orb1 咈
orc8 噀
oree 呧
orj/ 呀
orj5 呡
ork 哏
orm1 咂
ormx 唚
oroo 嘔
orpx 噉
orqd 噆
orr. 噚
os38 嗔
os8 哄
os; 吰
osan 嘜
osc/ 哤
osd 唶
osdj 嘲
ose 咁
osli 喋
osm1 咘
osm9 嚏
osmg 哱
osms 喃
osmt 嗦
oso 咕
osok 囔
osov 嘆
osoy 嚥
osu8 噴
osx 吱
ot 吆
ot, 呦
ot/ 吵
ot38 嗩
otb. 喲
otj 哨
otm7 噹
otmd 嚐
ott5 嘰
ou7 喵
ou8u 嗌
ouai 嗏
ouao 嗒
ouax 嚄
oud; 囆
oudl 噶
oudv 嗼
ouef 嘩
ouf 咩
ouf; 囈
oufq 唴
oufr 嗟
ouko 嚆
ounv 囌
ouof 嚾
ourt 嗛
ouso 喏
ouuv 噗
ouwj 嗍
oux 哎
ov. 吠
ov. 呔
ov1o 咶
ov1w 喢
ov8t 嘹
ovaa 唊
ovao 嘺
ovc 吒
ovdc 唵
ovf 唪
ovg5 哦
ovn 唎
ovrb 咦
ovuf 唾
ovy 啾
ow; 嗤
owaf 嗺
owh1 喘
owq 唲
oww 咄
ox 史
ox. 咚
ox2. 嗨
oxad 喒
oxcx 啵
oxo 咯
oxs1 唏
oxt/ 唦
oxxi 嗓
oxxx 啜
oy 吙
oyy 啖
oyy, 嘮
oz/ 咇
p 耳
p 身
p, 阞
p. 防
p/ 髟
p/. 髣
p//8 鬚
p/4k 鬟
p/7k 鬞
p/7u 髷
p/7z 聰
p/8d 鬙
p/9c 髭
p/; 隗
p/;/ 鬖
p/ad 鬠
p/af 髽
p/ai 髹
p/b, 髳
p/bc 髱
p/cx 髲
p/d/ 鬄
p/d1 陴
p/d5 隍
p/dx 鬘
p/eq 髡
p/fd 鬐
p/fo 髻
p/he 髬
p/i; 鬆
p/in 鬎
p/jj 鬅
p/kk 鬤
p/l5 鬣
p/m8 鬢
p/mf 髯
p/mq 髧
p/mt 鬃
p/mv 隩
p/no 髫
p/p 髶
p/s. 髮
p/s8 鬒
p/sa 鬗
p/sj 鬍
p/sj 鬌
p/tj 髾
p/uc 鬈
p/un 鬋
p/ut 鬑
p/uv 鬕
p/vc 髦
p/vd 鬊
p/vn 鬁
p/vo 髺
p/xf 髼
p0et 際
p0ev 聧
p23 阼
p2c 阣
p2i 陎
p2v 镻
p33 陫
p5sz 聽
p6 陣
p7ek 隈
p7sb 聘
p8 其
p81 諅
p8;b 聬
p8f 基
p8h1 斯
p8h; 蟴
p8j 期
p8na 欺
p8q 碁
p8t 綦
p8z 惎
p9 阯
p9o 阽
p9t 陟
p9tb 騭
p;q 阭
pab 耹
pab. 聆
pab; 陰
pag. 附
pai 除
pajn 隃
pamu 陯
paoa 險
pasf 陞
pb1c 阤
pb2w 陶
pb8 陱
pc 耴
pcc 阰
pccd 階
pccf 陛
pcs 阠
pcx 陂
pde/ 陽
pde9 隄
pdm; 隅
pdty 隰
pe8x 隬
pe9 聇
peeq 阮
pej 耵
pelr 陘
pema 陃
peml 陋
peo1 隔
peo5 聝
peoj 阿
peq 阢
peu 阱
pf.x 隞
pf8f 陸
pf8x 陵
pf9 陡
pf; 阹
pfd 陼
pffq 隢
pfo 聐
pg. 射
phbc 阨
phd 陌
phl5 隇
phm1 陑
phm1 聏
phmv 陾
phww 镼
phx 阪
pid 陳
pid; 螴
pidx 敶
pje 阻
pjri 檃
pjrz 隱
pk 長
pk8p 隮
pkd5 職
pkds 障
pkjx 隴
pkla 陔
pko 陪
pkoj 聤
pkq 阬
pkxe 隡
pm58 軉
pm8r 聜
pmc 陀
pmeq 院
pmf 聃
pmq 耽
pmzj 聹
pn 刵
pn,f 墬
pnh1 聸
pnn 陊
pnw 陷
po18 隤
po18 聵
po38 隕
po5 聖
pp 耶
pp 聑
ppp 聶
ppp8 顳
ppx 陬
pqi 躲
pql 甚
pql, 勘
pql5 戡
pqla 歁
pqls 斟
pr1p 聊
prb 躬
prdl 躽
pree 阺
prf 肆
prk 限
proo 軀
ps; 耾
psj 陏
psr0 隨
psrf 墮
psrj 隋
psrr 隓
psrw 嶞
psrz 隳
pt, 聈
pt/t 隙
ptmo 躺
ptt6 聯
pu, 隊
pu,0 隧
pu,0 鐆
pu,f 墜
pu8u 隘
purt 隒
pv1 阡
pv1o 聒
pv8t 镽
pva 镺
pvaa 陝
pvaa 陜
pvc 毦
pvdf 隀
pveb 陓
pvej 陭
pvuf 陲
pweu 隑
pwf 隉
px 取
px, 聚
px,p 鄹
px2 降
pxee 隆
pxi 棸
pxl 娶
pxp 郰
pxy 焣
py 耿
pykk 褧
pyy, 軂
pz 恥
q 儿
q 几
q 石
q. 凡
q.rk 硠
q.x 砯
q/; 磈
q/bx 砐
q/d1 碑
q/d5 凰
q/di 礫
q/dx 礉
q/k 砭
q0eu 磴
q23 砟
q2c 矻
q2i 硃
q2o 硞
q2q8 礸
q3q 硯
q4 硒
q4ds 磹
q5. 砡
q6 硨
q7. 砪
q777 礧
q7ek 碨
q7i 硱
q7t 磥
q7v 碅
q84d 磳
q8n 砏
q8n2 磷
q97y 磠
q9o 砧
q; 風
q;/ 颩
q;1x 颼
q;38 颶
q;5/ 飋
q;7z 颸
q;9o 颭
q;;/ 磣
q;;o 颱
q;b/ 飂
q;bc 颮
q;d/ 颺
q;en 颲
q;r/ 颬
q;r7 飀
q;vo 颳
q;vt 飉
q;x; 颾
qa11 砎
qab. 砱
qai 硢
qakf 碓
qaki 磼
qaky 礁
qamu 碖
qb 碼
qbb/ 磟
qbg 矷
qbrc 砲
qc29 礙
qcc 砒
qcn 砌
qcro 磄
qcs8 礦
qcx 破
qd1 砷
qdbl 碣
qde/ 碭
qe7e 礓
qed. 磚
qedx 硬
qee5 碔
qeh 鳳
qelr 硜
qen 夙
qeoj 砢
qeq 矹
qes 矸
qeu 研
qeu1 砰
qeug 揅
qeun 硎
qf.x 磝
qf38 磧
qf7. 碡
qf; 砝
qf;u 磕
qff 硅
qffq 磽
qfj 碃
qfo 硈
qfoo 礂
qh38 碩
qh95 磩
qhc 砨
qhcq 磃
qhe1 斫
qhe8 礩
qhej 磭
qhex 碬
qhmu 碥
qhos 礔
qhsk 碾
qhu; 礪
qhxx 碫
qi 朵
qi8; 硹
qide 碴
qie 砵
qii 碄
qii9 礎
qin 剁
qje 砠
qji7 磻
qjj 硼
qjrj 碀
qjtv 磎
qk 砬
qk8. 磅
qk81 碲
qk;q 硫
qkas 碎
qkf 砫
qko 碚
qkoj 碠
qkoo 碻
qkox 礅
qm7 礌
qm8r 硿
qmaf 確
qmaf 礭
qmc 砣
qme8 礗
qme9 碇
qmeo 硐
qmfo 碉
qmfo 磍
qmh1 礝
qmi3 礵
qmk 砃
qmnc 碗
qmo 碢
qms8 碘
qmuw 碙
qn 矽
qn 确
qn2i 磔
qna 砍
qnhc 硊
qnx 碌
qoos 磾
qoox 礹
qpq, 磡
qpql 碪
qq 砳
qqq 磊
qqx 砓
qr 矼
qr18 礥
qr1u 礛
qrb1 砩
qree 砥
qrf 硉
qrj/ 砑
qrk 硍
qrm1 砸
qs38 磌
qsc/ 硥
qsd 碏
qse8 磺
qsli 碟
qt/ 砂
qtj 硝
qtm7 礑
qtt5 磯
qu0t 礤
qu45 礣
qu6. 礡
quf5 礒
qufr 磋
qukl 硭
qum, 礞
qumv 碤
qurt 磏
qutt 磁
quu 碰
qv 砆
qvaa 硤
qvao 礄
qvc 矺
qvej 碕
qvg5 硪
qvuf 硾
qvxx 磢
qw 凸
qwaf 磪
qweu 磑
qwhy 碳
qwjj 磞
qx 泵
qx 殳
qx 砅
qxo 硌
qxri 磲
qxxi 磉
qy 兆
qy0 逃
qy3q 覜
qyh8 頫
qyyt 礯
r 工
r 匚
r, 功
r.1. 卵
r.18 贕
r.1g 孵
r.1x 毈
r.n0 遛
r.n7 留
r.n8 貿
r.nh 鶹
r.nn 劉
r.np 鄮
r/ 尹
r/3u 匴
r/o 君
r/o8 頵
r/of 群
r/oh 鵘
r/op 郡
r/oy 焄
r16 肅
r16h 鷫
r1;p 卿
r1c 巴
r1cw 岊
r1e8 賮
r1eu 盡
r1o8 頤
r1o8 賾
r1ol 媐
r1oy 熙
r1p 卯
r1r0 鑒
r1r0 鋻
r1r1 臣
r1r1 臦
r1r1 臩
r1r5 朢
r1r8 賢
r1r; 蜸
r1ra 臥
r1rf 堅
r1rg 掔
r1rj 腎
r1rl 婜
r1ro 臨
r1rq 覽
r1rq 硻
r1rt 緊
r1ru 監
r1ru 豎
r1ru 鹽
r1ru 盬
r1ry 燛
r2v. 瑿
r2v4 黳
r2v8 醫
r2v8 贀
r2va 毉
r2vb 翳
r2vh 鷖
r2vl 嫛
r2vt 繄
r33 匪
r33f 匷
r38 貢
r5 匡
r5 戉
r5, 劻
r6/c 匭
ra 夬
raa 巫
raaq 覡
rao 匼
rb 弓
rb.. 弱
rb.h 鶸
rb/ 匢
rb1 弔
rb1 引
rb11 弗
rb18 費
rb1c 匜
rb1c 艴
rb1n 刜
rb1p 鄪
rb1u 盄
rb3x 彏
rb4k 彋
rb81 鬻
rb8b 粥
rb; 弘
rb;1 謽
rb;; 強
rba. 弣
rbbc 弛
rbdf 彃
rbe. 瓕
rbee 彊
rber 弳
rbex 彌
rbfe 疆
rbh0 弧
rbhb 弼
rbjj 弸
rbkt 弦
rbkt 弶
rbno 弨
rbos 彈
rbp 弭
rbpk 張
rbrc 弝
rbre 弤
rbro 彄
rbs8 彉
rbtj 弰
rbwx 弢
rc 己
rc 已
rc 巳
rcl 妀
rcp 邔
rcr0 選
rcr8 巽
rcu 异
rcx 改
rcz 忌
rczh 鵋
rd1 匣
rdl 匽
rdlh 鶠
rdlp 郾
re5 氏
re5d 昏
re5e 氐
re5h 鴟
re5p 邸
reb 巧
rej. 瓨
rf 聿
rf0 建
rf7e 畫
rf7n 劃
rfd 書
rfde 晝
rfhu 衋
rh38 項
rhe1 匠
rhmu 匾
rj/ 牙
rj/0 迓
rj/f 雅
rj/h 鴉
rj/p 邪
rje3 睯
rje5 民
rjed 暋
rjen 刡
rjex 敃
rjex 敯
rjez 愍
rk 艮
rk0 退
rkq 匟
rm1 匝
rmm1 帚
ro 叵
ro18 匱
roo. 甌
rooa 歐
rooh 鷗
rooo 區
roos 匰
roox 毆
roox 敺
rp 印
rp 卬
rp 邛
rp0 迎
rpx 敢
rpxz 憨
rq 匹
rq 旡
rq.0 銎
rq.9 跫
rq.; 蛩
rq.s 鞏
rq.z 恐
rqh 鴄
rqr1 鬵
rqr; 蠶
rqrd 朁
rr 巨
rr;; 螶
rro. 尋
rrop 鄩
ruso 匿
rusz 慝
rx 攻
rx 汞
rx0 逮
rxaf 匯
s 十
s 廿
s 革
s, 勒
s,,, 協
s/bx 靸
s/d1 鞞
s0 丈
s1bg 存
s1f 在
s3 直
s38 真
s388 顛
s38h 鷏
s3s3 矗
s4v0 韆
s6g. 博
s8 共
s86 輂
s8; 蛬
s8g 拲
s8rc 巷
s8z 恭
s;af 雄
s;bb 翃
saa2 麰
saa8 麴
saa; 麮
saac 麧
saac 麭
saan 麥
saap 麵
saau 麷
saav 麩
sab 靲
sac 靴
saex 鞭
sao 鞈
saou 鞥
sb,i 鞣
sb. 靮
sb1 鞫
sb8 鞠
sbrc 鞄
sc 尢
sc. 尤
sc./ 尨
sc5 尪
sca1 尬
scb. 尥
scmj 尳
scru 尷
scvf 尰
scx 鞁
sd 昔
sdbl 鞨
sdcx 皵
sde 靼
sde9 鞮
sdh 鵲
sdh1 斮
sdn 剒
sdol 鞻
sds2 韓
sds5 戟
sds; 螒
sdsb 翰
sdsc 乾
sdsf 雗
sdsh 鶾
sdsi 榦
sdsj 朝
sdss 幹
sdss 斡
sdty 韅
se 甘
se78 黃
se7e 韁
se7f 黈
se7h 鷬
se7s 斢
sei 某
sei 靺
sej 靪
sep 邯
seq 靰
ses 靬
sf48 韇
sff 鞋
sfo 鞊
sfu0 韃
sgk 鞡
she1 靳
sj 有
sj0 迶
sje 靻
sje5 戫
sjp 郁
sjx 散
skop 鞹
sl 世
sl0 迣
sl3, 勩
sl38 貰
sm 冇
sm 巿
sm1 布
sm18 顢
sm79 疐
sm8r 鞚
smb, 勃
smbg 孛
smbh 鵓
smbp 郣
smf 冓
smf0 遘
smf8 顜
smfq 覯
smfs 斠
sml 鞍
smm0 遰
smm1 帶
smt 索
smus 南
smv 鞅
sno 鞀
snoq 鞔
so 古
so 右
so9h 鷰
so9p 酀
so9y 燕
sof 堇
sof, 勤
sof1 斳
sofp 鄞
sofq 覲
sofz 懃
soh 鴣
sohx 嘏
soj 胡
soj 鞙
sojh 鶘
sojh 鶦
soks 辜
som; 蠹
som; 蟗
somi 橐
somi 櫜
somk 囊
soq 克
soq, 勀
soqn 剋
soqq 兢
sova 歎
sova 臡
sovf 難
sovk 艱
sovz 戁
sox 故
sr 左
sr1c 靶
srb1 靷
srb; 鞃
srf0 鞬
srjc 毻
srk 鞎
ssl 靾
ssli 鞢
st, 靿
stj 鞘
stmo 鞝
stt5 鞿
su 卉
su38 賁
su3a 歕
su3b 翸
suao 鞳
suax 韄
sucy 韉
sufx 鼖
sus 靽
sv 鞂
svy 鞦
sx 友
sx 支
sx. 犮
sxbb 翅
sxd 鞜
sxh 鳷
sxh8 頍
sxr. 攳
sy 灰
symu 盔
t 小
t 糸
t, 幼
t,h 鴢
t. 紡
t.11 絒
t.29 縼
t.;; 蠁
t.;d 響
t.;k 饗
t.;o 嚮
t.;p 鄉
t.h 鶭
t.na 絘
t.s 紏
t/ 少
t/, 劣
t/3 省
t/7x 繌
t/7z 總
t/90 綖
t/bx 級
t/c. 紈
t/d1 綿
t/d1 綼
t/dx 線
t/dx 繳
t0 尐
t0et 縩
t111 紃
t2bc 絁
t2c 紇
t2dx 緮
t2i 絑
t2o2 緯
t2q8 纘
t2v 紩
t33 緋
t38l 纓
t3qy 繉
t3t/ 緲
t4ek 繯
t4et 縹
t4f 纆
t4fs 繹
t4tf 纙
t5 紝
t50 綎
t60 縺
t6g. 縛
t7 細
t7i 綑
t7s 紬
t7t 縲
t7uk 繷
t7v 絪
t7z 緦
t84d 繒
t88 緧
t88. 繜
t8n 紛
t8n2 繗
t8o 綌
t8r/ 綈
t9ds 綽
t;;/ 縿
t;o 紿
ta/ 紾
ta11 紒
ta4d 繪
taa5 纖
tab 紟
taex 緶
taf 絟
tag. 紨
tajn 緰
tak; 蠽
takf 雀
takf 維
tamu 綸
tao 給
tarv 緱
tauv 纀
tb 幻
tb,o 繘
tb. 約
tb2w 綯
tb9j 縃
tbb/ 繆
tbd 絢
tbf 紐
tbj 紓
tbo 絇
tbww 縐
tc4f 纏
tc7u 纑
tcc 紕
tcs8 纊
td1 紳
td4x 縵
td5 絏
tdb/ 緆
tdcc 緄
tde9 緹
tdmu 縕
tdol 縷
tdpx 繓
tdtq 虩
tduf 縪
te99 縰
te;d 縉
te;f 絰
te;x 緻
tea9 縱
ted. 縳
tedx 綆
tedz 繐
tee; 紜
teej 絎
teg 紆
tej 糽
telr 經
tema 緉
temc 纚
teo5 緎
teq 光
teq6 輝
teqf 耀
ter9 緁
terl 緀
tes5 絨
tewc 純
tex. 絿
tf38 績
tf48 續
tf8x 綾
tf; 紶
tfd 緒
tff 絓
tffq 繞
tfj 綪
tfo 結
tfo8 纈
tfp 綁
tft 縤
tfu0 繨
tfuu 盭
tg. 紂
th/t 縓
th7p 緬
th9 紑
thd 絔
the. 縟
thjx 纋
thmu 編
thmv 緛
tho5 緘
thr0 縋
thv. 綟
thxx 緞
ti3 緗
ti4 練
tie 絊
tii 綝
tio 綀
tj 肖
tj, 絼
tj0 逍
tjbg 綒
tje 組
tjex 緩
tji 綵
tji7 繙
tjl 綏
tjmx 綬
tjn 削
tjng 揱
tjq 胤
tjw 縚
tk7e 繵
tk8. 縍
tk81 締
tk8x 絞
tk;q 統
tkas 綷
tkd5 織
tkdk 縗
tkdz 繶
tkf 紸
tkla 絯
tkog 綧
tkok 纕
tkoo 縞
tkot 綡
tkt 絃
tkt2 縴
tkts 繂
tkx 紋
tkx; 縭
tl1 糾
tl7 緇
tldi 繅
tm6 緷
tma 納
tmad 縮
tmc 紽
tme8 縯
tme8 繽
tme9 綻
tmej 紵
tmeo 絧
tmeq 綄
tmer 繩
tmet 綜
tmfo 綢
tmfo 縖
tmh1 繻
tmhr 綰
tmj 縎
tmks 縡
tmnc 綩
tmo 尚
tmo 絅
tmo 緺
tmo. 瓽
tmo/ 牚
tmo1 常
tmo4 黨
tmo7 當
tmo8 賞
tmoc 氅
tmod 嘗
tmof 堂
tmog 掌
tmoi 棠
tmok 裳
tmoq 党
tmox 敞
tmq 紞
tmul 網
tmuw 綱
tmv 紻
tn, 緣
tn. 紉
tnn2 繲
tno 紹
tno. 纔
tnoq 絻
tnrc 絕
tnry 繺
tnx 綠
to/o 絽
to10 繾
to18 繢
to38 縜
toj 絹
tooi 繰
toos 繟
top 緝
tpx 緅
tqy 絩
tr 紅
tr16 繡
tr1q 纜
trb1 紼
trb1 紖
trb; 繈
trc 紀
tre5 紙
trfe 繣
trjd 緡
trmx 綅
ts 緙
ts38 縝
ts; 紘
tsc. 紌
tse 紺
tsjx 繖
tsl 紲
tsmg 綍
tsx. 紱
tt 絲
tt/ 紗
tte1 斷
tte5 幾
tte5 畿
tte; 蠿
tteq 絖
tth 鷥
ttj 綃
tttl 繼
ttw 幽
tu,0 繸
tu8u 縊
tudv 縸
tufo 繕
tufr 縒
turt 縑
tus 絆
tuu 絣
tuvc 綣
tuw0 縌
tv 尖
tv. 紎
tv4 纁
tv8t 繚
tvao 繑
tvdf 緟
tvej 綺
tvl 緌
tvuf 綞
tvv 縥
twaf 繀
twao 纗
twjj 繃
tww 絀
tx. 終
tx. 紁
tx2 絳
txao 綹
txf0 縫
txo 絡
txs1 絺
txxi 縔
txxx 綴
tyy 緂
tze. 緪
u 廾
u 艸
u,,, 荔
u,0 遂
u,88 蕕
u,bf 莥
u,o 茄
u,y 荻
u. 芳
u./y 蒍
u.29 蔙
u.2c 葹
u.2v 蔟
u.a. 菸
u.n8 薋
u.na 茨
u.qx 蒰
u.rk 莨
u/3 首
u/3. 導
u/30 道
u/35 馘
u/3x 夔
u/3z 蒠
u/7c 蓖
u/7z 蔥
u/90 莚
u/; 蒐
u/bb 艿
u/bx 芨
u/c 艽
u/c. 芄
u/d. 菂
u/d1 萆
u/d5 葟
u/dc 葩
u/di 藥
u/dx 薂
u/dy 蘤
u/mv 薁
u0 菳
u0et 蔡
u0ev 葵
u1 卅
u11 卌
u1dl 藹
u1fd 藷
u1qx 蔎
u23 苲
u27. 莓
u27t 蘩
u2i 茱
u2o2 葦
u2sa 蒛
u2sy 蕪
u2v 苵
u2vf 薙
u3 苜
u30v 藈
u33 菲
u36 薡
u38 蔝
u38l 蘡
u38n 萴
u3q 莧
u3u 萛
u4 茜
u41n 藅
u4;c 藣
u4ds 蕈
u4et 蔈
u4f. 薽
u4fd 薯
u4h5 蔑
u4hn 蘮
u4l 葽
u4m. 甍
u4m3 瞢
u4m5 薎
u4mc 薨
u4mn 夢
u4mp 鄸
u4mx 蘉
u4tf 蘿
u50 莛
u5oo 蘁
u6 莆
u6 莗
u6 菛
u60 蓮
u6af 藺
u6es 蓒
u6f 莊
u6fo 蘠
u6g. 蒪
u6h1 蔪
u6i4 蘭
u6j 蕑
u6j. 蔣
u6wt 蘻
u6z 蕄
u7 苗
u7, 莮
u7. 苺
u777 藟
u77f 蘲
u77t 虆
u78m 莔
u7a 苬
u7e5 蔮
u7ek 葨
u7o 茴
u7s 苖
u7t 蔂
u7uc 蔨
u7v 茵
u7v 菌
u7z 葸
u8 莤
u8 黹
u86 黼
u88n 黺
u8;b 蓊
u8ay 蘸
u8m; 蠲
u8mh 鷁
u8mu 益
u8n 芬
u8ni 棻
u8nu 葐
u8oq 莌
u8s. 黻
u8v8 蘱
u8x 蔽
u8xr 虌
u9 芷
u97y 蓾
u9c 茈
u9ex 菽
u9h5 薉
u9o 苫
u9t8 蘋
u9xo 蕗
u; 茧
u;8x 荾
u;;/ 蔘
u;a 苡
u;m 萭
u;o 苔
u;rq 蔇
ua11 芥
ua1i 蓧
ua1j 蓨
ua23 莋
ua4d 薈
ua5 荏
ua;x 葰
uaa0 遳
uaae 葅
uaaf 莝
uaax 蓌
uab 芩
uab. 苓
uac 花
uae5 茷
uaee 芢
uaej 荷
uaf 荃
uaf 茌
uafj 蒨
uag. 苻
uai 茶
uai 荼
uai 茠
uaih 鷋
uakf 萑
uakf 雈
uako 蓓
uakw 舊
uaky 蕉
uamu 菕
uao 荅
uaoa 薟
uaoi 葆
uaou 蘥
uaox 蘞
uaro 蒼
uarv 葔
uarx 葠
uat 苶
uav. 茯
ub, 茅
ub,, 蓩
ub,h 鶜
ub,i 葇
ub. 芍
ub.0 蓪
ub/ 芴
ub2w 萄
ub6 葡
ub8 菊
ub9q 蔬
ub; 芶
ubb/ 蓼
ubbf 藋
ubd 荀
ube7 蔔
ubg 芓
ubg0 菰
ubgc 芤
ubgt 蓀
ubj 芧
ubj8 蕷
ubo 苟
ubo 茍
ubo1 警
ubo; 蟼
ubob 驚
ubog 擎
uboi 檠
ubox 敬
uboz 憼
ubrc 苞
ubxw 菡
ubxy 蒸
uc 芞
uc 蔍
uc.; 藱
uc.v 蒺
uc29 薿
uc7u 蘆
uc7z 藘
uc8 蘪
ucc 芘
uce0 蘧
uci3 蘼
ucig 藦
uciq 蘑
ucje 蔖
ucra 菮
ucre 菧
ucro 蓎
ucry 薦
ucs1 蓆
ucsy 蔗
ucut 薕
ucy 藨
ud23 葃
ud3 萺
ud4x 蔓
udbl 葛
udcc 菎
udd 菖
udf. 蒔
udj 萌
udm, 勱
udm0 邁
udm9 躉
udm; 萬
udm; 蠆
udol 蔞
udox 藪
udpx 蕞
uds 草
uduf 蓽
udv 莫
udv, 募
udv1 幕
udvb 驀
udvd 暮
udvf 墓
udvg 摹
udvp 鄚
udvz 慕
ue,0 蓫
ue,d 藸
ue,e 蕤
ue2p 蓹
ue2t 蘌
ue2x 蕧
ue5 芅
ue7e 薑
ue8x 薾
ue9 芐
ue99 蓰
ue9y 蔫
ue;f 荎
ue;n 菿
uea9 蓯
ued. 蓴
uede 荁
uedz 蕙
uee; 芸
ueej 荇
ueeq 芫
ueet 蒜
uef9 蓗
ueg 芋
uei 茉
uej 艼
uel, 葝
uelr 莖
uema 苪
uen/ 薚
uene 薤
uenj 蘅
uenk 薞
uenn 茢
uenu 葬
ueo1 蒚
ueo7 葍
ueof 菫
ueoj 苛
ueou 荳
uer9 萐
uerf 葎
uerl 萋
ues5 茙
ueu1 苹
ueuf 華
ueuh 鷨
ueun 荊
uewc 芚
uewx 薇
uewx 藢
uex. 莍
uf 羊
uf 羋
uf 芏
uf.a 羡
uf.x 羕
uf.x 蔜
uf/f 羞
uf/r 差
uf/w 齹
uf48 藚
uf8. 蓺
uf8; 藝
uf8k 養
uf8n 羒
uf8n 鯗
uf8x 菱
uf8y 爇
uf;u 蓋
uf;x 羧
ufa. 羚
ufaa 羷
ufan 羭
ufao 薔
ufb. 荺
ufb. 薵
ufbb 翔
ufc 荖
ufcd 蓍
ufcx 羻
ufd 著
ufdl 羯
ufe5 羢
ufer 羥
uff 茥
uff. 葑
uffq 蕘
uffr 蔧
uffu 蘴
ufh5 羬
ufht 羱
ufj 菁
ufj7 羳
ufke 羶
ufkk 蘾
ufl 姜
ufm1 羺
ufmj 羜
ufmq 羦
ufn0 羑
ufna 莰
ufo0 薳
ufof 薹
ufox 薣
ufq 羌
ufqx 羖
ufr1 羛
ufre 羝
ufs8 羵
ufu. 蓻
ufu0 薘
ufuo 善
ufup 鄯
ufux 蕔
ufv 美
ufv5 義
ufv5 羲
ufvb 羠
ufwy 羰
ufxa 羨
ufy 羔
ufyv 羹
ufz 恙
ug4s 蘀
ug6 蒱
ug7s 菗
ugaf 蓷
ugbc 菢
ugk 菈
ugs. 菝
ugt/ 莏
ugv 荴
ugvo 葀
uh 蔦
uh/t 蒝
uh35 蕆
uh3o 虈
uh5 茂
uh9 芣
uh95 蒧
uh9e 苤
uh;0 蓏
uhb5 荿
uhc 苨
uhe. 蔚
uhe. 蓐
uhe0 蕸
uhe1 芹
uheo 茩
uhex 葭
uhez 藯
uhj 菺
uhl5 藏
uhl5 葳
uhm1 荋
uhmu 萹
uho5 葴
uhoc 蔰
uhoi 蘗
uhos 薜
uhr1 蘬
uhr9 躠
uhrg 孽
uhrs 薛
uhrx 蒑
uhua 蕨
uhv9 藶
uhxx 葮
ui3 葙
ui4 萰
ui8; 菘
uiaa 萊
uid 菄
uie 苯
uih1 菥
uii 菻
uim 茦
uimm 蕀
uimn 莿
uimx 蓛
uio 莕
uio0 藗
uio8 藾
uioa 蔌
uiro 藲
uj,f 薶
uj,q 藐
uj23 葄
ujbc 薍
ujbg 莩
ujbo 葋
uje 苴
uje, 莇
ujex 萲
ujhx 菔
uji 菜
uji7 蕃
ujjn 蒯
ujl 荽
ujm2 蕣
ujmx 薆
ujn 前
ujnb 翦
ujnn 剪
ujny 煎
ujrc 萉
ujux 藤
uk 芝
uk 苙
uk,x 藙
uk8. 蒡
uk80 藡
uk81 蒂
uk8e 虀
uk8k 蔉
uk8o 蔏
uk8p 薺
uk8q 萒
uk8x 茭
uk;q 茺
ukas 萃
ukbo 蒟
ukd 萻
ukdf 蕫
ukdk 蓑
ukdz 薏
uki1 薪
ukiq 藽
ukjx 蘢
ukl 芒
ukl 菨
ukla 荄
uklq 荒
uklz 莣
ukm1 芾
uko 菩
ukoa 藃
ukoc 薧
ukoj 葶
ukok 蘘
ukoo 蒿
ukop 蔀
ukq 苀
uks 莘
ukt7 蓄
uktf 蕹
ukx 芠
ukxf 蘺
ul1x 荍
ul7 菑
ulh. 薅
ulll 葌
ulo 茹
ulog 蒘
ulot 蕠
ulso 菇
um1; 繭
um6 葷
um7 蕾
um8 蒏
um8. 葖
um8b 藭
um8o 蓉
uma 芮
uma. 蕶
umad 蓿
umaf 蒮
umaf 藿
umd8 蓂
ume, 蒙
ume8 蔩
ume9 萣
ume; 蕓
umec 氋
umee 萱
umeh 鸏
umej 苧
umeo 茼
umeq 莞
umex 蔻
umf 苒
umh1 薷
umhr 菅
umj 蓇
umje 萓
uml 荌
umn. 蒬
umnc 菀
umo 萵
umo0 薖
umoo 蘦
umq 菪
umsg 藆
umul 菵
umv 英
umvh 鶧
umzj 薴
umzw 蔤
un 芀
un.z 荵
un2 荈
un38 萯
una 芡
unak 蕵
unbc 苑
unbz 葾
unh1 薝
unmx 藑
unn 薊
unn 茤
unn2 薢
uno 茗
uno 苕
uno, 葂
uno. 菟
unuf 蘚
unv 蘇
unw 萏
unx 菉
uo/o 莒
uo18 蕢
uoab 荶
uoaf 蓶
uoei 菋
uoo, 勸
uoo1 蘄
uoo8 顴
uoo; 飌
uooa 歡
uoob 萼
uoof 雚
uooh 鸛
uoop 鄿
uoop 酄
uooq 觀
uoos 蕇
uop 葺
uop5 蕺
up 茸
up8 萁
up8t 藄
upa; 蔭
upai 蒢
upid 蔯
upjz 蘟
upk 萇
upk5 蘵
upke 薩
upo8 藬
upqj 蘛
upql 葚
uprf 蕼
ups8 蕻
upx 菆
upx, 藂
uq. 芃
uqqq 藞
uqx 芟
uqy 脁
ur.n 藰
ur/ 芛
ur/o 莙
ur/y 蔒
ur11 茞
ur16 蕭
ur1c 芭
ur1p 茆
ur1u 藍
ur1u 藎
ur1x 菣
ur2b 蘙
ura 芵
uraa 莁
urb 芎
urb. 蒻
urb1 茀
urb; 苰
urbf 虇
urbp 葞
urc 芑
urc. 茿
urj/ 芽
urj5 苠
urjf 蕥
urk 茛
uroo 蓲
urql 蒆
urr 苣
urr. 蕁
urt 兼
urta 歉
urth 鶼
us 半
us1f 茬
us1g 荐
usb8 蘜
usef 蘣
usef 蘳
ush8 頖
ushx 叛
usli 葉
usm, 葧
usm1 蔕
usmg 荸
usms 萳
usn 判
uso 苦
uso 若
uso0 逽
usof 蓳
usoj 葫
usop 鄀
usoz 惹
usu8 蕡
usx 芰
usx 苃
usx. 茇
ut, 苭
ut.p 薌
ut8n 蒶
utb. 葯
utdu 蘊
uteq 茪
utj 莦
utjn 萷
utnc 蕝
utr 葒
utt 茲
utt 蕬
uttg 孳
utth 鶿
uttz 慈
utx. 蔠
uu 并
uu 並
uu 茻
uu0 迸
uud 普
uue. 瓶
uuei 業
uuep 鄴
uuf. 對
uufr 蒫
uufx 叢
uufy 蓔
uufz 懟
uuh 鵧
uuh8 頩
uujn 葥
uunc 艵
uup 郱
uurt 蒹
uus0 鑿
uuu 荓
uuu. 薱
uuvc 菤
uuwj 蒴
uv 芙
uv/b 莠
uv0 送
uv1 芊
uv2 牶
uv22 韏
uv3 眷
uv4 薰
uv8/ 蕛
uv9w 齤
uva 芺
uvaa 莢
uvao 蕎
uvaz 菾
uvb, 菞
uvbc 卷
uvbi 蔾
uvbx 藜
uvc 芼
uvd 萶
uvd 萫
uvdc 菴
uvdf 董
uvdl 藒
uve, 豢
uveb 荂
uvf 菶
uvfx 薐
uvg 拳
uvg5 莪
uvi; 藕
uvid 藉
uvje 蒩
uvl 萎
uvl; 蘶
uvm1 帣
uvn 券
uvn 莉
uvn 觠
uvrb 弮
uvrb 荑
uvt 絭
uvtj 蕱
uvu. 莽
uvuf 菙
uvv 蓁
uvy 萩
uw0 逆
uwa 萸
uwby 蕮
uwe8 藇
uwj 朔
uwjf 塑
uwji 槊
uwjz 愬
uwmn 虋
uww 茁
ux 艾
ux. 苳
ux.k 蒗
ux/x 莈
ux4s 藫
ux4t 薸
ux6 蒲
ux6. 薄
ux9x 蔋
uxa5 虃
uxai 蒤
uxak 蒞
uxan 蕍
uxbc 范
uxcx 菠
uxd/ 蕩
uxd; 蕅
uxdu 薀
uxe1 萍
uxej 菏
uxf0 蓬
uxfx 蔆
uxh 鴱
uxh8 蕦
uxhk 蒎
uxi 莯
uxix 蔱
uxj7 藩
uxje 菹
uxkl 茫
uxko 薃
uxkq 蓅
uxmq 莐
uxno 菬
uxo 茖
uxoi 藻
uxr 茳
uxri 蕖
uxrr 菃
uxs1 莃
uxt/ 莎
uxuu 蓱
uxvo 萿
uxxo 落
uy 苂
uyds 蔊
uyh8 薠
uyy 菼
uyyy 藀
uz 芯
uz/ 苾
uzkk 蘹
uzzz 蕊
v 大
v 夫
v 禾
v, 夯
v. 太
v. 犬
v.0 迖
v.rk 稂
v.s 科
v.v. 猋
v.v; 飆
v/b/ 秭
v/b0 透
v/bb 秀
v/d/ 穆
v/d1 稗
v0et 穄
v1 千
v111 夼
v19c 乖
v1n 刊
v1o 舌
v1o0 适
v1o5 舐
v1oe 甜
v1of 舑
v1oh 鴰
v1on 刮
v1oy 舕
v1oz 舔
v1oz 憩
v1vc 麉
v1vy 烎
v1w 臿
v1wa 歃
v1yq 覝
v2o2 稦
v2v 秩
v3q 規
v3qi 槼
v3ql 嫢
v4 熏
v4, 勳
v78x 稷
v7s 秞
v7so 稒
v7uk 穠
v7v 稛
v8d0 遼
v8dh 鷯
v8n 秎
v8oq 稅
v8r/ 稊
v9ec 乘
v9en 剩
v9h5 穢
v; 私
v;8x 稄
v;;/ 穇
v;o 秮
v;rq 穊
va 夭
va0 迗
va; 蚕
vaa 夾
vaa8 頰
vaa; 麶
vaah 鵊
vaap 郟
vabz 稔
vad 昋
vai 稌
vak. 奪
vak7 奮
vakf 稚
vaky 穛
vamu 稐
vao 吞
vaoh 鷮
vaoo 喬
vaox 敿
vax 黍
vax; 黐
vaxo 黏
vaz 忝
vb/4 黧
vb/h 鵹
vb/n 鯬
vb/x 黎
vb9j 稰
vbb/ 穋
vbg 季
vc 毛
vc 乇
vc2q 毨
vc4 毢
vc7z 毸
vcc 秕
vce. 毬
vce5 毧
vcko 毰
vcnd 氌
vcr0 毽
vcs8 穬
vcud 氆
vcuv 氁
vcvc 毳
vcy 穮
vcyy 毯
vd 春
vd 香
vd2x 馥
vd33 馡
vd;; 蠢
vdc 奄
vdch 鵪
vddu 馧
vdes 稈
vdf 重
vdf, 動
vdfd 濌
vdhc 馜
vdi 稞
vdsg 馞
vdut 馦
vdvc 馣
vdvd 馫
vdz 惷
vdz/ 馝
ve;f 秷
vedz 穗
vee0 瓠
veeb 夸
veec 匏
veeh 鴮
veen 刳
veet 奈
veet 綔
vei 秣
veo5 稢
veo7 稫
veoa 欹
veoh 鵸
veoj 奇
veon 剞
veox 攲
veox 敧
veu1 秤
vf 奉
vf38 積
vf8f 稑
vf8x 稜
vfao 穡
vfd 奢
vfds 奲
vff 奎
vffq 穘
vfo 秸
vg5 我
vg5h 鵝
vh 鳺
vh9e 秠
vhc 秜
vhdd 奭
vhmu 稨
vi 耒
vi,o 耞
vi. 秫
vi;o 耛
vibg 耔
vid; 耦
vidl 耬
vie; 耘
vieu 耕
vih. 耨
vih8 頛
vihr 耜
vihx 耰
vij, 耡
vik. 耪
virc 耙
virr 耟
visd 耤
visf 耩
vit/ 耖
vivc 耗
vj4. 穱
vjbg 稃
vje 租
vjmf 稱
vjrz 穩
vjw 稻
vk8p 穧
vkdf 穜
vkog 稕
vkok 穰
vkoo 稿
vl 委
vl/2 犩
vl/; 魏
vl0 逶
vl3q 覣
vle. 瓾
vme, 稼
vmfo 稠
vmu 盉
vmv 秧
vmvc 秺
vn 利
vn2 犁
vnd 穭
vni 梨
vnn 移
vnn 奓
vo 和
vo1 种
vo5 程
vp 套
vp 邞
vp 耷
vp8 稘
vq 禿
vqh 鵚
vqh8 頹
vqo8 穨
vr 秉
vr1p 奅
vrb 夷
vre5 秖
vree 秪
vroo 奩
vrr 秬
vs3 看
vs3 稙
vs38 稹
vscd 稽
vscw 嵇
vsd 稓
vse8 穔
vsef 拜
vsu 奔
vt/ 秒
vtj 稍
vtt5 穖
vu,0 穟
vuax 穫
vuf 垂
vuf 羍
vuf. 甀
vufp 郵
vv 秦
vv 秝
vv6 輦
vv6p 鄻
vva 奏
vvao 穚
vvc 秅
vvc 秏
vvd 替
vvdf 種
vw 秈
vw 舂
vwz 憃
vx 泰
vx6 奫
vxs1 稀
vxwx 稯
vxxh 鷞
vxxx 爽
vy 秋
vy; 蝵
vye. 甃
vyg 揫
vyh 鶖
vyl 媝
vyz 愁
vz/ 秘
w 山
w 臼
w 凵
w 屮
w, 屴
w,1. 嶽
w.2c 崺
w.rk 崀
w/3i 嵲
w/; 嵬
w/bx 岌
w/bx 岋
w/d 岶
w/d1 崥
w/d5 崲
w0 辿
w0 崟
w0eu 嶝
w0na 嶔
w23 岝
w2c 屹
w2q8 巑
w38n 崱
w3q 峴
w433 嶵
w4fs 嶧
w5 鼠
w52e 鼪
w53. 鼳
w538 鼰
w550 鼮
w57s 鼬
w58n 鼢
w5bo 鼩
w5eo 鼯
w5f. 鼭
w5fj 鼱
w5hq 鼶
w5jv 鼷
w5kx 鼤
w5m. 鼵
w5m6 鼲
w5q 鼫
w5rl 鼴
w5s. 鼥
w5ut 鼸
w5v. 鼣
w5x. 鼨
w6 峬
w6h1 嶄
w6j. 嶈
w7, 舅
w7s 岫
w7so 崮
w7uk 嶩
w7z 崽
w84d 嶒
w88 崷
w88. 嶟
w8n2 嶙
w8o 峪
w8qx 毇
w; 蚩
w;8x 峻
w;;/ 嵾
wa 臾
wa.s 斞
wa11 岕
wab 岑
wab 岒
wab. 岭
wab8 嶺
wajn 崳
wakf 崔
wako 巂
wako 雟
wakp 酅
wakw 嶲
waky 嶕
wamu 崙
wao 峇
wao 峆
waoa 嶮
waro 嵢
wb/ 岉
wbb/ 嵺
wbbd 嶍
wbd 峋
wbo 岣
wby 舄
wc29 嶷
wce. 巘
wce5 巇
wcro 嵣
wcx 岥
wd1 岬
wdbl 嵑
wdcc 崑
wde/ 崵
wde9 崼
wdm; 嵎
wdol 嶁
we 凹
we2o 峿
we7d 嶆
we8 與
we8 興
we8 輿
we81 譽
we86 轝
we8a 歟
we8f 舉
we8h 鸒
we8q 礜
we;f 峌
wea9 嵷
wede 峘
weeq 岏
wej8 嵿
wenn 峛
weo. 獃
weo8 顗
weo; 颽
weoj 岢
weon 剴
weoq 凱
weoq 覬
weou 豈
weox 敳
weq 兕
weq 屼
wer9 崨
werf 嵂
weu 岍
weuf 崋
wf8x 崚
wfc 峔
wff. 崶
wffq 嶢
wffu 豐
wfg. 峙
wfj 崝
wfna 崁
wfqx 毀
wgaf 嶊
wh13 嵋
wh38 崸
wh9e 岯
whb5 峸
whe1 岓
whes 岸
whey 嵨
whff 崖
whl5 崴
whm0 遄
whm1 耑
whm1 峏
whm8 顓
whma 歂
whmn 剬
who5 嵅
whr1 巋
whr8 糱
whr; 蠥
whri 櫱
whrs 嶭
whso 崌
whua 嶡
whww 崛
why 炭
wi8; 崧
wiaa 崍
wid 崠
wikf 嵀
wje 岨
wji7 嶓
wjj 崩
wjjn 剻
wjrj 崢
wk 岦
wk81 崹
wkas 崒
wkds 嶂
wkjx 巃
wkla 峐
wkog 崞
wkoj 嵉
wkoo 嵩
wks 峷
wkx/ 嵃
wkxe 嵼
wl7 崰
wm1 帠
wm2o 嚳
wm3q 覺
wm5. 璺
wm8 岤
wm8n 釁
wm8o 嵱
wm8r 崆
wmbg 學
wmc 岮
wmeb 嶀
wmeo 峒
wmet 崇
wmf 壆
wmh 鷽
wmiy 爨
wmje 舋
wml 峖
wmn 鱟
wmn 觷
wmq 礐
wms8 黌
wmuw 崗
wmv 岟
wmw 嶨
wmx 澩
wmy 燢
wmy 爂
wn. 屻
wn2i 嵥
wnhc 峗
wnhc 峞
wnn2 嶰
wno 岧
wno. 巉
woaf 嶉
woob 崿
woox 巖
wop5 嶯
wpql 嵁
wq 兒
wq 岩
wq; 嵐
wqh 鶂
wqp 郳
wr.7 嵧
wr/o 峮
wrb1 岪
wrb; 嵹
wrc 屺
wred 崏
wrj/ 岈
wrj5 岷
wrk 峎
wroo 嶇
wrqd 嶜
wrr 岠
ws38 巔
wsea 嵌
wsj 峟
wsm1 嵽
wso 岵
wsx 岐
wt, 岰
wta5 巀
wtj 峭
wu 芔
wu 舁
wudl 嶱
wuf5 嶬
wufr 嵯
wuhl 巕
wuof 巏
wutt 嵫
wuui 嶪
wv.s 嵙
wv6 輋
wv8t 嶚
wv9c 嵊
wva 岆
wvaa 峽
wvao 嶠
wvdc 崦
wvej 崎
wvf 峚
wvg5 峨
wvl 崣
wvl; 巍
wvrb 峓
ww 出
ww 屾
ww8f 糶
wwe8 嶼
wwet 祟
wxf 峰
wxo 峉
wxo 峈
wxsj 崤
wxwx 嵕
wyy, 嶗
wyyi 嶸
wyyo 巆
x 又
x 水
x 乂
x, 氻
x,fd 瀦
x,o 泇
x,vj 漪
x. 叉
x. 冬
x. 汸
x. 洀
x..; 蚤
x./y 溈
x.11 洲
x.29 漩
x.2c 湤
x.2g 游
x.;; 螽
x.a. 淤
x.n8 澬
x.qu 瀊
x.rk 浪
x.x 泳
x/3 洎
x/3. 溴
x/3u 濞
x/90 涎
x/; 溾
x/;p 瀄
x/b/ 泲
x/bx 汲
x/c 氿
x/c. 汍
x/ci 染
x/d 泊
x/d5 湟
x/di 濼
x/do 澔
x/ds 滜
x/dx 激
x/k 泛
x/mu 洫
x/mv 澳
x/ra 澨
x/rf 潷
x0 淦
x0et 漈
x0eu 澄
x0ev 湀
x0rx 潑
x111 汌
x12s 滸
x1rx 溲
x1tb 灣
x1ti 灤
x20 逄
x27. 海
x27t 瀿
x2c 汔
x2d/ 漡
x2e 泩
x2i 洙
x2mn 淛
x2o 浩
x2o2 湋
x2q 洗
x2q0 鍌
x2q8 灒
x2s 汻
x2sy 潕
x2v 泆
x2vd 潪
x33f 灈
x36 濎
x38 浿
x385 濺
x38l 瀴
x38n 測
x3af 濉
x3q 涀
x3t/ 渺
x3v. 湨
x4 泗
x4 洒
x4 潶
x4, 潀
x4b; 濁
x4ds 潭
x4ek 澴
x4et 漂
x4f 湮
x4fs 澤
x4hn 瀱
x4i 溧
x5 汪
x55/ 濏
x6 浦
x6 淵
x6 沜
x6/x 灁
x60 漣
x65 潤
x6d 澗
x6g. 溥
x6h1 漸
x6h; 魙
x6i 澖
x6i4 瀾
x6kx 潣
x6os 灛
x7 沺
x711 淠
x722 潿
x77f 灅
x7a 泅
x7e, 溷
x7e5 漍
x7ek 渨
x7i 涃
x7j 渭
x7o 洄
x7s 油
x7s8 潩
x7sb 涄
x7so 涸
x7t 漯
x7u 浀
x7uk 濃
x7us 溿
x7uu 澧
x7v 洇
x8 酒
x8 汃
x84d 潧
x878 瀵
x8;b 滃
x8n 汾
x8n2 潾
x8nu 湓
x8o 浴
x8oq 涗
x8r/ 涕
x8x 潎
x8xu 滏
x9 沚
x9 浞
x938 湞
x97y 滷
x9c 泚
x9ds 淖
x9ex 淑
x9h5 濊
x9m3 濬
x9n8 澯
x9ne 瀣
x9o 沾
x9oz 惉
x9t 涉
x9t8 瀕
x9xo 潞
x;2 洠
x;2v 涘
x;8x 浚
x;;/ 滲
x;ke 灗
x;o 治
x;q 沇
x;r 渱
x;rq 溉
xa/ 沴
xa1i 滌
xa1j 滫
xa1j 潃
xa1x 浟
xa4d 澮
xaa5 瀸
xaau 溣
xab. 泠
xabz 淰
xac 沎
xad 昝
xag. 泭
xai 涂
xaif 塗
xaix 漵
xajn 渝
xakf 淮
xaki 潗
xaks 準
xaky 潐
xamu 淪
xao 咎
xao 洽
xaoa 澰
xaob 潝
xaou 渰
xaou 瀹
xaox 瀲
xar/ 洢
xaro 滄
xaso 涻
xauv 濮
xav. 洑
xb 溤
xb,o 潏
xb. 汋
xb., 湧
xb.f 涌
xb/ 沕
xb/z 淴
xb1 渹
xb1c 池
xb2w 淘
xb8 淗
xb9j 湑
xbb/ 漻
xbb8 瀷
xbbf 濯
xbbs 濢
xbc 氾
xbd 洵
xbeo 泀
xbj 沀
xbj8 澦
xbo 泃
xbrc 泡
xbxw 洶
xbxw 涵
xc 汽
xc 漉
xc/. 汛
xc4f 瀍
xc7u 瀘
xc7z 濾
xc8 滊
xcc 沘
xccd 湝
xce, 澽
xci 柒
xci3 灖
xcjg 滹
xcn 沏
xcov 澞
xcq 淲
xcq/ 滮
xcrf 滽
xcro 溏
xcrx 漮
xcs8 瀇
xcsx 渡
xcut 濂
xcx 波
xcxl 婆
xcxq 碆
xcy 瀌
xd 汨
xd 汩
xd 沓
xd2e 湦
xd4x 漫
xd5 洩
xdbl 渴
xdcc 混
xdd 淐
xde 泹
xde. 淂
xde. 璗
xde/ 湯
xde9 湜
xdes 涆
xdeu 盪
xdev 淏
xdey 燙
xdf 浬
xdf 涅
xdf. 溡
xdi 淉
xdk8 灝
xdkt 澋
xdm; 湡
xdmu 溫
xdol 漊
xdsx 瀑
xdt8 灦
xdtq 滉
xdty 濕
xduf 滭
xe,. 涿
xe1b 沔
xe2o 浯
xe2x 澓
xe55 淺
xe7d 漕
xe8x 濔
xe99 漇
xe9y 漹
xe;d 溍
xe;f 洷
xea9 漎
xed. 漙
xede 洹
xedx 浭
xedz 潓
xee; 沄
xeeb 污
xeej 洐
xeeq 沅
xeet 沶
xeg 汙
xegu 盓
xei 沫
xei 沬
xej 汀
xelr 涇
xemc 灑
xen 沍
xenn 洌
xeo1 滆
xeo5 淢
xeo7 湢
xeo; 瀜
xeoj 河
xeoj 滒
xeou 浢
xerl 淒
xes 汗
xetf 溼
xeu 汧
xeu 汫
xeu1 泙
xewc 沌
xf 圣
xf 夆
xf.x 滶
xf0 逢
xf38 漬
xf48 瀆
xf75 瀻
xf8f 淕
xf8x 淩
xf; 法
xf;u 溘
xfao 濇
xfb. 濤
xfbg 涍
xfd 渚
xfeb 洘
xff 洼
xffq 澆
xffu 灃
xffu 灩
xfj 清
xfmx 濲
xfmx 瀔
xfmx 瀫
xfnt 潔
xfo 洁
xfo. 澍
xfo/ 澎
xfo5 渽
xfok 溒
xfq 湱
xft 浾
xft 溹
xfus 涬
xgao 湁
xgh1 浙
xh/t 源
xh13 湄
xh1; 灟
xh7p 湎
xh;0 泒
xhbg 潺
xhc 泥
xhdf 湹
xhe. 溽
xhe1 沂
xhe9 泝
xhef 渥
xhef 垽
xhej 漘
xheo 洉
xhfd 潳
xhff 涯
xhi 淭
xhjx 瀀
xhm 漏
xhm1 洏
xhmn 涮
xhmv 渜
xho5 減
xhoc 滬
xhos 澼
xhoz 澸
xhr1 溮
xhrk 派
xhso 涺
xhsx 澱
xhu; 濿
xhv. 淚
xhv9 瀝
xhvc 浘
xhww 淈
xhx 汳
xhy5 滅
xi 沐
xi. 沭
xi.a 閷
xi.n 剎
xi.x 殺
xi2/ 灪
xi3 湘
xi4 湅
xi8; 淞
xiaa 淶
xiax 漆
xid 涷
xide 渣
xie 泍
xie5 弒
xiet 渿
xiff 溎
xih1 淅
xii 淋
xii. 滼
xii9 濋
xiif 漜
xiij 潸
xiit 澿
xio 涑
xio8 瀨
xioa 漱
xj3q 漞
xj4. 灂
xj5 淫
xjbg 浮
xje 沮
xjex 湲
xji7 潘
xjj 淜
xjl 浽
xjrc 淝
xjrj 淨
xjrz 濦
xjth 鸂
xjtv 溪
xjw 滔
xk 泣
xk4k 瀤
xk7e 澶
xk8. 滂
xk81 渧
xk8k 滾
xk8o 滴
xk8p 濟
xk8x 洨
xk9 汴
xk;0 鎏
xk;j 淯
xk;q 流
xk;x 澈
xka0 液
xkas 淬
xkd 湆
xkd8 灨
xkdf 潼
xkdk 滖
xkds 漳
xkdz 澺
xkf 注
xkiq 瀙
xkj 湇
xkjx 瀧
xkl 汒
xkl. 瀛
xkm1 沛
xko 涪
xko, 濠
xkog 淳
xkoj 渟
xkok 瀼
xkoo 滈
xkop 漷
xkoq 湸
xkot 涼
xkox 潡
xkq 沆
xkt 泫
xkt7 滀
xktf 澭
xkx 汶
xkx; 漓
xkxe 滻
xkxf 灕
xl 汝
xl7 淄
xldi 漅
xlo 洳
xlof 灉
xm2 浶
xm6 渾
xm8 泬
xm8. 湥
xm80 溛
xm8f 漥
xm8i 深
xm8o 溶
xm8o 浻
xm8o 滘
xm8r 涳
xm9x 漃
xma 汭
xma. 澪
xmaf 瀖
xmc 沱
xmd8 溟
xme8 演
xme8 濱
xme9 淀
xme; 澐
xmee 渲
xmej 泞
xmeo 洞
xmeq 浣
xmer 澠
xmet 淙
xmex 滱
xmfo 淍
xmfz 瀗
xmh1 濡
xmhr 涫
xmi 浨
xmi3 灀
xmj 滑
xmj7 瀋
xmk/ 浵
xmks 滓
xml 洝
xmnc 涴
xmo 渦
xmo 泂
xmo0 濄
xmq 沈
xmq 沉
xms8 淟
xms; 浤
xmsj 灞
xmv 泱
xmv. 涋
xmwy 瀉
xmzj 濘
xmzw 滵
xn 刈
xn 汐
xn 漁
xn.8 粱
xn.9 澀
xn.i 梁
xn.z 涊
xn2i 滐
xn4, 潒
xnau 盜
xnh1 澹
xnhc 洈
xnmv 渙
xnn2 澥
xno 沼
xno 洺
xno. 瀺
xnoq 浼
xnw 淊
xnx 淥
xo 各
xo 呇
xo1 沖
xo10 瀢
xo18 潰
xo38 溳
xo5 浧
xo5f 塣
xoaf 雒
xoh 鵅
xoj 涓
xolv 洖
xooi 澡
xoos 潬
xooz 漶
xop 湒
xop5 濈
xoq 況
xorc 浥
xp 洱
xp, 泐
xp8 淇
xp81 澌
xp8t 濝
xpai 滁
xpo8 濻
xppp 灄
xpql 湛
xps0 瀡
xq 沰
xq. 汎
xq; 渢
xqen 洬
xqo 沿
xqx 沒
xqy 洮
xr 江
xr.7 溜
xr.n 瀏
xr/o 涒
xr11 洍
xr16 潚
xr1p 泖
xr1q 灠
xr1u 濫
xr1u 濜
xr5 泧
xr5 洭
xra 決
xrb. 溺
xrb1 沸
xrb; 泓
xrb; 漒
xrbk 漲
xrbp 渳
xrbx 瀰
xrc 汜
xrc8 潠
xre5 汦
xred 涽
xree 泜
xrf 津
xrf0 湕
xrfe 澅
xrff 堻
xrh 鴻
xrh8 澒
xrj5 泯
xrjd 湣
xrmx 浸
xroo 漚
xrpx 澉
xrq1 灊
xrqd 潛
xrr 洰
xrr. 潯
xrri 渠
xrru 盥
xs 汁
xs1g 洊
xs3 淔
xs38 滇
xs8 洪
xs80 鍙
xs8c 港
xs; 汯
xsc. 沋
xsdb 瀚
xsdc 漧
xsdj 潮
xsds 澣
xsds 濣
xse 泔
xse8 潢
xsj 肴
xsj 洧
xsjp 郩
xsjx 殽
xsl 泄
xsli 渫
xsm, 渤
xsm. 瓻
xsm1 希
xsm1 滯
xsma 滿
xsma 欷
xsmf 溝
xsmg 浡
xsmh 鵗
xsmp 郗
xsms 湳
xsmz 懣
xsmz 懘
xso 沽
xsof 灘
xsoj 湖
xsok 灢
xsov 漢
xsu8 濆
xsx 汥
xsx. 沷
xsy 洃
xt, 泑
xt/ 沙
xt/0 逤
xt/3 渻
xt/g 挲
xt/k 裟
xt/l 娑
xt/n 鯊
xt/q 硰
xtaf 濰
xteq 洸
xtj 消
xtjq 湚
xtm7 澢
xtmf 漟
xtmo 淌
xu. 淓
xu45 瀎
xu64 灡
xu7 渵
xu8u 溢
xu9 淽
xuax 濩
xubz 濍
xucy 瀳
xud; 澫
xudv 漠
xuef 澕
xuej 渮
xuf 洋
xufk 瀁
xufr 溠
xufu 濭
xufv 渼
xufx 漾
xufy 溔
xujn 湔
xum, 濛
xumv 渶
xuof 灌
xur6 瀟
xurt 溓
xuru 灆
xus 泮
xuso 渃
xutt 滋
xutz 濨
xuu 洴
xuu 湴
xuu. 濧
xuud 潽
xuui 澲
xuv 浂
xuv. 漭
xuvt 潫
xuwj 溯
xv 汏
xv. 汰
xv. 汱
xv1o 活
xv8t 潦
xv9c 溗
xva 沃
xva0 鋈
xvaa 浹
xvax 潻
xvaz 添
xvd 湷
xvdc 淹
xvdf 湩
xveb 洿
xvg5 涐
xvn 浰
xvq8 瀩
xvrb 洟
xvsf 湃
xvsu 渀
xvtj 潲
xvv 溱
xvva 湊
xvx 溙
xvxx 漺
xvy 湫
xw 凶
xw 汕
xw; 滍
xwab 涔
xwaf 漼
xwby 潟
xweu 溰
xwex 溦
xwex 澂
xwh1 湍
xwhy 湠
xwjj 漰
xwmq 灚
xwq 兇
xwq 淣
xww 泏
xx 爻
xx 沝
xx. 汊
xx.; 溞
xx2 洚
xxd 涾
xxo 洛
xxs1 浠
xxsa 濷
xxsj 淆
xxx 淼
xxx8 顙
xxxa 欼
xxxa 歠
xxxc 毲
xxxh 鵽
xxxi 桑
xxxn 剟
xxxx 敪
xywc 鬯
xyy 淡
xyy, 澇
xyy. 瀅
xyyo 瀯
xyyt 瀠
xyyx 濴
xz 沁
xz/ 泌
xzo1 浺
xzvo 湉
y 火
y,,8 燲
y,,j 熁
y.rk 烺
y.s 炓
y/3z 熄
y/7z 熜
y/90 烻
y/d5 煌
y/di 爍
y/mu 烅
y/mv 燠
y0eu 燈
y23 炸
y27. 烸
y2o2 煒
y2q 烍
y4b; 燭
y4ds 燂
y4et 熛
y4f 煙
y4f 爅
y4fs 燡
y50 烶
y6 烳
y6i4 爛
y6kx 燘
y6nw 爓
y6z 燜
y7ek 煨
y7j 煟
y7s8 熼
y88 煪
y88. 燇
y8; 炂
y8n2 燐
y8o 焀
y8r/ 焍
y9ds 焯
y9n8 燦
y;8x 焌
y;;; 爞
ya4d 燴
yab. 炩
yabo 焓
yaf 烇
yaky 燋
yao 烚
yaou 爚
yaro 熗
yb,i 煣
yb,o 燏
yb. 灼
yb.0 熥
yb1c 灺
ybbd 熠
ybbf 燿
ybbk 熤
ybf 炄
ybrc 炮
yc 熝
yc7u 爐
yc8 熂
ycro 煻
ycs8 爌
ycy 爊
yd2e 煋
yd4x 熳
ydas 焺
ydcc 焜
yde 炟
yde/ 煬
ydes 焊
ydj 焨
ydk 煜
ydmu 熅
ydol 熡
ydsx 爆
ydtq 熀
yduf 熚
ye, 烼
ye2o 焐
ye9 炡
yede 烜
yeej 烆
yelr 烴
yema 炳
yer5 烒
yewc 炖
yf 灶
yf;u 熆
yfb. 燽
yfeb 烤
yff 烓
yffq 燒
yh13 煝
yh38 煩
yhbb 煽
yhbo 焗
yhe1 炘
yhea 焮
yhex 煆
yhmu 煸
yho5 煘
yhoc 熩
yi 炑
yi2/ 爩
yi4 煉
yi9 烞
yj4. 爝
yjbg 烰
yjex 煖
yji7 燔
yjmx 燰
yjvy 燃
yk8o 熵
yka0 焲
ykas 焠
ykd5 熾
ykdz 燱
ykf 炷
ykla 烗
yko 焙
ykog 焞
ykok 爙
ykoo 熇
ykox 燉
ykq 炕
ykt 炫
ykx 炆
ym6 煇
ym8o 熔
ym8r 焢
ymd8 熐
ymeo 烔
ymeq 烷
ymh1 燸
ymk/ 烿
ymnc 焥
ymo 炯
ymoa 爧
ymv 炴
ymvc 烢
yn 灱
yna 炊
ynmv 煥
yno 炤
ynw 焰
yo38 熉
yoj 焆
yooi 燥
yooo 煰
yoos 燀
yoq 炾
ypql 煁
yqy 烑
yr 灴
yr16 熽
yr1f 熞
yr1q 爦
yr1u 燼
yr1u 爁
yra 炔
yrf0 煡
yroo 熰
yrqd 熸
yrr 炬
yrr. 燖
ys 煂
ys8 烘
ys8 烡
ysd 焟
yse8 熿
ysei 煤
ysj 烠
ysli 煠
ysov 熯
yt/ 炒
ytm4 爣
yu,0 燧
yud; 燤
yuef 燁
yuf 烊
yuf5 燨
yuf5 爔
yumv 煐
yuof 爟
yv4 燻
yv8t 燎
yvao 燆
yvdf 煄
yvff 煃
yvnn 熪
yvy 煍
ywfx 燬
ywh1 煓
yx. 炵
yxf 烽
yxo 烙
yxs1 烯
yy 炎
yy3q 覢
yy9o 煔
yy9x 敥
yyh8 顃
yyh8 顲
yym, 勞
yym. 瑩
yym. 煢
yym. 甇
yym0 鎣
yym1 謍
yym2 犖
yym8 醟
yym; 螢
yymf 塋
yymh 鶯
yymi 榮
yymj 膋
yymk 褮
yyml 嫈
yymo 營
yymq 覮
yymt 縈
yymt 禜
yymw 罃
yymx 滎
yymy 熒
yyn 剡
yyna 欻
yyp 郯
yyy 焱
yyyi 燊
yyyi 爃
yyyy 燚
z 心
z,,j 愶
z.rk 悢
z/ 必
z/3q 覕
z/; 愧
z/bx 忣
z/d 怕
z/d5 惶
z/dx 憿
z/im 憡
z/mo 恦
z/mu 恤
z/mv 懊
z/p 邲
z0 惍
z23 怍
z27. 悔
z2c 忔
z2dx 愎
z2e 性
z2o 悎
z2o0 慥
z2s 忤
z2sy 憮
z2v 怢
z33 悱
z33f 懼
z33x 戄
z38n 惻
z4 怬
z4 恓
z4. 愣
z48 憟
z4ds 憛
z4ek 懁
z4et 慓
z4fs 懌
z4i 慄
z6h1 慚
z6j 憪
z6kx 憫
z7e5 慖
z7ek 愄
z7i 悃
z7m4 戃
z7o 恛
z7s 怞
z7s8 慣
z84d 憎
z8; 忪
z8n2 憐
z8oq 悅
z8r/ 悌
z9ds 悼
z9e8 懻
z9o 怗
z;8x 悛
z;;/ 慘
z;o 怡
z;rq 慨
zaa5 懺
zab 忴
zab. 怜
zaf 恮
zai 恘
zai 悇
zajn 愉
zakf 惟
zakx 愯
zaky 憔
zamu 惀
zao 恰
zaoa 憸
zaro 愴
zb,o 憰
zb.f 悀
zb/z 惚
zbb/ 憀
zbbd 慴
zbd 恂
zbdg 惸
zbf 忸
zbo 怐
zbrc 怉
zbxw 恟
zc8 愾
zc9o 惦
zcd 恉
zce, 懅
zcrf 慵
zcrx 慷
zcs8 懭
zd2e 惺
zd4x 慢
zdb/ 惕
zdbl 愒
zdcc 惃
zde 怛
zde/ 愓
zde9 惿
zdes 悍
zdf 悝
zdi 惈
zdkt 憬
zdmu 慍
zdol 慺
zdsx 懪
ze2o 悟
ze7d 慒
ze9 怔
zed. 慱
zede 恒
zedz 憓
zeeq 忨
zema 怲
zen. 恆
zeo7 愊
zerl 悽
zeu1 怦
zeu5 悈
zewc 忳
zf; 怯
zfb. 懤
zfc 恅
zffq 憢
zfg. 恃
zfj 情
zfo 恄
zfo/ 憉
zft 愫
zfus 悻
zg. 忖
zh7p 愐
zh9 怀
zh9e 怌
zhc 怩
zhd. 懨
zhe1 忻
zhe8 懫
zhjx 懮
zhmu 惼
zho5 憾
zhv. 悷
zi. 怵
zii 惏
zio 悚
zio8 懶
zj2w 愮
zje 怚
zjex 愋
zjtv 慀
zjw 慆
zk4k 懷
zk7v 懍
zk8p 懠
zk8x 恔
zk9 忭
zkas 悴
zkd 愔
zkdf 憧
zkds 慞
zkdz 憶
zkl 忙
zko. 憱
zkog 惇
zkok 懹
zkot 惊
zkt7 慉
zl/x 惱
zlx 怓
zm6 惲
zm8r 悾
zmd8 慏
zmee 愃
zmeo 恫
zmer 憴
zmet 悰
zmfo 惆
zmh1 懦
zmhr 悺
zmj 愲
zmnc 惋
zmq 忱
zmul 惘
zmv 怏
zmxo 愘
zmzj 懧
zn 忉
zna 忺
znh1 憺
znn 恀
znn2 懈
zno 怊
znoq 悗
zo1 忡
zo18 憒
zo5 悜
zoj 悁
zoob 愕
zooi 懆
zoos 憚
zoq 怳
zorc 悒
zpk 悵
zppp 懾
zpql 愖
zqy 恌
zr.n 懰
zr1f 慳
zr38 愩
zr5 怴
zr5 恇
zra 快
zrb1 怫
zrbt 惤
zrdl 愝
zre5 忯
zred 惛
zrj5 怋
zrk 恨
zroo 慪
zrqd 憯
zrr. 憳
zrva 愜
zs 愅
zs38 慎
zsd 惜
zsli 惵
zsm1 怖
zsm9 懥
zsma 慲
zsmg 悖
zso 怙
zsof 慬
zsrf 墯
zsrj 惰
zsu8 憤
zsx 忮
zsy 恢
zt, 怮
zteq 恍
ztj 悄
ztmo 惝
zu43 懵
zu45 懱
zudv 慔
zufk 懩
zukq 慌
zum, 懞
zuof 懽
zurt 慊
zus 怑
zuu 恲
zuvc 惓
zuvf 懂
zv 忕
zv1 忏
zv1o 恬
zv8t 憭
zvao 憍
zvaz 悿
zvbg 悸
zvd, 慟
zvrb 恞
zvxx 慡
zvy 愀
zwaf 慛
zweu 愷
zwh1 惴
zx.; 慅
zxf 怪
zxo 恪
zxs1 悕
zxw 忷
zxwx 惾
zxxx 惙
zyy 惔
zz/ 怭
zzz 惢
zzzi 橤
zzzt 繠



